/* This file is part of DSX.
 *
 * DSX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DSX is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with DSX; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Copyright (c) Lip6, Thalès
 *      Joel Porquet <joel.porquet@lip6.fr>, 2006-2007
 *
 * Based on Martin Fielder's work (http://keyj.s2000.ws/?page_id=41)
 */

#include "dec_bitstream.h"
#include "utils.h"

/***********************/
/* Decode Slice Header */
/***********************/
void decode_slice_header(slice_inf_t *slice_inf, nal_inf_t *nal_inf, bitreader_t *bits,
                         ps_t *ps)
{
    slice_inf->first_mb = bitreader_get_ugolomb(bits);
    /* read the slice type */
    slice_inf->type = bitreader_get_ugolomb(bits) % 5;
    /* read and skip pic_parameter_set_id */
    bitreader_get_ugolomb(bits);
    /* read frame_num */
    slice_inf->frame_num = bitreader_get(bits, ps->sps.log2_max_frame_num);
    /* skip field_pic_flag and bottom_field_flag since sps_frame_mbs_only_flag is 1 :
        frame coding and not field coding */
    /* skip MbaffFrameFlag since no field coding */
    if(nal_inf->type==5)
        /* read and skip idr_pic_id */
        bitreader_get_ugolomb(bits);
    if(ps->sps.pic_order_cnt_type==0)
    {
        /* read and skip pic_order_cnt_lsb */
        slice_inf->pic_order_cnt_lsb = bitreader_get(bits, ps->sps.log2_max_pic_order_cnt_lsb);
        /* skip delta_pic_order_cnt_bottom */
    }
    /* skip delta_pic_order_cnt since sps_pic_order_cnt_type is 0 */
    /* skip redundant_pic_cnt since pps_redundant_pic_cnt_present_flag is 0 */
    if(slice_inf->type==B_SLICE)
        /* read and skip direct_spatial_mv_pred_flag */
        bitreader_get_one(bits);
    if(slice_inf->type==P_SLICE || slice_inf->type==B_SLICE || slice_inf->type==SP_SLICE)
    {
		char num_ref_idx_active_override_flag;
        /* read num_ref_idx_active_override_flag */
        num_ref_idx_active_override_flag = bitreader_get_one(bits);
		if (num_ref_idx_active_override_flag == 1)
			/* read and skip num_ref_idx_l0_active */
			bitreader_get_ugolomb(bits);
        /* skip num_ref_idx_l1_active since no B_SLICE */
    }

    if(slice_inf->type!=I_SLICE && slice_inf->type!=SI_SLICE)
    {
        /* read and skip ref_pic_list_reordering_flag_l0 */
        bitreader_get_one(bits);
    }
    /* skip ref_pic_list_reordering_flag_l1 since slice_type cannot be B_SLICE type */
    /* skip weighted prediction */
    if(nal_inf->ref_idc != 0)
    {
        if(nal_inf->type==5)
        {
            /* read and skip no_output_of_prior_pics_flag and long_term_reference_flag */
            bitreader_get_one(bits);
            bitreader_get_one(bits);
        } else {
            /* read and skip adaptive_ref_pic_marking_mode_flag */
            bitreader_get_one(bits);
            /* skip skip_adaptive_ref_pic_marking */
        }
    }
    /* skip cabac_init_idc since entropy_coding_mode_flag is vlc and not cabac */
    /* read qp_delta */
    slice_inf->qp_delta = bitreader_get_sgolomb(bits);
    slice_inf->QPy = ps->pps.pic_init_qp + slice_inf->qp_delta;
    
    /* skip sp_for_switch_flag and slice_qs_delta since no SP/SI slice type */

    /* skip disable_deblocking_filter_idc since deblocking_filter_control_present_flag is 0 */
	if (ps->pps.deblocking_filter_control_present_flag == 1)
		bitreader_get_ugolomb(bits);
		
    /* skip slice_group_change_cycle since pps_num_slice_groups is 1 */

    srl_log_printf(DEBUG, "first_mb=%d; slice_type=%d; slice_pic_cnt=%d, slice_qp_delta=%d, slice_QPy=%d\n", 
            slice_inf->first_mb, slice_inf->type, slice_inf->pic_order_cnt_lsb, slice_inf->qp_delta, slice_inf->QPy);
}

/***********************/
/* Decode MB Modes     */
/***********************/
static const char ISliceMbModes[26][4]={
    {Intra_4x4,   NA, NA, NA},
    {Intra_16x16,  0,  0,  0},
    {Intra_16x16,  1,  0,  0},
    {Intra_16x16,  2,  0,  0},
    {Intra_16x16,  3,  0,  0},
    {Intra_16x16,  0,  1,  0},
    {Intra_16x16,  1,  1,  0},
    {Intra_16x16,  2,  1,  0},
    {Intra_16x16,  3,  1,  0},
    {Intra_16x16,  0,  2,  0},
    {Intra_16x16,  1,  2,  0},
    {Intra_16x16,  2,  2,  0},
    {Intra_16x16,  3,  2,  0},
    {Intra_16x16,  0,  0, 15},
    {Intra_16x16,  1,  0, 15},
    {Intra_16x16,  2,  0, 15},
    {Intra_16x16,  3,  0, 15},
    {Intra_16x16,  0,  1, 15},
    {Intra_16x16,  1,  1, 15},
    {Intra_16x16,  2,  1, 15},
    {Intra_16x16,  3,  1, 15},
    {Intra_16x16,  0,  2, 15},
    {Intra_16x16,  1,  2, 15},
    {Intra_16x16,  2,  2, 15},
    {Intra_16x16,  3,  2, 15},
    {NA, NA, NA, NA} // I_PCM
};

static const char PSliceMbModes[][5]={
    {1, Pred_L0, NA,      16, 16},
    {2, Pred_L0, Pred_L0, 16,  8},
    {2, Pred_L0, Pred_L0,  8, 16},
    {4, NA,      NA,       8,  8},
    {4, NA,      NA,       8,  8},
    {1, Pred_L0, NA,      16, 16}  // P_Skip
};

static const char PSliceSubMbModes[][4]={
    {1, Pred_L0, 8, 8},
    {2, Pred_L0, 8, 4},
    {2, Pred_L0, 4, 8},
    {4, Pred_L0, 4, 4},
};

/* According to slice_type and raw_mb_type, fill the mb_mode struct */
void decode_mb_mode(mb_mode_t *mb, int slice_type, int raw_mb_type)
{
    if (slice_type == I_SLICE) {
        srl_assert(raw_mb_type < 25); // PCM is excluded
        mb->mb_type                 = raw_mb_type+5;
        mb->NumMbPart               = 1;
        mb->MbPartPredMode[0]       = ISliceMbModes[raw_mb_type][0];
        mb->MbPartPredMode[1]       = NA;
        mb->Intra16x16PredMode      = ISliceMbModes[raw_mb_type][1];
        mb->MbPartWidth             = 16;
        mb->MbPartHeight            = 16;
        mb->CodedBlockPatternChroma = ISliceMbModes[raw_mb_type][2];
        mb->CodedBlockPatternLuma   = ISliceMbModes[raw_mb_type][3];
    } else if (slice_type == P_SLICE) {
        if (raw_mb_type>4)
            decode_mb_mode(mb, I_SLICE, raw_mb_type-5);
        else {
            srl_assert(raw_mb_type < 5);
            mb->mb_type                 = raw_mb_type;
            mb->NumMbPart               = PSliceMbModes[raw_mb_type][0];
            mb->MbPartPredMode[0]       = PSliceMbModes[raw_mb_type][1];
            mb->MbPartPredMode[1]       = PSliceMbModes[raw_mb_type][2];
            mb->Intra16x16PredMode      = NA;
            mb->MbPartWidth             = PSliceMbModes[raw_mb_type][3];
            mb->MbPartHeight            = PSliceMbModes[raw_mb_type][4];
            mb->CodedBlockPatternChroma = NA;
            mb->CodedBlockPatternLuma   = NA;
        }
    }
}

void decode_sub_mb_mode(sub_mb_mode_t *sub, int slice_type, int raw_sub_mb_type)
{
    if (slice_type == P_SLICE) {
        srl_assert(raw_sub_mb_type < 4);
        sub->sub_mb_type        = raw_sub_mb_type;
        sub->NumSubMbPart       = PSliceSubMbModes[raw_sub_mb_type][0];
        sub->SubMbPredMode      = PSliceSubMbModes[raw_sub_mb_type][1];
        sub->SubMbPartWidth     = PSliceSubMbModes[raw_sub_mb_type][2];
        sub->SubMbPartHeight    = PSliceSubMbModes[raw_sub_mb_type][3];
    }
}

/***********************/
/* Decode Residual     */
/***********************/

#define COEFF_TOKEN(TrailingOnes,TotalCoeff) (((TotalCoeff)<<2)|(TrailingOnes))

static const code_item_t CoeffTokenCodes[4][62]={ {
///// 0  <=  nC  <  2 /////
  { 0x80000000,  1, COEFF_TOKEN(0, 0) },  // 1
  { 0x40000000,  2, COEFF_TOKEN(1, 1) },  // 01
  { 0x20000000,  3, COEFF_TOKEN(2, 2) },  // 001
  { 0x18000000,  5, COEFF_TOKEN(3, 3) },  // 0001 1
  { 0x14000000,  6, COEFF_TOKEN(0, 1) },  // 0001 01
  { 0x10000000,  6, COEFF_TOKEN(1, 2) },  // 0001 00
  { 0x0C000000,  6, COEFF_TOKEN(3, 4) },  // 0000 11
  { 0x0A000000,  7, COEFF_TOKEN(2, 3) },  // 0000 101
  { 0x08000000,  7, COEFF_TOKEN(3, 5) },  // 0000 100
  { 0x07000000,  8, COEFF_TOKEN(0, 2) },  // 0000 0111
  { 0x06000000,  8, COEFF_TOKEN(1, 3) },  // 0000 0110
  { 0x05000000,  8, COEFF_TOKEN(2, 4) },  // 0000 0101
  { 0x04000000,  8, COEFF_TOKEN(3, 6) },  // 0000 0100
  { 0x03800000,  9, COEFF_TOKEN(0, 3) },  // 0000 0011 1
  { 0x03000000,  9, COEFF_TOKEN(1, 4) },  // 0000 0011 0
  { 0x02800000,  9, COEFF_TOKEN(2, 5) },  // 0000 0010 1
  { 0x02000000,  9, COEFF_TOKEN(3, 7) },  // 0000 0010 0
  { 0x01C00000, 10, COEFF_TOKEN(0, 4) },  // 0000 0001 11
  { 0x01800000, 10, COEFF_TOKEN(1, 5) },  // 0000 0001 10
  { 0x01400000, 10, COEFF_TOKEN(2, 6) },  // 0000 0001 01
  { 0x01000000, 10, COEFF_TOKEN(3, 8) },  // 0000 0001 00
  { 0x00E00000, 11, COEFF_TOKEN(0, 5) },  // 0000 0000 111
  { 0x00C00000, 11, COEFF_TOKEN(1, 6) },  // 0000 0000 110
  { 0x00A00000, 11, COEFF_TOKEN(2, 7) },  // 0000 0000 101
  { 0x00800000, 11, COEFF_TOKEN(3, 9) },  // 0000 0000 100
  { 0x00780000, 13, COEFF_TOKEN(0, 6) },  // 0000 0000 0111 1
  { 0x00700000, 13, COEFF_TOKEN(1, 7) },  // 0000 0000 0111 0
  { 0x00680000, 13, COEFF_TOKEN(2, 8) },  // 0000 0000 0110 1
  { 0x00600000, 13, COEFF_TOKEN(3,10) },  // 0000 0000 0110 0
  { 0x00580000, 13, COEFF_TOKEN(0, 7) },  // 0000 0000 0101 1
  { 0x00500000, 13, COEFF_TOKEN(1, 8) },  // 0000 0000 0101 0
  { 0x00480000, 13, COEFF_TOKEN(2, 9) },  // 0000 0000 0100 1
  { 0x00400000, 13, COEFF_TOKEN(0, 8) },  // 0000 0000 0100 0
  { 0x003C0000, 14, COEFF_TOKEN(0, 9) },  // 0000 0000 0011 11
  { 0x00380000, 14, COEFF_TOKEN(1, 9) },  // 0000 0000 0011 10
  { 0x00340000, 14, COEFF_TOKEN(2,10) },  // 0000 0000 0011 01
  { 0x00300000, 14, COEFF_TOKEN(3,11) },  // 0000 0000 0011 00
  { 0x002C0000, 14, COEFF_TOKEN(0,10) },  // 0000 0000 0010 11
  { 0x00280000, 14, COEFF_TOKEN(1,10) },  // 0000 0000 0010 10
  { 0x00240000, 14, COEFF_TOKEN(2,11) },  // 0000 0000 0010 01
  { 0x00200000, 14, COEFF_TOKEN(3,12) },  // 0000 0000 0010 00
  { 0x001E0000, 15, COEFF_TOKEN(0,11) },  // 0000 0000 0001 111
  { 0x001C0000, 15, COEFF_TOKEN(1,11) },  // 0000 0000 0001 110
  { 0x001A0000, 15, COEFF_TOKEN(2,12) },  // 0000 0000 0001 101
  { 0x00180000, 15, COEFF_TOKEN(3,13) },  // 0000 0000 0001 100
  { 0x00160000, 15, COEFF_TOKEN(0,12) },  // 0000 0000 0001 011
  { 0x00140000, 15, COEFF_TOKEN(1,12) },  // 0000 0000 0001 010
  { 0x00120000, 15, COEFF_TOKEN(2,13) },  // 0000 0000 0001 001
  { 0x00100000, 15, COEFF_TOKEN(3,14) },  // 0000 0000 0001 000
  { 0x00020000, 15, COEFF_TOKEN(1,13) },  // 0000 0000 0000 001
  { 0x000F0000, 16, COEFF_TOKEN(0,13) },  // 0000 0000 0000 1111
  { 0x000E0000, 16, COEFF_TOKEN(1,14) },  // 0000 0000 0000 1110
  { 0x000D0000, 16, COEFF_TOKEN(2,14) },  // 0000 0000 0000 1101
  { 0x000C0000, 16, COEFF_TOKEN(3,15) },  // 0000 0000 0000 1100
  { 0x000B0000, 16, COEFF_TOKEN(0,14) },  // 0000 0000 0000 1011
  { 0x000A0000, 16, COEFF_TOKEN(1,15) },  // 0000 0000 0000 1010
  { 0x00090000, 16, COEFF_TOKEN(2,15) },  // 0000 0000 0000 1001
  { 0x00080000, 16, COEFF_TOKEN(3,16) },  // 0000 0000 0000 1000
  { 0x00070000, 16, COEFF_TOKEN(0,15) },  // 0000 0000 0000 0111
  { 0x00060000, 16, COEFF_TOKEN(1,16) },  // 0000 0000 0000 0110
  { 0x00050000, 16, COEFF_TOKEN(2,16) },  // 0000 0000 0000 0101
  { 0x00040000, 16, COEFF_TOKEN(0,16) },  // 0000 0000 0000 0100
},{
///// 2  <=  nC  <  4 /////
  { 0xC0000000,  2, COEFF_TOKEN(0, 0) },  // 11
  { 0x80000000,  2, COEFF_TOKEN(1, 1) },  // 10
  { 0x60000000,  3, COEFF_TOKEN(2, 2) },  // 011
  { 0x50000000,  4, COEFF_TOKEN(3, 3) },  // 0101
  { 0x40000000,  4, COEFF_TOKEN(3, 4) },  // 0100
  { 0x38000000,  5, COEFF_TOKEN(1, 2) },  // 0011 1
  { 0x30000000,  5, COEFF_TOKEN(3, 5) },  // 0011 0
  { 0x2C000000,  6, COEFF_TOKEN(0, 1) },  // 0010 11
  { 0x28000000,  6, COEFF_TOKEN(1, 3) },  // 0010 10
  { 0x24000000,  6, COEFF_TOKEN(2, 3) },  // 0010 01
  { 0x20000000,  6, COEFF_TOKEN(3, 6) },  // 0010 00
  { 0x1C000000,  6, COEFF_TOKEN(0, 2) },  // 0001 11
  { 0x18000000,  6, COEFF_TOKEN(1, 4) },  // 0001 10
  { 0x14000000,  6, COEFF_TOKEN(2, 4) },  // 0001 01
  { 0x10000000,  6, COEFF_TOKEN(3, 7) },  // 0001 00
  { 0x0E000000,  7, COEFF_TOKEN(0, 3) },  // 0000 111
  { 0x0C000000,  7, COEFF_TOKEN(1, 5) },  // 0000 110
  { 0x0A000000,  7, COEFF_TOKEN(2, 5) },  // 0000 101
  { 0x08000000,  7, COEFF_TOKEN(3, 8) },  // 0000 100
  { 0x07000000,  8, COEFF_TOKEN(0, 4) },  // 0000 0111
  { 0x06000000,  8, COEFF_TOKEN(1, 6) },  // 0000 0110
  { 0x05000000,  8, COEFF_TOKEN(2, 6) },  // 0000 0101
  { 0x04000000,  8, COEFF_TOKEN(0, 5) },  // 0000 0100
  { 0x03800000,  9, COEFF_TOKEN(0, 6) },  // 0000 0011 1
  { 0x03000000,  9, COEFF_TOKEN(1, 7) },  // 0000 0011 0
  { 0x02800000,  9, COEFF_TOKEN(2, 7) },  // 0000 0010 1
  { 0x02000000,  9, COEFF_TOKEN(3, 9) },  // 0000 0010 0
  { 0x01E00000, 11, COEFF_TOKEN(0, 7) },  // 0000 0001 111
  { 0x01C00000, 11, COEFF_TOKEN(1, 8) },  // 0000 0001 110
  { 0x01A00000, 11, COEFF_TOKEN(2, 8) },  // 0000 0001 101
  { 0x01800000, 11, COEFF_TOKEN(3,10) },  // 0000 0001 100
  { 0x01600000, 11, COEFF_TOKEN(0, 8) },  // 0000 0001 011
  { 0x01400000, 11, COEFF_TOKEN(1, 9) },  // 0000 0001 010
  { 0x01200000, 11, COEFF_TOKEN(2, 9) },  // 0000 0001 001
  { 0x01000000, 11, COEFF_TOKEN(3,11) },  // 0000 0001 000
  { 0x00F00000, 12, COEFF_TOKEN(0, 9) },  // 0000 0000 1111
  { 0x00E00000, 12, COEFF_TOKEN(1,10) },  // 0000 0000 1110
  { 0x00D00000, 12, COEFF_TOKEN(2,10) },  // 0000 0000 1101
  { 0x00C00000, 12, COEFF_TOKEN(3,12) },  // 0000 0000 1100
  { 0x00B00000, 12, COEFF_TOKEN(0,10) },  // 0000 0000 1011
  { 0x00A00000, 12, COEFF_TOKEN(1,11) },  // 0000 0000 1010
  { 0x00900000, 12, COEFF_TOKEN(2,11) },  // 0000 0000 1001
  { 0x00800000, 12, COEFF_TOKEN(0,11) },  // 0000 0000 1000
  { 0x00780000, 13, COEFF_TOKEN(0,12) },  // 0000 0000 0111 1
  { 0x00700000, 13, COEFF_TOKEN(1,12) },  // 0000 0000 0111 0
  { 0x00680000, 13, COEFF_TOKEN(2,12) },  // 0000 0000 0110 1
  { 0x00600000, 13, COEFF_TOKEN(3,13) },  // 0000 0000 0110 0
  { 0x00580000, 13, COEFF_TOKEN(0,13) },  // 0000 0000 0101 1
  { 0x00500000, 13, COEFF_TOKEN(1,13) },  // 0000 0000 0101 0
  { 0x00480000, 13, COEFF_TOKEN(2,13) },  // 0000 0000 0100 1
  { 0x00400000, 13, COEFF_TOKEN(3,14) },  // 0000 0000 0100 0
  { 0x00380000, 13, COEFF_TOKEN(0,14) },  // 0000 0000 0011 1
  { 0x00300000, 13, COEFF_TOKEN(2,14) },  // 0000 0000 0011 0
  { 0x00080000, 13, COEFF_TOKEN(3,15) },  // 0000 0000 0000 1
  { 0x002C0000, 14, COEFF_TOKEN(1,14) },  // 0000 0000 0010 11
  { 0x00280000, 14, COEFF_TOKEN(2,15) },  // 0000 0000 0010 10
  { 0x00240000, 14, COEFF_TOKEN(0,15) },  // 0000 0000 0010 01
  { 0x00200000, 14, COEFF_TOKEN(1,15) },  // 0000 0000 0010 00
  { 0x001C0000, 14, COEFF_TOKEN(0,16) },  // 0000 0000 0001 11
  { 0x00180000, 14, COEFF_TOKEN(1,16) },  // 0000 0000 0001 10
  { 0x00140000, 14, COEFF_TOKEN(2,16) },  // 0000 0000 0001 01
  { 0x00100000, 14, COEFF_TOKEN(3,16) },  // 0000 0000 0001 00
},{
///// 4  <=  nC  <  8 /////
  { 0xF0000000,  4, COEFF_TOKEN(0, 0) },  // 1111
  { 0xE0000000,  4, COEFF_TOKEN(1, 1) },  // 1110
  { 0xD0000000,  4, COEFF_TOKEN(2, 2) },  // 1101
  { 0xC0000000,  4, COEFF_TOKEN(3, 3) },  // 1100
  { 0xB0000000,  4, COEFF_TOKEN(3, 4) },  // 1011
  { 0xA0000000,  4, COEFF_TOKEN(3, 5) },  // 1010
  { 0x90000000,  4, COEFF_TOKEN(3, 6) },  // 1001
  { 0x80000000,  4, COEFF_TOKEN(3, 7) },  // 1000
  { 0x78000000,  5, COEFF_TOKEN(1, 2) },  // 0111 1
  { 0x70000000,  5, COEFF_TOKEN(2, 3) },  // 0111 0
  { 0x68000000,  5, COEFF_TOKEN(3, 8) },  // 0110 1
  { 0x60000000,  5, COEFF_TOKEN(1, 3) },  // 0110 0
  { 0x58000000,  5, COEFF_TOKEN(2, 4) },  // 0101 1
  { 0x50000000,  5, COEFF_TOKEN(1, 4) },  // 0101 0
  { 0x48000000,  5, COEFF_TOKEN(2, 5) },  // 0100 1
  { 0x40000000,  5, COEFF_TOKEN(1, 5) },  // 0100 0
  { 0x3C000000,  6, COEFF_TOKEN(0, 1) },  // 0011 11
  { 0x38000000,  6, COEFF_TOKEN(1, 6) },  // 0011 10
  { 0x34000000,  6, COEFF_TOKEN(2, 6) },  // 0011 01
  { 0x30000000,  6, COEFF_TOKEN(3, 9) },  // 0011 00
  { 0x2C000000,  6, COEFF_TOKEN(0, 2) },  // 0010 11
  { 0x28000000,  6, COEFF_TOKEN(1, 7) },  // 0010 10
  { 0x24000000,  6, COEFF_TOKEN(2, 7) },  // 0010 01
  { 0x20000000,  6, COEFF_TOKEN(0, 3) },  // 0010 00
  { 0x1E000000,  7, COEFF_TOKEN(0, 4) },  // 0001 111
  { 0x1C000000,  7, COEFF_TOKEN(1, 8) },  // 0001 110
  { 0x1A000000,  7, COEFF_TOKEN(2, 8) },  // 0001 101
  { 0x18000000,  7, COEFF_TOKEN(3,10) },  // 0001 100
  { 0x16000000,  7, COEFF_TOKEN(0, 5) },  // 0001 011
  { 0x14000000,  7, COEFF_TOKEN(2, 9) },  // 0001 010
  { 0x12000000,  7, COEFF_TOKEN(0, 6) },  // 0001 001
  { 0x10000000,  7, COEFF_TOKEN(0, 7) },  // 0001 000
  { 0x0F000000,  8, COEFF_TOKEN(0, 8) },  // 0000 1111
  { 0x0E000000,  8, COEFF_TOKEN(1, 9) },  // 0000 1110
  { 0x0D000000,  8, COEFF_TOKEN(2,10) },  // 0000 1101
  { 0x0C000000,  8, COEFF_TOKEN(3,11) },  // 0000 1100
  { 0x0B000000,  8, COEFF_TOKEN(0, 9) },  // 0000 1011
  { 0x0A000000,  8, COEFF_TOKEN(1,10) },  // 0000 1010
  { 0x09000000,  8, COEFF_TOKEN(2,11) },  // 0000 1001
  { 0x08000000,  8, COEFF_TOKEN(3,12) },  // 0000 1000
  { 0x07800000,  9, COEFF_TOKEN(0,10) },  // 0000 0111 1
  { 0x07000000,  9, COEFF_TOKEN(1,11) },  // 0000 0111 0
  { 0x06800000,  9, COEFF_TOKEN(2,12) },  // 0000 0110 1
  { 0x06000000,  9, COEFF_TOKEN(3,13) },  // 0000 0110 0
  { 0x05800000,  9, COEFF_TOKEN(0,11) },  // 0000 0101 1
  { 0x05000000,  9, COEFF_TOKEN(1,12) },  // 0000 0101 0
  { 0x04800000,  9, COEFF_TOKEN(2,13) },  // 0000 0100 1
  { 0x04000000,  9, COEFF_TOKEN(0,12) },  // 0000 0100 0
  { 0x03800000,  9, COEFF_TOKEN(1,13) },  // 0000 0011 1
  { 0x03400000, 10, COEFF_TOKEN(0,13) },  // 0000 0011 01
  { 0x03000000, 10, COEFF_TOKEN(1,14) },  // 0000 0011 00
  { 0x02C00000, 10, COEFF_TOKEN(2,14) },  // 0000 0010 11
  { 0x02800000, 10, COEFF_TOKEN(3,14) },  // 0000 0010 10
  { 0x02400000, 10, COEFF_TOKEN(0,14) },  // 0000 0010 01
  { 0x02000000, 10, COEFF_TOKEN(1,15) },  // 0000 0010 00
  { 0x01C00000, 10, COEFF_TOKEN(2,15) },  // 0000 0001 11
  { 0x01800000, 10, COEFF_TOKEN(3,15) },  // 0000 0001 10
  { 0x01400000, 10, COEFF_TOKEN(0,15) },  // 0000 0001 01
  { 0x01000000, 10, COEFF_TOKEN(1,16) },  // 0000 0001 00
  { 0x00C00000, 10, COEFF_TOKEN(2,16) },  // 0000 0000 11
  { 0x00800000, 10, COEFF_TOKEN(3,16) },  // 0000 0000 10
  { 0x00400000, 10, COEFF_TOKEN(0,16) },  // 0000 0000 01
},{
///// 8  <=  nC /////
  { 0xFC000000,  6, COEFF_TOKEN(3,16) },  // 1111 11
  { 0xF8000000,  6, COEFF_TOKEN(2,16) },  // 1111 10
  { 0xF4000000,  6, COEFF_TOKEN(1,16) },  // 1111 01
  { 0xF0000000,  6, COEFF_TOKEN(0,16) },  // 1111 00
  { 0xEC000000,  6, COEFF_TOKEN(3,15) },  // 1110 11
  { 0xE8000000,  6, COEFF_TOKEN(2,15) },  // 1110 10
  { 0xE4000000,  6, COEFF_TOKEN(1,15) },  // 1110 01
  { 0xE0000000,  6, COEFF_TOKEN(0,15) },  // 1110 00
  { 0xDC000000,  6, COEFF_TOKEN(3,14) },  // 1101 11
  { 0xD8000000,  6, COEFF_TOKEN(2,14) },  // 1101 10
  { 0xD4000000,  6, COEFF_TOKEN(1,14) },  // 1101 01
  { 0xD0000000,  6, COEFF_TOKEN(0,14) },  // 1101 00
  { 0xCC000000,  6, COEFF_TOKEN(3,13) },  // 1100 11
  { 0xC8000000,  6, COEFF_TOKEN(2,13) },  // 1100 10
  { 0xC4000000,  6, COEFF_TOKEN(1,13) },  // 1100 01
  { 0xC0000000,  6, COEFF_TOKEN(0,13) },  // 1100 00
  { 0xBC000000,  6, COEFF_TOKEN(3,12) },  // 1011 11
  { 0xB8000000,  6, COEFF_TOKEN(2,12) },  // 1011 10
  { 0xB4000000,  6, COEFF_TOKEN(1,12) },  // 1011 01
  { 0xB0000000,  6, COEFF_TOKEN(0,12) },  // 1011 00
  { 0xAC000000,  6, COEFF_TOKEN(3,11) },  // 1010 11
  { 0xA8000000,  6, COEFF_TOKEN(2,11) },  // 1010 10
  { 0xA4000000,  6, COEFF_TOKEN(1,11) },  // 1010 01
  { 0xA0000000,  6, COEFF_TOKEN(0,11) },  // 1010 00
  { 0x9C000000,  6, COEFF_TOKEN(3,10) },  // 1001 11
  { 0x98000000,  6, COEFF_TOKEN(2,10) },  // 1001 10
  { 0x94000000,  6, COEFF_TOKEN(1,10) },  // 1001 01
  { 0x90000000,  6, COEFF_TOKEN(0,10) },  // 1001 00
  { 0x8C000000,  6, COEFF_TOKEN(3, 9) },  // 1000 11
  { 0x88000000,  6, COEFF_TOKEN(2, 9) },  // 1000 10
  { 0x84000000,  6, COEFF_TOKEN(1, 9) },  // 1000 01
  { 0x80000000,  6, COEFF_TOKEN(0, 9) },  // 1000 00
  { 0x7C000000,  6, COEFF_TOKEN(3, 8) },  // 0111 11
  { 0x78000000,  6, COEFF_TOKEN(2, 8) },  // 0111 10
  { 0x74000000,  6, COEFF_TOKEN(1, 8) },  // 0111 01
  { 0x70000000,  6, COEFF_TOKEN(0, 8) },  // 0111 00
  { 0x6C000000,  6, COEFF_TOKEN(3, 7) },  // 0110 11
  { 0x68000000,  6, COEFF_TOKEN(2, 7) },  // 0110 10
  { 0x64000000,  6, COEFF_TOKEN(1, 7) },  // 0110 01
  { 0x60000000,  6, COEFF_TOKEN(0, 7) },  // 0110 00
  { 0x5C000000,  6, COEFF_TOKEN(3, 6) },  // 0101 11
  { 0x58000000,  6, COEFF_TOKEN(2, 6) },  // 0101 10
  { 0x54000000,  6, COEFF_TOKEN(1, 6) },  // 0101 01
  { 0x50000000,  6, COEFF_TOKEN(0, 6) },  // 0101 00
  { 0x4C000000,  6, COEFF_TOKEN(3, 5) },  // 0100 11
  { 0x48000000,  6, COEFF_TOKEN(2, 5) },  // 0100 10
  { 0x44000000,  6, COEFF_TOKEN(1, 5) },  // 0100 01
  { 0x40000000,  6, COEFF_TOKEN(0, 5) },  // 0100 00
  { 0x3C000000,  6, COEFF_TOKEN(3, 4) },  // 0011 11
  { 0x38000000,  6, COEFF_TOKEN(2, 4) },  // 0011 10
  { 0x34000000,  6, COEFF_TOKEN(1, 4) },  // 0011 01
  { 0x30000000,  6, COEFF_TOKEN(0, 4) },  // 0011 00
  { 0x2C000000,  6, COEFF_TOKEN(3, 3) },  // 0010 11
  { 0x28000000,  6, COEFF_TOKEN(2, 3) },  // 0010 10
  { 0x24000000,  6, COEFF_TOKEN(1, 3) },  // 0010 01
  { 0x20000000,  6, COEFF_TOKEN(0, 3) },  // 0010 00
  { 0x18000000,  6, COEFF_TOKEN(2, 2) },  // 0001 10
  { 0x14000000,  6, COEFF_TOKEN(1, 2) },  // 0001 01
  { 0x10000000,  6, COEFF_TOKEN(0, 2) },  // 0001 00
  { 0x0C000000,  6, COEFF_TOKEN(0, 0) },  // 0000 11
  { 0x04000000,  6, COEFF_TOKEN(1, 1) },  // 0000 01
  { 0x00000000,  6, COEFF_TOKEN(0, 1) },  // 0000 00
} };

static const code_item_t CoeffTokenCodes_ChromaDC[14]={
  { 0x80000000,  1, COEFF_TOKEN(1, 1) },  // 1
  { 0x40000000,  2, COEFF_TOKEN(0, 0) },  // 01
  { 0x20000000,  3, COEFF_TOKEN(2, 2) },  // 001
  { 0x1C000000,  6, COEFF_TOKEN(0, 1) },  // 0001 11
  { 0x18000000,  6, COEFF_TOKEN(1, 2) },  // 0001 10
  { 0x14000000,  6, COEFF_TOKEN(3, 3) },  // 0001 01
  { 0x10000000,  6, COEFF_TOKEN(0, 2) },  // 0001 00
  { 0x0C000000,  6, COEFF_TOKEN(0, 3) },  // 0000 11
  { 0x08000000,  6, COEFF_TOKEN(0, 4) },  // 0000 10
  { 0x06000000,  7, COEFF_TOKEN(1, 3) },  // 0000 011
  { 0x04000000,  7, COEFF_TOKEN(2, 3) },  // 0000 010
  { 0x00000000,  7, COEFF_TOKEN(3, 4) },  // 0000 000
  { 0x03000000,  8, COEFF_TOKEN(1, 4) },  // 0000 0011
  { 0x02000000,  8, COEFF_TOKEN(2, 4) },  // 0000 0010
};

static const code_item_t TotalZerosCodes_4x4[15][16]={ {
///// 1 /////
  { 0x80000000,  1,  0 },  // 1
  { 0x60000000,  3,  1 },  // 011
  { 0x40000000,  3,  2 },  // 010
  { 0x30000000,  4,  3 },  // 0011
  { 0x20000000,  4,  4 },  // 0010
  { 0x18000000,  5,  5 },  // 0001 1
  { 0x10000000,  5,  6 },  // 0001 0
  { 0x0C000000,  6,  7 },  // 0000 11
  { 0x08000000,  6,  8 },  // 0000 10
  { 0x06000000,  7,  9 },  // 0000 011
  { 0x04000000,  7, 10 },  // 0000 010
  { 0x03000000,  8, 11 },  // 0000 0011
  { 0x02000000,  8, 12 },  // 0000 0010
  { 0x01800000,  9, 13 },  // 0000 0001 1
  { 0x01000000,  9, 14 },  // 0000 0001 0
  { 0x00800000,  9, 15 },  // 0000 0000 1
},{
///// 2 /////
  { 0xE0000000,  3,  0 },  // 111
  { 0xC0000000,  3,  1 },  // 110
  { 0xA0000000,  3,  2 },  // 101
  { 0x80000000,  3,  3 },  // 100
  { 0x60000000,  3,  4 },  // 011
  { 0x50000000,  4,  5 },  // 0101
  { 0x40000000,  4,  6 },  // 0100
  { 0x30000000,  4,  7 },  // 0011
  { 0x20000000,  4,  8 },  // 0010
  { 0x18000000,  5,  9 },  // 0001 1
  { 0x10000000,  5, 10 },  // 0001 0
  { 0x0C000000,  6, 11 },  // 0000 11
  { 0x08000000,  6, 12 },  // 0000 10
  { 0x04000000,  6, 13 },  // 0000 01
  { 0x00000000,  6, 14 },  // 0000 00
},{
///// 3 /////
  { 0xE0000000,  3,  1 },  // 111
  { 0xC0000000,  3,  2 },  // 110
  { 0xA0000000,  3,  3 },  // 101
  { 0x80000000,  3,  6 },  // 100
  { 0x60000000,  3,  7 },  // 011
  { 0x50000000,  4,  0 },  // 0101
  { 0x40000000,  4,  4 },  // 0100
  { 0x30000000,  4,  5 },  // 0011
  { 0x20000000,  4,  8 },  // 0010
  { 0x18000000,  5,  9 },  // 0001 1
  { 0x10000000,  5, 10 },  // 0001 0
  { 0x08000000,  5, 12 },  // 0000 1
  { 0x04000000,  6, 11 },  // 0000 01
  { 0x00000000,  6, 13 },  // 0000 00
},{
///// 4 /////
  { 0xE0000000,  3,  1 },  // 111
  { 0xC0000000,  3,  4 },  // 110
  { 0xA0000000,  3,  5 },  // 101
  { 0x80000000,  3,  6 },  // 100
  { 0x60000000,  3,  8 },  // 011
  { 0x50000000,  4,  2 },  // 0101
  { 0x40000000,  4,  3 },  // 0100
  { 0x30000000,  4,  7 },  // 0011
  { 0x20000000,  4,  9 },  // 0010
  { 0x18000000,  5,  0 },  // 0001 1
  { 0x10000000,  5, 10 },  // 0001 0
  { 0x08000000,  5, 11 },  // 0000 1
  { 0x00000000,  5, 12 },  // 0000 0
},{
///// 5 /////
  { 0xE0000000,  3,  3 },  // 111
  { 0xC0000000,  3,  4 },  // 110
  { 0xA0000000,  3,  5 },  // 101
  { 0x80000000,  3,  6 },  // 100
  { 0x60000000,  3,  7 },  // 011
  { 0x50000000,  4,  0 },  // 0101
  { 0x40000000,  4,  1 },  // 0100
  { 0x30000000,  4,  2 },  // 0011
  { 0x20000000,  4,  8 },  // 0010
  { 0x10000000,  4, 10 },  // 0001
  { 0x08000000,  5,  9 },  // 0000 1
  { 0x00000000,  5, 11 },  // 0000 0
},{
///// 6 /////
  { 0xE0000000,  3,  2 },  // 111
  { 0xC0000000,  3,  3 },  // 110
  { 0xA0000000,  3,  4 },  // 101
  { 0x80000000,  3,  5 },  // 100
  { 0x60000000,  3,  6 },  // 011
  { 0x40000000,  3,  7 },  // 010
  { 0x20000000,  3,  9 },  // 001
  { 0x10000000,  4,  8 },  // 0001
  { 0x08000000,  5,  1 },  // 0000 1
  { 0x04000000,  6,  0 },  // 0000 01
  { 0x00000000,  6, 10 },  // 0000 00
},{
///// 7 /////
  { 0xC0000000,  2, 5 },  // 11
  { 0xA0000000,  3, 2 },  // 101
  { 0x80000000,  3, 3 },  // 100
  { 0x60000000,  3, 4 },  // 011
  { 0x40000000,  3, 6 },  // 010
  { 0x20000000,  3, 8 },  // 001
  { 0x10000000,  4, 7 },  // 0001
  { 0x08000000,  5, 1 },  // 0000 1
  { 0x04000000,  6, 0 },  // 0000 01
  { 0x00000000,  6, 9 },  // 0000 00
},{
///// 8 /////
  { 0xC0000000,  2, 4 },  // 11
  { 0x80000000,  2, 5 },  // 10
  { 0x60000000,  3, 3 },  // 011
  { 0x40000000,  3, 6 },  // 010
  { 0x20000000,  3, 7 },  // 001
  { 0x10000000,  4, 1 },  // 0001
  { 0x08000000,  5, 2 },  // 0000 1
  { 0x04000000,  6, 0 },  // 0000 01
  { 0x00000000,  6, 8 },  // 0000 00
},{
///// 9 /////
  { 0xC0000000,  2, 3 },  // 11
  { 0x80000000,  2, 4 },  // 10
  { 0x40000000,  2, 6 },  // 01
  { 0x20000000,  3, 5 },  // 001
  { 0x10000000,  4, 2 },  // 0001
  { 0x08000000,  5, 7 },  // 0000 1
  { 0x04000000,  6, 0 },  // 0000 01
  { 0x00000000,  6, 1 },  // 0000 00
},{
///// 10 /////
  { 0xC0000000,  2, 3 },  // 11
  { 0x80000000,  2, 4 },  // 10
  { 0x40000000,  2, 5 },  // 01
  { 0x20000000,  3, 2 },  // 001
  { 0x10000000,  4, 6 },  // 0001
  { 0x08000000,  5, 0 },  // 0000 1
  { 0x00000000,  5, 1 },  // 0000 0
},{
///// 11 /////
  { 0x80000000,  1, 4 },  // 1
  { 0x60000000,  3, 5 },  // 011
  { 0x40000000,  3, 3 },  // 010
  { 0x20000000,  3, 2 },  // 001
  { 0x10000000,  4, 1 },  // 0001
  { 0x00000000,  4, 0 },  // 0000
},{
///// 12 /////
  { 0x80000000,  1, 3 },  // 1
  { 0x40000000,  2, 2 },  // 01
  { 0x20000000,  3, 4 },  // 001
  { 0x10000000,  4, 1 },  // 0001
  { 0x00000000,  4, 0 },  // 0000
},{
///// 13 /////
  { 0x80000000,  1, 2 },  // 1
  { 0x40000000,  2, 3 },  // 01
  { 0x20000000,  3, 1 },  // 001
  { 0x00000000,  3, 0 },  // 000
},{
///// 14 /////
  { 0x80000000,  1, 2 },  // 1
  { 0x40000000,  2, 1 },  // 01
  { 0x00000000,  2, 0 },  // 00
},{
///// 15 /////
  { 0x80000000,  1, 1 },  // 1
  { 0x00000000,  1, 0 },  // 0
} };

static const code_item_t TotalZerosCodes_ChromaDC[3][4]={ {
///// 1 /////
  { 0x80000000,  1, 0 },  // 1
  { 0x40000000,  2, 1 },  // 01
  { 0x20000000,  3, 2 },  // 001
  { 0x00000000,  3, 3 },  // 000
},{
///// 2 /////
  { 0x80000000,  1, 0 },  // 1
  { 0x40000000,  2, 1 },  // 01
  { 0x00000000,  2, 2 },  // 00
},{
///// 3 /////
  { 0x80000000,  1, 0 },  // 1
  { 0x00000000,  1, 1 },  // 0
} };

static const code_item_t RunBeforeCodes[6][7]={ {
///// 1 /////
  { 0x80000000,  1, 0 },  // 1
  { 0x00000000,  1, 1 },  // 0
},{
///// 2 /////
  { 0x80000000,  1, 0 },  // 1
  { 0x40000000,  2, 1 },  // 01
  { 0x00000000,  2, 2 },  // 00
},{
///// 3 /////
  { 0xC0000000,  2, 0 },  // 11
  { 0x80000000,  2, 1 },  // 10
  { 0x40000000,  2, 2 },  // 01
  { 0x00000000,  2, 3 },  // 00
},{
///// 4 /////
  { 0xC0000000,  2, 0 },  // 11
  { 0x80000000,  2, 1 },  // 10
  { 0x40000000,  2, 2 },  // 01
  { 0x20000000,  3, 3 },  // 001
  { 0x00000000,  3, 4 },  // 000
},{
///// 5 /////
  { 0xC0000000,  2, 0 },  // 11
  { 0x80000000,  2, 1 },  // 10
  { 0x60000000,  3, 2 },  // 011
  { 0x40000000,  3, 3 },  // 010
  { 0x20000000,  3, 4 },  // 001
  { 0x00000000,  3, 5 },  // 000
},{
///// 6 /////
  { 0xC0000000,  2, 0 },  // 11
  { 0xA0000000,  3, 5 },  // 101
  { 0x80000000,  3, 6 },  // 100
  { 0x60000000,  3, 3 },  // 011
  { 0x40000000,  3, 4 },  // 010
  { 0x20000000,  3, 2 },  // 001
  { 0x00000000,  3, 1 },  // 000
} };

static int get_code(const code_item_t *table, int size, bitreader_t *bits)
{
    unsigned int code = 0;
    
    int prevnbbits = 0;
    int i;
    for (i=0; i<size; i++)
    {
        int diffbits = table[i].nbbits - prevnbbits;
        if (diffbits > 0)
        {
            int ret = bitreader_get(bits, diffbits);
            code = (code << diffbits) | ret;
            prevnbbits = table[i].nbbits;
        }

        if ((code << (32 - table[i].nbbits)) == table[i].code)
            return table[i].data;
    }
    /* does not happen */
    srl_assert(0);
    return 0;
}

int residual_block (short int *coeffLevel, int maxNumCoeff, int nC, bitreader_t *bits)
{
    int coeff_token, TotalCoeff, TrailingOnes;
    int i, suffixLength, zerosLeft, coeffNum;
    int level[16], run[16];

    /* Init coeffLevel */
    memset (coeffLevel, 0, maxNumCoeff*sizeof(short int));
	//for (i=0; i<maxNumCoeff; i++)
	//	coeffLevel[i] = 0;
    
    srl_log(DEBUG, "coeff_token :\n");
    switch (nC)
    {
        case -1:
            coeff_token = get_code (CoeffTokenCodes_ChromaDC, 14, bits);
            break;
        case 0:
        case 1:
            coeff_token = get_code (CoeffTokenCodes[0], 62, bits);
            break;
        case 2:
        case 3:
            coeff_token = get_code (CoeffTokenCodes[1], 62, bits);
            break;
        case 4:
        case 5:
        case 6:
        case 7:
            coeff_token = get_code (CoeffTokenCodes[2], 62, bits);
            break;
        default:
            coeff_token = get_code (CoeffTokenCodes[3], 62, bits);
    }

    srl_log_printf(DEBUG, "Coeff_token = %X (total_coeff=%d; trailing_ones=%d)\n", 
                    coeff_token, coeff_token>>2, coeff_token&3);
    TotalCoeff = coeff_token >> 2;
    TrailingOnes = coeff_token & 3;

    /* no coefficients */
    if (!TotalCoeff)
        return 0;

    if (TotalCoeff > 10 && TrailingOnes < 3)
        suffixLength = 1;
    else
        suffixLength = 0;

#if 0
    for (i = 0; i < TotalCoeff; ++i)
    {
        if (i < TrailingOnes)
            level[i] = 1 - 2 * bitreader_get_one(bits);
        else
        {
#else

    srl_log(DEBUG, "Trailing Ones :\n");
    for (i = 0; i < TrailingOnes && i < TotalCoeff; i++)
        level[i] = 1 - 2 * bitreader_get_one(bits);

    srl_log(DEBUG, "Levels :\n");
    for (; i < TotalCoeff; i++)
    {
        {
#endif
            int level_prefix;
            int levelSuffixSize = suffixLength;
            int levelCode;
            for (level_prefix=0; !bitreader_get_one(bits); ++level_prefix);
            levelCode = min(15, level_prefix) << suffixLength;

            if (level_prefix == 14 && suffixLength == 0)
                levelSuffixSize = 4;
            else if (level_prefix == 15)
                levelSuffixSize = 12;
            if (levelSuffixSize)
                levelCode += bitreader_get(bits, levelSuffixSize);
            if (level_prefix == 15 && suffixLength == 0)
                levelCode += 15;
            if (i == TrailingOnes && TrailingOnes < 3)
                levelCode += 2;
            if (levelCode & 1)
                level[i] = (-levelCode - 1) >> 1;
            else
                level[i] = (levelCode + 2) >> 1;
            if (suffixLength == 0)
                suffixLength = 1;
            if (abs (level[i]) > (3 << (suffixLength - 1)) && suffixLength < 6)
                ++suffixLength;
        }
        srl_log_printf(DEBUG, "level[%d] = %X\n", TotalCoeff-i-1, level[i]);
    }

    srl_log(DEBUG, "zerosLeft :\n");
    if (TotalCoeff < maxNumCoeff)
    {
        if (nC < 0)
            zerosLeft = get_code(TotalZerosCodes_ChromaDC[TotalCoeff - 1], 4, bits);
        else
            zerosLeft = get_code(TotalZerosCodes_4x4[TotalCoeff - 1], 16, bits);
    }
    else
        zerosLeft = 0;

    srl_log_printf(DEBUG, "zerosLeft = %X\n", zerosLeft);

    srl_log(DEBUG, "Runs :\n");
    for (i = 0; i < TotalCoeff - 1; ++i)
    {
        if (zerosLeft > 6)
        {
            int run_before = 7 - bitreader_get(bits, 3);
            if (run_before == 7)
                while (!bitreader_get_one(bits))
                    ++run_before;
            run[i] = run_before;
        }
        else if (zerosLeft > 0)
            run[i] = get_code (RunBeforeCodes[zerosLeft - 1], 7, bits);
        else
            run[i] = 0;

        zerosLeft -= run[i];
        srl_assert(zerosLeft >= 0);

        srl_log_printf(DEBUG, "run[%d] = %X\n", TotalCoeff-i-2, run[i]);
    }
    run[TotalCoeff - 1] = zerosLeft;

    coeffNum = -1;
    for (i = TotalCoeff - 1; i >= 0; --i)
    {
        coeffNum += run[i] + 1;
        coeffLevel[coeffNum] = level[i];
    }

    return TotalCoeff;
}

