/* This file is part of DSX.
 *
 * DSX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DSX is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with DSX; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Copyright (c) Lip6, Thalès
 *      Joel Porquet <joel.porquet@lip6.fr>, 2006-2007
 *
 * Based on Martin Fielder's work (http://keyj.s2000.ws/?page_id=41)
 */

#include <srl.h>
#include "bitstream_io.h"

/*
 * Block_io
 */

/* Init Functions */
static void block_io_init_out ( block_io_t *block, void *buffer, 
                                int size, srl_mwmr_t mwmr)
{
    block->type = BLOCK_IO_OUT;
    block->io = mwmr;
    block->buffer = buffer;
    block->ptr = block->buffer;
    block->size = size;
    block->left = size;
}

static void block_io_init_in ( block_io_t *block, void *buffer, 
                               int size, srl_mwmr_t mwmr)
{
    block->type = BLOCK_IO_IN;
    block->io = mwmr;
    block->buffer = buffer;
    block->ptr = block->buffer;
    block->size = size;
    block->left = 0;
}

/* I/O Functions  */
static inline unsigned char block_io_read_char (block_io_t *block)
{
    srl_assert(block->type == BLOCK_IO_IN);
    srl_assert(block->left != 0);

    unsigned char ret;
    block->left -= 1;
    ret = *(block->ptr++);
    srl_log_printf(DEBUG, "bio_read_char: %02x\n", ret);
    return ret;
}

static inline void block_io_write_char (block_io_t *block, unsigned char val)
{
    srl_assert(block->type == BLOCK_IO_OUT);
    srl_assert(block->left != 0);

    block->left -= 1;
    *(block->ptr++) = val;
}

/* Misc Functions */
static inline unsigned char block_io_isempty (block_io_t *block)
{
    srl_assert(block->type == BLOCK_IO_IN);

    return (block->left == 0);
}

static inline unsigned char block_io_isfull (block_io_t *block)
{
    srl_assert(block->type == BLOCK_IO_OUT);

    return (block->left == 0);
}

static inline void block_io_fill (block_io_t *block)
{
    srl_assert(block->type == BLOCK_IO_IN);

    srl_log(DEBUG, "Refilling block\n" );  
    srl_mwmr_read(block->io, block->buffer, block->size); 
    int i;
    srl_log_printf(DEBUG, "bio_refill: ");
    for (i=0; i<block->size; i++)
        srl_log_printf(DEBUG, "%X ", block->ptr[i]);
    srl_log_printf(DEBUG, "\n");
    block->left = block->size;                 
    block->ptr = block->buffer;                
}

static inline void block_io_flush (block_io_t *block)
{
    srl_assert(block->type == BLOCK_IO_OUT);

    srl_log(DEBUG, "Flushing block\n" );   
    srl_mwmr_write(block->io, block->buffer, block->size);
    block->left = block->size;                 
    block->ptr = block->buffer;                
}


/*
 * Nal unit
 */

/* Init function */
void nal_unit_init_in (nal_unit_t *nal, void *buffer, int size, srl_mwmr_t mwmr)
{
    block_io_init_in(&nal->block, buffer, size, mwmr);
    nal->current = 0xFFFFFFFF;
}

/* I/O functions */
static inline unsigned char _nal_unit_read_char (nal_unit_t *nal)
{
    if (block_io_isempty(&nal->block))
        block_io_fill(&nal->block);

    return block_io_read_char(&nal->block);
}

unsigned char nal_unit_read_char (void *io)
{
    nal_unit_t *nal = (nal_unit_t*) io;

    srl_assert(!nal_unit_isend(nal));

    unsigned char ret = nal->current >> 24;
    
    nal->current <<= 8;
    nal->current |= _nal_unit_read_char(nal) ;

    return ret;
}

/* Misc functions */
void nal_unit_get_next(nal_unit_t *nal)
{
    while (nal->current != 0x00000001 && (nal->current&0xFFFFFF00) != 0x00000100)
    {
        nal->current <<= 8;
        nal->current |= _nal_unit_read_char(nal);
    }

    if ((nal->current&0xFFFFFF00) != 0x00000100)
    {
        nal->current = 0;
        nal->current |= _nal_unit_read_char(nal) << 24 ;
    } else
        nal->current <<= 24;

    nal->current |= _nal_unit_read_char(nal) << 16 ;
    nal->current |= _nal_unit_read_char(nal) << 8 ;
    nal->current |= _nal_unit_read_char(nal) ;
}

unsigned char nal_unit_isend(nal_unit_t *nal)
{
    return (nal->current == 0x00000001 || (nal->current&0xFFFFFF00) == 0x00000100);
}


/*
 * Slice
 */

/* for thread_slice_demux */
void slice_init_out (slice_t *slice, srl_mwmr_t mwmr)
{
    slice->type = SLICE_OUT;
    slice->io = mwmr;
    slice->index = 0;
}

void slice_write_char (slice_t *slice, unsigned char val)
{
    srl_assert(slice->type == SLICE_OUT);

    slice->buffer[++slice->index] = val;

    srl_log_printf(DEBUG,"slice_write_char = %X\n", slice->buffer[slice->index]);

    if (slice->index == SLICE_WIDTH - 1)
        slice_end(slice);
}

void slice_end (slice_t *slice)
{
    srl_assert(slice->type == SLICE_OUT);

    slice->buffer[0] = slice->index;
    srl_mwmr_write(slice->io, slice->buffer, SLICE_WIDTH);
    srl_log_printf(DEBUG,"Sending a new slice segment, size = %d\n", slice->buffer[0]);
    slice->index = 0;
}

/* for thread_decode */
void slice_init_in (slice_t *slice, srl_mwmr_t mwmr)
{
    slice->type = SLICE_IN;
    slice->io = mwmr;
    slice->index = 0;
}

unsigned char slice_isover (slice_t *slice)
{
    srl_assert(slice->type == SLICE_IN);

    if (slice->buffer[0] != SLICE_WIDTH-1 && slice->index == slice->buffer[0])
        return 1;
    else if (slice->buffer[0] == SLICE_WIDTH-1 && slice->index == SLICE_WIDTH-1)
    {
        slice_begin(slice);
        if (slice->buffer[0] == 0)
            return 1;
    }

    return 0;
}

unsigned char slice_read_char (void *io)
{
    slice_t *slice = (slice_t*) io;

    srl_assert(!slice_isover(slice));

    srl_log_printf(DEBUG,"slice_read_char = %X\n", slice->buffer[slice->index+1]);
    return slice->buffer[++slice->index];
}

void slice_begin (slice_t *slice)
{
    srl_assert(slice->type == SLICE_IN);

    srl_mwmr_read(slice->io, slice->buffer, SLICE_WIDTH);
    srl_log_printf(DEBUG,"Receiving a new slice segment, size = %d\n", slice->buffer[0]);
    slice->index = 0;
}


/*
 * Bitreader
 */

/* Init function */
void bitreader_init(bitreader_t *bitreader, void *io, unsigned char (*read_char)(void *))
{
    bitreader->available = 0;
    bitreader->current = 0;
    bitreader->io = io;
    bitreader->read_char = read_char;
}

/* I/O functions */
unsigned long bitreader_get(bitreader_t *bitreader, int nb)
{
    unsigned int ret = 0;
    unsigned int number = nb;

    if (bitreader->available)
        bitreader->current &= (1<<bitreader->available)-1;

    while (number)
    {
        if ( bitreader->available == 0 )
        {
            bitreader->current = bitreader->read_char( bitreader->io );
            bitreader->available = 8;
        }
        if ( number == bitreader->available )
        {
            bitreader->available = 0;
            ret = (ret<<number) | bitreader->current;
            break;
        }
        if ( number < bitreader->available )
        {
            ret = (ret<<number) | (bitreader->current>>(bitreader->available-number));
            bitreader->available -= number;
            break;
        }
        if ( number > bitreader->available )
        {
            ret = (ret<<bitreader->available) | bitreader->current;
            number -= bitreader->available;
            bitreader->available = 0;
        }
    }
    
    //srl_log_printf(DEBUG, "BITS Getting %d bits: %x\n", nb, ret);
    {
        int i;
        srl_log_printf(DEBUG, "BITS Getting %d bits: ", nb);
        
        for (i=0; i<nb; i++)
        {
            unsigned int tmp = ret >> (nb-i-1);
            tmp &= 0x1;
            srl_log_printf(DEBUG, "%d", tmp);
        }
        srl_log(DEBUG, "\n");
    }
    return ret;
}

unsigned char bitreader_get_one(bitreader_t *bitreader)
{
    unsigned int ret = 0, tmp;

    if ( bitreader->available == 0 )
    {
        bitreader->current = bitreader->read_char( bitreader->io );
        bitreader->available = 7;
    } else
        --(bitreader->available);
        
    tmp = 1<<(bitreader->available);
    ret = !!(bitreader->current&tmp);
    srl_log_printf(DEBUG, "BITS Getting 1 bits: %x\n", ret);
    return ret;
}

int bitreader_get_ugolomb(bitreader_t *bitreader)
{
  int exp;
  for(exp=0; !bitreader_get_one(bitreader); ++exp);
  
  if(exp) 
      return (1<<exp)-1+bitreader_get(bitreader, exp);
  else 
      return 0;
}

int bitreader_get_sgolomb(bitreader_t *bitreader)
{
  int code=bitreader_get_ugolomb(bitreader);
  return (code&1) ? (code+1)>>1 : -(code>>1);
}

/* Misc functions */
void bitreader_refill(bitreader_t *bitreader)
{
    if ( bitreader->available == 0 ) 
    {
        bitreader->current = bitreader->read_char( bitreader->io );
        bitreader->available = 8;
    }
}

unsigned char bitreader_isover(bitreader_t *bitreader)
{
    srl_assert( bitreader->available > 0 );

    srl_log_printf(DEBUG, "bitreader_isover : %02X ; available : %d\n", 
                          bitreader->current, bitreader->available);
    
    unsigned char tmp = 1 << (bitreader->available - 1);
    tmp = !!(bitreader->current&tmp);
    if (tmp == 0)
        return 0;

    int i;
    for (i=bitreader->available - 2; i>=0; i--)
    {
        tmp = 1 << i;
        tmp = !!(bitreader->current&tmp);
        if (tmp != 0)
            return 0;
    }

    return 1;
}

