/* This file is part of DSX.
 *
 * DSX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DSX is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with DSX; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Copyright (c) Lip6, Thalès
 *      Joel Porquet <joel.porquet@lip6.fr>, 2006-2007
 *
 * Based on Martin Fielder's work (http://keyj.s2000.ws/?page_id=41)
 */

#include "transform.h"
#include "utils.h"
#include "h264.h"

#include <srl.h>

/* arrays */
static const int LevelScale[6]   = {10,11,13,14,16,18};
static const int ZigZagOrder[16] = {0,1,4,8,5,2,3,6,9,12,13,10,7,11,14,15};
static const int ScanOrder[16]   = {0,1,4,5,2,3,6,7,8,9,12,13,10,11,14,15};

/* functions */
static void coeff_scan(core_block_t *res, short int *scan) 
{
    int i;
    for (i=0; i<16; ++i)
        res->items[ZigZagOrder[i]]=scan[i];
}

static void core_block_multiply(core_block_t *res, const core_block_t *a, const core_block_t *b)
{
    int i,j,k;
    short int sum;

    for(i=0; i<4; ++i)
    {
        for(j=0; j<4; ++j) 
        {
            sum=0;
            for(k=0; k<4; ++k)
                sum += CoreBlock(a, i, k) * CoreBlock(b, k, j);
            CoreBlock(res, i, j) = sum;
        }
    }
}

static void hadamard(core_block_t *res, const core_block_t *coeff)
{
    core_block_t tmp;
    static const core_block_t transform = {{1,1,1,1,1,1,-1,-1,1,-1,-1,1,1,-1,1,-1}};
    core_block_multiply(&tmp, coeff, &transform);
    core_block_multiply(res, &transform, &tmp);
}

static void inverse_quantize(core_block_t *res, core_block_t *quantized, int quantizer, int without_dc) 
{
    static const core_block_t factors[6] = {
	{{DQP0,DQP0}},{{DQP1,DQP1}},{{DQP2,DQP2}},
        {{DQP3,DQP3}},{{DQP4,DQP4}},{{DQP5,DQP5}}
    };
    int qbits = quantizer/6;
    int table = quantizer%6;
    int i;

    if (without_dc)
	res->items[0] = quantized->items[0];
    for ( i=without_dc; i<16; ++i )
    {
        short int value=quantized->items[i];
        res->items[i] = CombineSign(
	    ExtractSign(value),
	    (abs(value)*factors[table].items[i]) << qbits
	    );
    }
}

static void direct_idct(short int *scan, core_block_t *coeff)
{
    int i;
    short int e0,e1,e2,e3;
    short int *l = &(coeff->items[0]);

    for (i=0; i<4; ++i) 
    {
        e0 = l[0] + l[2];
        e1 = l[0] - l[2];
        e2 = (l[1]>>1) - l[3];
        e3 = l[1] + (l[3]>>1);

        *l++ = e0 + e3;
        *l++ = e1 + e2;
        *l++ = e1 - e2;
        *l++ = e0 - e3;
    }
    for (i=0; i<4; ++i)
    {
        l = &(coeff->items[i]);
        e0 = l[0] + l[8];
        e1 = l[0] - l[8];
        e2 = (l[4]>>1) - l[12];
        e3 = l[4] + (l[12]>>1);
        
        scan[0+i]  = ((e0+e3+32) >> 6);
        scan[4+i]  = ((e1+e2+32) >> 6);
        scan[8+i]  = ((e1-e2+32) >> 6);
        scan[12+i] = ((e0-e3+32) >> 6);
    }
}

/* transforms */
void transform_luma_dc (short int *in, short int *out, int qp)
{
    core_block_t block;
    int scale = LevelScale[qp%6];
    int i;

    coeff_scan(&block, in);
    hadamard(&block, &block);
    if (qp>=12)
    {
        for (i=0; i<16; ++i)
            out[ScanOrder[i]<<4] = (block.items[i]*scale)<<(qp/6-2);
    } else {
        int round_adj = 1<<(1-qp/6);
        for (i=0; i<16; ++i)
            out[ScanOrder[i]<<4] = (block.items[i]*scale + round_adj)>>(2-qp/6);
    }
}


void transform_luma_ac (short int *scan, int qp, int without_dc)
{
    core_block_t block, scan_b;
    coeff_scan(&scan_b, scan);
    inverse_quantize(&block, &scan_b, qp, without_dc);
    direct_idct(scan, &block);
}

void transform_chroma_dc (short int *in, short int *out, int qp)
{
    short int block[4];
    int scale = LevelScale[qp%6];
    int i;
    
    block[0] = in[0] + in[1] + in[2] + in[3];
    block[1] = in[0] - in[1] + in[2] - in[3];
    block[2] = in[0] + in[1] - in[2] - in[3];
    block[3] = in[0] - in[1] - in[2] + in[3];
    
    if (qp>=6) 
    {
        for (i=0; i<4; i++)
            out[i<<4] = (block[i]*scale) << (qp/6-1);
    } else {
        for (i=0; i<4; i++)
            out[i<<4] = (block[i]*scale) >> 1;
    }
}

