/* This file is part of DSX.
 *
 * DSX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DSX is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with DSX; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Copyright (c) Lip6, Thalès
 *      Joel Porquet <joel.porquet@lip6.fr>, 2006-2007
 *
 * Based on Martin Fielder's work (http://keyj.s2000.ws/?page_id=41)
 */

#include <srl.h>
#include "plane_pool.h"

#define LPLANE_SIZE WIDTH*HEIGHT
#define CPLANE_SIZE CWIDTH*CHEIGHT

void init_plane_pool(plane_pool_t *plane_pool, void *plane_buffers)
{
    /* assert there are at least two planes (one current and one previous) */
    srl_assert(NB_PLANE_BUFFERS >= 2);

    plane_pool->current_frame = 0;

    int i;
    for (i=0; i<NB_PLANE_BUFFERS; i++)
    {
        plane_pool->planes[i].L     = plane_buffers + i*(LPLANE_SIZE + 2*CPLANE_SIZE);
        plane_pool->planes[i].C[0]  = plane_pool->planes[i].L + LPLANE_SIZE;
        plane_pool->planes[i].C[1]  = plane_pool->planes[i].C[0] + CPLANE_SIZE;
        memset(plane_pool->planes[i].m_available, 0, MBLOCKS_Y*MBLOCKS_X);
    }
}

void get_plane_pool(plane_pool_t *plane_pool, unsigned int pic_count, frame_t *frame[2])
{
#if defined (SRL_SOCLIB)
    srl_sched_wait_gt_cpu(&plane_pool->current_frame, (signed)pic_count - NB_PLANE_BUFFERS + 1);
#elif defined(SRL_POSIX)
    while(*(volatile int*)&plane_pool->current_frame <= (signed)pic_count - NB_PLANE_BUFFERS + 1)
    {
        srl_log_printf(DEBUG, "(SLICE) - current_frame = %d, pic_count = %d\n", plane_pool->current_frame, pic_count);
        sched_yield();
    }
#else
# error FIXME
#endif

    unsigned int plane_cur = pic_count % NB_PLANE_BUFFERS;
    unsigned int plane_ref = (pic_count + NB_PLANE_BUFFERS - 1) % NB_PLANE_BUFFERS;
    frame[0] = &plane_pool->planes[plane_cur];
    frame[1] = &plane_pool->planes[plane_ref];
}

void next_plane_pool(plane_pool_t *plane_pool)
{
    unsigned int plane_ref = (plane_pool->current_frame + NB_PLANE_BUFFERS - 1) % NB_PLANE_BUFFERS;
    memset(plane_pool->planes[plane_ref].m_available, 0, MBLOCKS_Y*MBLOCKS_X);
    
    plane_pool->current_frame = (plane_pool->current_frame + 1) % VIDEO_FRAMES;
}
