/* This file is part of DSX.
 *
 * DSX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DSX is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with DSX; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Copyright (c) Lip6, Thalès
 *      Joel Porquet <joel.porquet@lip6.fr>, 2006-2007
 *
 * Based on Martin Fielder's work (http://keyj.s2000.ws/?page_id=41)
 */

#include <srl.h>
#include "demux_proto.h"

#include "bitstream_io.h"
#include "utils.h"
#include "parameter_set.h"

FUNC(demux)
{
    /* I/O args */
    srl_mwmr_t input_mwmr   = GET_ARG(input);
    srl_mwmr_t output_mwmr[] = {
        GET_ARG(output0),
#if NB_PIPES >= 2
        GET_ARG(output1),
#endif
#if NB_PIPES >= 3
        GET_ARG(output2),
#endif
#if NB_PIPES >= 4
        GET_ARG(output3),
#endif
#if NB_PIPES >= 5
#error "Max value authorized for NB_PIPES :  4"
#endif
    };

    /* Memspace */
    srl_memspace_t ps_mem = GET_ARG(parameter_set);
    ps_t *ps = SRL_MEMSPACE_ADDR(ps_mem);
    
    unsigned char input_buffer[IN_FIFO_WIDTH];
    nal_unit_t nal_unit;
    nal_unit_init_in(&nal_unit, input_buffer, IN_FIFO_WIDTH, input_mwmr);

    slice_t slice;
    slice_init_out(&slice, output_mwmr[0]);

    unsigned char current_pipe = 0;
    
    //srl_mwmr_config(tg_ctrl, 0, 1);

    srl_log(TRACE, "NAL thread is starting...\n");
    
    while(1)
    {
        unsigned int nal_size = 0;
        
        // start marker
        nal_unit_get_next(&nal_unit);
        srl_log(TRACE, "(NAL) - Start marker found\n");
        

        /* 
         * 2. Retrieve infos about this nal unit 
         */
        unsigned char token = nal_unit_read_char(&nal_unit);
        
        unsigned char f_zero_bit     = (token&0x80)>>7;
        unsigned char nal_ref_idc    = (token&0x60)>>5;
        unsigned char nal_unit_type  = (token&0x1F);
        srl_log_printf(DEBUG, "nal_info : forbidden_zero_bit = %d, nal_ref_idc = %d, nal_unit_type = %d\n", 
                              f_zero_bit, nal_ref_idc, nal_unit_type);

        /* (skip the unit if not a coded slice) */
        if (nal_unit_type == 1 || nal_unit_type == 5)
        {
            srl_log_printf(TRACE, "(NAL) - Coded slice (sending to pipeline #%d\n", current_pipe);
            slice_write_char(&slice, nal_ref_idc);
            slice_write_char(&slice, nal_unit_type);
            nal_size += 2;
        } 
        else if (nal_unit_type == 7)
        {
            srl_log(TRACE, "(NAL) - Coded slice (sequence parameter set) \n");
            sequence_parameter_set(&nal_unit, &ps->sps);
        } 
        else if (nal_unit_type == 8)
        {
            srl_log(TRACE, "(NAL) - Coded slice (picture parameter set) \n");
            picture_parameter_set(&nal_unit, &ps->pps);
        } 
        else
            srl_log_printf(TRACE, "(NAL) - Non-coded slice (type=%d): skipping...\n", nal_unit_type);
        
        /* 
         * 3. Scan the bistream until the end marker (0x0000001 or 0x000001) 
         *    (skip the unit if not a coded slice)
         */
        while(!nal_unit_isend(&nal_unit))
        {
            token = nal_unit_read_char(&nal_unit);
            if (nal_unit_type == 1 || nal_unit_type == 5)
            {
                slice_write_char(&slice, token);
                nal_size++;
            }
        }
        
        if (nal_unit_type == 1 || nal_unit_type == 5)
        {
            slice_end(&slice);
            current_pipe = (current_pipe + 1)%NB_PIPES;
            slice.io = output_mwmr[current_pipe];
        }

        srl_log_printf(TRACE, "(NAL) - End marker found (nal size = %d)\n", nal_size);
    }
}

