/* This file is part of DSX.
 *
 * DSX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DSX is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with DSX; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Copyright (c) Lip6, Thalès
 *      Joel Porquet <joel.porquet@lip6.fr>, 2006-2007
 *
 * Based on Martin Fielder's work (http://keyj.s2000.ws/?page_id=41)
 */

#include "modepredinfo.h"

/* clear only one of the two lines of mpi according x and y */
void clear_half_mode_pred_info(mode_pred_inf_t *mpi, int line)
{
    char *MbMode, *TotalCoeffC[2], *TotalCoeffL, *Intra4x4PredMode;
    short int *MVx, *MVy;

    if (line == 0)
    {
        MbMode           = mpi->MbMode;
        TotalCoeffC[0]   = mpi->TotalCoeffC[0];
        TotalCoeffC[1]   = mpi->TotalCoeffC[1];
        TotalCoeffL      = mpi->TotalCoeffL;
        Intra4x4PredMode = mpi->Intra4x4PredMode;
        MVx              = mpi->MVx;
        MVy              = mpi->MVy;
    } else {
        MbMode           = &(mpi->MbMode[MB_1LINE*MBLOCKS_X]);
        TotalCoeffC[0]   = &(mpi->TotalCoeffC[0][CHROMA_1LINE*CBLOCKS_X]);
        TotalCoeffC[1]   = &(mpi->TotalCoeffC[1][CHROMA_1LINE*CBLOCKS_X]);
        TotalCoeffL      = &(mpi->TotalCoeffL[TB_1LINE*TBLOCKS_X]);
        Intra4x4PredMode = &(mpi->Intra4x4PredMode[TB_1LINE*TBLOCKS_X]);
        MVx              = &(mpi->MVx[TB_1LINE*TBLOCKS_X]);
        MVy              = &(mpi->MVy[TB_1LINE*TBLOCKS_X]);
    }

    memset(MbMode,             NA,         MB_1LINE*MBLOCKS_X*sizeof(char));
    memset(TotalCoeffC[0],     0,          CHROMA_1LINE*CBLOCKS_X*sizeof(unsigned char));
    memset(TotalCoeffC[1],     0,          CHROMA_1LINE*CBLOCKS_X*sizeof(unsigned char));
    memset(TotalCoeffL,        0,          TB_1LINE*TBLOCKS_X*sizeof(unsigned char));
    memset(Intra4x4PredMode,   NA,         TB_1LINE*TBLOCKS_X*sizeof(char));
    memset(MVx,                MV_NA&0xFF, TB_1LINE*TBLOCKS_X*sizeof(short int));
    memset(MVy,                MV_NA&0xFF, TB_1LINE*TBLOCKS_X*sizeof(short int));
}

/* Retrieve the mode of a macroblock (coords are in 16x16 world) */
int get_mb_mode(mode_pred_inf_t *mpi, int mb_x, int mb_y)
{
    if (mb_x<0 || mb_y<0) return -1;
    return ModePredInfo_MbMode(mpi, mb_x, mb_y);
}

/* Retrieve nC for luma/chroma (coords are in real world) */
static inline int get_luma_nN(mode_pred_inf_t *mpi, int x, int y)
{
    if (x<0 || y<0) return -1;

    //joel
    if (get_mb_mode(mpi, x>>4, y>>4) == NA) return -1;

    return ModePredInfo_TotalCoeffL(mpi, x>>2, y>>2);
}
static inline int get_chroma_nN(mode_pred_inf_t *mpi, int x, int y, int iCbCr)
{
    if (x<0 || y<0) return -1;
    
    //joel
    if (get_mb_mode(mpi, x>>4, y>>4) == NA) return -1;
    
    return ModePredInfo_TotalCoeffC(mpi, x>>3, y>>3, iCbCr);
}

int get_luma_nC(mode_pred_inf_t *mpi, int x, int y)
{
    int nA = get_luma_nN(mpi, x-4, y);
    int nB = get_luma_nN(mpi, x, y-4);
    if (nA<0  && nB<0)  return 0;
    if (nA>=0 && nB>=0) return (nA+nB+1)>>1;
    if (nA>=0) return nA;
    else return nB;
}

int get_chroma_nC(mode_pred_inf_t *mpi, int x, int y, int iCbCr)
{
    int nA = get_chroma_nN(mpi, x-8, y, iCbCr);
    int nB = get_chroma_nN(mpi, x, y-8, iCbCr);
    if (nA<0  && nB<0)  return 0;
    if (nA>=0 && nB>=0) return (nA+nB+1)>>1;
    if (nA>=0) return nA;
    else return nB;
}

/* Compute the spatial prediction for a 4x4 macroblock (coord are in real world) */
int get_Intra4x4PredModeN(mode_pred_inf_t *mpi, int x, int y)
{
    int i;
    if (x<0 || y<0) return -1;
    i = ModePredInfo_Intra4x4PredMode(mpi, x>>2, y>>2);
    return i;
}

int get_predIntra4x4PredMode(mode_pred_inf_t *mpi, int x, int y)
{
    int A = get_Intra4x4PredModeN(mpi, x-4, y);
    int B = get_Intra4x4PredModeN(mpi, x, y-4);
    int mode = (A<B)?A:B;
    if (mode<0) mode = 2; // if nothing available, force DC
    return mode;
}

/* Compute the temporal prediction */
typedef struct {
    short int x,y;
    int available;  // i.e. inside the image
    int valid;      // i.e. usable for prediction
} mv_t;

/* Retrieve the MV of a block (coord are in real world) */
static void get_MV(mode_pred_inf_t *mpi, mv_t *res, int x, int y) 
{
    if (x<0 || y<0 || x>=WIDTH || y>=HEIGHT)
        return;
    x>>=2; y>>=2;
    res->x = ModePredInfo_MVx(mpi, x, y);
    res->y = ModePredInfo_MVy(mpi, x, y);
    if (res->x == MV_NA) 
    {
        res->x = 0;
        res->y = 0;
        if (IsIntra(ModePredInfo_MbMode(mpi, x>>2, y>>2)))
            res->available = 1;
    } else {
        res->available = 1;
        res->valid = 1;
    }
}

#define return_mv(src) \
do {                                    \
    memcpy(res, (src), sizeof(mv_t));   \
    return;                             \
} while(0)

#define Max(a,b)        ((a)>(b)?(a):(b))
#define Min(a,b)        ((a)<(b)?(a):(b))
#define Median(a,b,c)   Max(Min(a,b),Min(c,Max(a,b)))

/* Get the temporal prediction of a width*height block (coord are in real world) */
static void PredictMV(mode_pred_inf_t *mpi, mv_t *res, int org_x, int org_y, int width, int height) 
{
    mv_t A = {0, 0, 0, 0},
         B = {0, 0, 0, 0},
         C = {0, 0, 0, 0};

    get_MV(mpi, &A, org_x-1, org_y);
    get_MV(mpi, &B, org_x, org_y-1);
    get_MV(mpi, &C, org_x+width, org_y-1);
    
    if (!C.available)
        get_MV(mpi, &C, org_x-1, org_y-1);

    // Directional segmentation prediction for 8x16 / 16x8 partitions
    if (width == 16 && height == 8) 
    {
        if (org_y&8) { if (A.valid) return_mv(&A); }
        else { if (B.valid) return_mv(&B); }
    }
    if (width == 8 && height == 16) 
    {
        if (org_x&8) { if (C.valid) return_mv(&C); }
        else { if (A.valid) return_mv(&A); }
    }
    // If one and only one of the candidate predictors is available and valid,
    // it is returned
    if (!B.valid && !C.valid) 
        return_mv(&A);
    if (!A.valid &&  B.valid && !C.valid) 
        return_mv(&B);
    if (!A.valid && !B.valid &&  C.valid) 
        return_mv(&C);
    // median prediction
    res->x = Median(A.x, B.x, C.x);
    res->y = Median(A.y, B.y, C.y);
}

/* Get the temporal prediction of a skip block (coord are in real world) */
static void Predict_P_Skip_MV(mode_pred_inf_t *mpi, mv_t *v, int org_x, int org_y)
{
    if (org_x<=0 || org_y<=0) return;
    
    if (ModePredInfo_MVx(mpi, (org_x>>2)-1, org_y>>2) == 0 &&
        ModePredInfo_MVy(mpi, (org_x>>2)-1, org_y>>2) == 0) return;
    if (ModePredInfo_MVx(mpi, org_x>>2, (org_y>>2)-1) == 0 &&
        ModePredInfo_MVy(mpi, org_x>>2, (org_y>>2)-1) == 0) return;
        
    if (ModePredInfo_MVx(mpi, (org_x>>2)-1, org_y>>2) == MV_NA &&
        ModePredInfo_MVy(mpi, (org_x>>2)-1, org_y>>2) == MV_NA) return;
    if (ModePredInfo_MVx(mpi, org_x>>2, (org_y>>2)-1) == MV_NA &&
        ModePredInfo_MVy(mpi, org_x>>2, (org_y>>2)-1) == MV_NA) return;
        
    return PredictMV(mpi, v, org_x, org_y, 16, 16);
}

/* Fill all the 4x4 inter block included by width * height with the mv */
static void FillMVs(mode_pred_inf_t *mpi, int org_x, int org_y, int width, int height, int mvx, int mvy)
{
    int x,y;
    org_x >>= 2; 
    org_y >>= 2;
    width >>= 2; 
    height >>= 2;
    for (y=org_y+height-1; y>=org_y; --y)
    {
        for (x=org_x+width-1; x>=org_x; --x) 
        {
            ModePredInfo_MVx(mpi, x, y) = mvx;
            ModePredInfo_MVy(mpi, x, y) = mvy;
        }
    }
}

/* Derive the mvs for a width * height inter block */
void DeriveMVs(mode_pred_inf_t *mpi, int org_x, int org_y, int width, int height, int mvdx, int mvdy) 
{
    mv_t v = {0, 0, 0, 0};
    PredictMV(mpi, &v, org_x, org_y, width, height);
    FillMVs(mpi, org_x, org_y, width, height, v.x+mvdx, v.y+mvdy);
}

/* Derive the mv for a skip inter block */
void Derive_P_Skip_MVs(mode_pred_inf_t *mpi, int org_x, int org_y)
{
    mv_t v = {0, 0, 0, 0};
    Predict_P_Skip_MV(mpi, &v, org_x, org_y);
    FillMVs(mpi, org_x, org_y, MBLOCK_WIDTH, MBLOCK_HEIGHT, v.x, v.y);
}
