/* This file is part of DSX.
 *
 * DSX is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * DSX is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with DSX; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 *
 * Copyright (c) Lip6, Thalès
 *      Joel Porquet <joel.porquet@lip6.fr>, 2006-2007
 *
 * Based on Martin Fielder's work (http://keyj.s2000.ws/?page_id=41)
 */

#include <srl.h>

#include "parameter_set.h"
#include "bitstream_io.h"
#include "h264.h"

#define CHECK_PARAMETER(assertion) \
    do { \
        if (!(assertion)) {\
            srl_log_printf(NONE, "Error : "#assertion" is false\n"); \
            srl_assert(0); \
        } \
    } while(0)
    
void sequence_parameter_set(nal_unit_t *nalu, sps_t *sps)
{
    bitreader_t bitreader;

    bitreader_init(&bitreader, nalu, &nal_unit_read_char);

    /* profile_idc */
    sps->profile_idc = bitreader_get(&bitreader, 8);
    CHECK_PARAMETER(sps->profile_idc == 66);

    /* constraint_set_flag */
    bitreader_get_one(&bitreader);
    bitreader_get_one(&bitreader);
    bitreader_get_one(&bitreader);
    bitreader_get_one(&bitreader);
    
    /* reserved_zero_4bits */
    bitreader_get(&bitreader, 4);

    /* level_idc */
    sps->level_idc = bitreader_get(&bitreader, 8);
    CHECK_PARAMETER(sps->level_idc <= 12);

    /* seq_parameter_set_id */
    sps->seq_parameter_set_id = bitreader_get_ugolomb(&bitreader);
    CHECK_PARAMETER(sps->seq_parameter_set_id == 0);

    /* log2_max_frame_num */
    sps->log2_max_frame_num = bitreader_get_ugolomb(&bitreader)+4;
    CHECK_PARAMETER(sps->log2_max_frame_num <= 12);

    /* pic_order_cnt_type */
    sps->pic_order_cnt_type = bitreader_get_ugolomb(&bitreader);
    CHECK_PARAMETER(sps->pic_order_cnt_type == 0);
    /* pic_order_cnt_type is always 0 */
    sps->log2_max_pic_order_cnt_lsb = bitreader_get_ugolomb(&bitreader)+4;

    sps->num_ref_frames = bitreader_get_ugolomb(&bitreader);
    CHECK_PARAMETER(sps->num_ref_frames == 1);

    /* gaps_in_frame_num_value_allowed_flag */
    bitreader_get_one(&bitreader);

    /* pic_width_in_mbs */
    sps->PicWidthInMbs = bitreader_get_ugolomb(&bitreader)+1;
    CHECK_PARAMETER(sps->PicWidthInMbs == MBLOCKS_X);
    
    /* PicHeightInMbs (equal to PicHeightInMapUnits cause frame_mbs_only_flag == 1) */
    sps->PicHeightInMbs = bitreader_get_ugolomb(&bitreader)+1;
    CHECK_PARAMETER(sps->PicHeightInMbs == MBLOCKS_Y);

    /* frame_mbs_only_flag */
    sps->frame_mbs_only_flag = bitreader_get_one(&bitreader);
    CHECK_PARAMETER(sps->frame_mbs_only_flag == 1);
    
    /* direct_8x8_inference_flag */
    bitreader_get_one(&bitreader);

    /* frame_cropping_flag */
    sps->frame_cropping_flag = bitreader_get_one(&bitreader);
    CHECK_PARAMETER(sps->frame_cropping_flag == 0);

    /* vui_parameters_present_flag */
    sps->vui_parameters_present_flag = bitreader_get_one(&bitreader);
    CHECK_PARAMETER(sps->vui_parameters_present_flag == 0);

    /* display parameters */
    srl_log_printf(DEBUG, "Sequence Parameter Set :\n");
    srl_log_printf(DEBUG, "\tprofile_idc                = %d\n", sps->profile_idc);
    srl_log_printf(DEBUG, "\tlevel_idc                  = %d\n", sps->level_idc);
    srl_log_printf(DEBUG, "\tseq_parameter_set_id       = %d\n", sps->seq_parameter_set_id);
    srl_log_printf(DEBUG, "\tlog2_max_frame_num         = %d\n", sps->log2_max_frame_num);
    srl_log_printf(DEBUG, "\tpic_order_cnt_type         = %d\n", sps->pic_order_cnt_type);
    srl_log_printf(DEBUG, "\tlog2_max_pic_order_cnt_lsb = %d\n", sps->log2_max_pic_order_cnt_lsb);
    srl_log_printf(DEBUG, "\tnum_ref_frames             = %d\n", sps->num_ref_frames);
    srl_log_printf(DEBUG, "\tPicWidthInMbs              = %d\n", sps->PicWidthInMbs);
    srl_log_printf(DEBUG, "\tPicHeightInMbs             = %d\n", sps->PicHeightInMbs);
    srl_log_printf(DEBUG, "\tframe_mbs_only_flag        = %d\n", sps->frame_mbs_only_flag);
}

void picture_parameter_set(nal_unit_t *nalu, pps_t *pps)
{
    bitreader_t bitreader;

    bitreader_init(&bitreader, nalu, &nal_unit_read_char);

    /* pic_parameter_set_id */
    pps->pic_parameter_set_id = bitreader_get_ugolomb(&bitreader);
    CHECK_PARAMETER(pps->pic_parameter_set_id == 0);

    /* seq_parameter_set_id */
    pps->seq_parameter_set_id = bitreader_get_ugolomb(&bitreader);
    CHECK_PARAMETER(pps->seq_parameter_set_id == 0);
    
    /* entropy_coding_mode_flag */
    pps->entropy_coding_mode_flag = bitreader_get_one(&bitreader);
    CHECK_PARAMETER(pps->entropy_coding_mode_flag == 0);

    /* pic_order_present_flag */
    pps->pic_order_present_flag = bitreader_get_one(&bitreader);
    CHECK_PARAMETER(pps->pic_order_present_flag == 0);

    /* num_slice_groups */
    pps->num_slice_groups = bitreader_get_ugolomb(&bitreader)+1;
    CHECK_PARAMETER(pps->num_slice_groups == 1);

    /* num_ref_idx_l*_active */
    pps->num_ref_idx_l0_active = bitreader_get_ugolomb(&bitreader)+1;
    pps->num_ref_idx_l1_active = bitreader_get_ugolomb(&bitreader)+1;
    CHECK_PARAMETER(pps->num_ref_idx_l0_active == 1 && pps->num_ref_idx_l1_active == 1);

    /* weighted_ */
    pps->weighted_pred_flag = bitreader_get_one(&bitreader);
    pps->weighted_bipred_idc = bitreader_get(&bitreader, 2);
    CHECK_PARAMETER(pps->weighted_pred_flag == 0 && pps->weighted_bipred_idc == 0);
    
    /* pic_init_q* */
    pps->pic_init_qp = bitreader_get_sgolomb(&bitreader)+26;
    pps->pic_init_qs = bitreader_get_sgolomb(&bitreader)+26;

    /* chroma_qp_index_offset */
    pps->chroma_qp_index_offset = bitreader_get_sgolomb(&bitreader);

    /* deblocking_filter_control_present_flag */
    pps->deblocking_filter_control_present_flag = bitreader_get_one(&bitreader);

    /* constrained_intra_pred_flag */
    pps->constrained_intra_pred_flag = bitreader_get_one(&bitreader);

    /* redundant_pic_cnt_present_flag */
    pps->redundant_pic_cnt_present_flag = bitreader_get_one(&bitreader);
    CHECK_PARAMETER(pps->redundant_pic_cnt_present_flag == 0);


    srl_log_printf(DEBUG, "Picture Parameter Set :\n");
    srl_log_printf(DEBUG, "\tpic_parameter_set_id                   = %d\n", pps->pic_parameter_set_id);
    srl_log_printf(DEBUG, "\tseq_parameter_set_id                   = %d\n", pps->seq_parameter_set_id);
    srl_log_printf(DEBUG, "\tentropy_coding_mode_flag               = %d\n", pps->entropy_coding_mode_flag);
    srl_log_printf(DEBUG, "\tpic_order_present_flag                 = %d\n", pps->pic_order_present_flag);
    srl_log_printf(DEBUG, "\tnum_slice_groups                       = %d\n", pps->num_slice_groups);
    srl_log_printf(DEBUG, "\tnum_ref_idx_l0_active                  = %d\n", pps->num_ref_idx_l0_active);
    srl_log_printf(DEBUG, "\tnum_ref_idx_l1_active                  = %d\n", pps->num_ref_idx_l1_active);
    srl_log_printf(DEBUG, "\tweighted_pred_flag                     = %d\n", pps->weighted_pred_flag);
    srl_log_printf(DEBUG, "\tweighted_bipred_idc                    = %d\n", pps->weighted_bipred_idc);
    srl_log_printf(DEBUG, "\tpic_init_qp                            = %d\n", pps->pic_init_qp);
    srl_log_printf(DEBUG, "\tpic_init_qs                            = %d\n", pps->pic_init_qs);
    srl_log_printf(DEBUG, "\tchroma_qp_index_offset                 = %d\n", pps->chroma_qp_index_offset);
    srl_log_printf(DEBUG, "\tdeblocking_filter_control_present_flag = %d\n", pps->deblocking_filter_control_present_flag);
    srl_log_printf(DEBUG, "\tconstrained_intra_pred_flag            = %d\n", pps->constrained_intra_pred_flag);
    srl_log_printf(DEBUG, "\tredundant_pic_cnt_present_flag         = %d\n", pps->redundant_pic_cnt_present_flag);
}
