/*
 * dev_pic.c - PIC (External Interrupt Controler) generic device API implementation.
 *
 * Authors   Alain Greiner  (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH.is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH.is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTPICLAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH.; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_special.h>
#include <dev_icu.h>
#include <chdev.h>
#include <memcpy.h>
#include <printk.h>
#include <soclib_pic.h>
#include <dev_pic.h>

/////////////////////////////////////////////////////////////////////////////////////////
// Extern global variables
/////////////////////////////////////////////////////////////////////////////////////////

extern chdev_directory_t  chdev_dir;         // allocated in kernel_init.c

extern chdev_pic_input_t  chdev_pic_input;   // allocated in kernel_init.c

///////////////////////////////////
void dev_pic_init( chdev_t * chdev,
                   uint32_t  irq_nr )
{
    // set PIC chdev extension field
    chdev->ext.pic.irq_nr = irq_nr;

    // get implementation 
    uint32_t impl = chdev->impl;

    // set chdev name
    strcpy( chdev->name , "pic" );
  
    // call the relevant driver init function
    if( impl == IMPL_PIC_SOC )
    {
        soclib_pic_init( chdev );
    }
    else
    {
        assert( false , __FUNCTION__ , "illegal PIC device implementation" );
    }
} // end dev_pic_init()

/////////////////////////////////////////
void dev_pic_bind_irq( uint32_t   irq_id,
                       cxy_t      cxy,
                       uint32_t   wti_id )
{
    // get extended pointer on WTI mailbox
    xptr_t wti_xp = XPTR( cxy , dev_icu_wti_ptr( wti_id ) );

    // get extended pointer on PIC chdev from directory
    xptr_t dev_xp = chdev_dir.pic;

    // get PIC chdev cluster and local pointer
    cxy_t     dev_cxy = GET_CXY( dev_xp );
    chdev_t * dev_ptr = (chdev_t *)GET_PTR( dev_xp );

    // get implementation index and segment base 
    uint32_t impl   = hal_remote_lw( XPTR( dev_cxy , &dev_ptr->impl ) );

    // call the implementation specific driver function
    if( impl == IMPL_PIC_SOC )
    {
        soclib_pic_bind_irq( dev_xp , irq_id , wti_xp );
    }
}  // end dev_pic_link_wti()

//////////////////////////////////////////
void dev_pic_unbind_irq( uint32_t irq_id )
{
    // get extended pointer on PIC chdev from directory
    xptr_t dev_xp = chdev_dir.pic;

    // get PIC chdev cluster and local pointer
    cxy_t     dev_cxy = GET_CXY( dev_xp );
    chdev_t * dev_ptr = (chdev_t *)GET_PTR( dev_xp );

    // get implementation index 
    uint32_t impl = hal_remote_lw( XPTR( dev_cxy , &dev_ptr->impl ) );

    // call the implementation specific driver function
    if( impl == IMPL_PIC_SOC )
    {
        soclib_pic_unbind_irq( dev_xp , irq_id );
    }
} // end dev_pic_disable_irq()


