/*
 * fatfs.h - FATFS file system API definition.
 *
 * Author    Mohamed Lamine Karaoui (2014,2015)
 *           Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _FATFS_H_
#define _FATFS_H_

#include <hal_types.h>
#include <rwlock.h>
#include <vfs.h>


///////////////////////////////////////////////////////////////////////////////////////////
// The FATFS File System implements a FAT32 read/write file system.
///////////////////////////////////////////////////////////////////////////////////////////

/*************** Partition Boot Sector Format **********************************/
//                                     offset |  length
#define BS_JMPBOOT                          0 ,  3
#define BS_OEMNAME                          3 ,  8
#define BPB_BYTSPERSEC                     11 ,  2
#define BPB_SECPERCLUS                     13 ,  1
#define BPB_RSVDSECCNT                     14 ,  2
#define BPB_NUMFATS                        16 ,  1
#define BPB_ROOTENTCNT                     17 ,  2
#define BPB_TOTSEC16                       19 ,  2
#define BPB_MEDIA                          21 ,  1
#define BPB_FATSZ16                        22 ,  2
#define BPB_SECPERTRK                      24 ,  2
#define BPB_NUMHEADS                       26 ,  2
#define BPB_HIDDSEC                        28 ,  4
#define BPB_TOTSEC32                       32 ,  4
#define BPB_PARTITION_TABLE               446 , 64 

// FAT 32
#define BPB_FAT32_FATSZ32                  36 ,  4
#define BPB_FAT32_EXTFLAGS                 40 ,  2
#define BPB_FAT32_FSVER                    42 ,  2
#define BPB_FAT32_ROOTCLUS                 44 ,  4
#define BPB_FAT32_FSINFO                   48 ,  2
#define BPB_FAT32_BKBOOTSEC                50 ,  2
#define BS_FAT32_DRVNUM                    64 ,  1
#define BS_FAT32_BOOTSIG                   66 ,  1
#define BS_FAT32_VOLID                     67 ,  4
#define BS_FAT32_VOLLAB                    71 , 11
#define BS_FAT32_FILSYSTYPE                82 ,  8

// Partitions
#define FIRST_PARTITION_ACTIVE            446 ,  8
#define FIRST_PARTITION_BEGIN_LBA         454 ,  4
#define FIRST_PARTITION_SIZE              458 ,  4 
#define SECOND_PARTITION_ACTIVE           462 ,  8
#define SECOND_PARTITION_BEGIN_LBA        470 ,  4
#define SECOND_PARTITION_SIZE             474 ,  4
#define THIRD_PARTITION_ACTIVE            478 ,  8
#define THIRD_PARTITION_BEGIN_LBA         486 ,  4
#define THIRD_PARTITION_SIZE              490 ,  4
#define FOURTH_PARTITION_ACTIVE           494 ,  8
#define FOURTH_PARTITION_BEGIN_LBA        502 ,  4
#define FOURTH_PARTITION_SIZE             506 ,  4    
/*******************************************************************************/

#define MBR_SIGNATURE_POSITION            510 , 2
#define MBR_SIGNATURE_VALUE               0xAA55  

/************** FAT_FS_INFO SECTOR  ********************************************/
#define FS_SIGNATURE_VALUE_1              0x52526141
#define FS_SIGNATURE_VALUE_2              0x72724161
#define FS_SIGNATURE_VALUE_3              0x000055AA  
#define FS_SIGNATURE_POSITION_1           0   , 4  
#define FS_SIGNATURE_POSITION_2           484 , 4
#define FS_SIGNATURE_POSITION_3           508 , 4  
#define FS_FREE_CLUSTERS                  488 , 4
#define FS_FREE_CLUSTER_HINT              492 , 4
/*******************************************************************************/

#define DIR_ENTRY_SIZE          32
                   
#define NAME_MAX_SIZE           31

/******* Directory Entry Structure (32 bytes) **********************************/
//                            offset | length
#define DIR_NAME                   0 , 11   // dir_entry name
#define DIR_ATTR                  11 ,  1   // attributes
#define DIR_NTRES                 12 ,  1   // reserved for the OS        
#define DIR_CRT_TIMES_TENTH       13 ,  1 
#define DIR_FST_CLUS_HI           20 ,  2   // cluster index 16 MSB bits
#define DIR_WRT_TIME              22 ,  2   // time of last write
#define DIR_WRT_DATE              24 ,  2   // date of last write
#define DIR_FST_CLUS_LO           26 ,  2   // cluster index 16 LSB bit
#define DIR_FILE_SIZE             28 ,  4   // dir_entry size (up to 4 Gbytes)
/*******************************************************************************/

/******* LFN Directory Entry Structure  (32 bytes) *****************************/
//                            offset | length
#define LDIR_ORD                   0 ,  1   // Sequence number (from 0x01 to 0x0f)    
#define LDIR_NAME_1                1 , 10   // name broken into 3 parts 
#define LDIR_ATTR                 11 ,  1   // attributes (must be 0x0F) 
#define LDIR_TYPE                 12 ,  1   // directory type (must be 0x00)
#define LDIR_CHKSUM               13 ,  1   // checksum of name in short dir  
#define LDIR_NAME_2               14 , 12 
#define LDIR_RSVD                 26 ,  2   // artifact of previous fat (must be 0)
#define LDIR_NAME_3               28 ,  4   
/*******************************************************************************/

/***********************  DIR_ATTR values  (attributes) ************************/
#define ATTR_READ_ONLY            0x01
#define ATTR_HIDDEN               0x02
#define ATTR_SYSTEM               0x04
#define ATTR_VOLUME_ID            0x08
#define ATTR_DIRECTORY            0x10
#define ATTR_ARCHIVE              0x20
#define ATTR_LONG_NAME_MASK       0x0f      // READ_ONLY|HIDDEN|SYSTEM|VOLUME_ID
/*******************************************************************************/

/********************* DIR_ORD special values **********************************/
#define FREE_ENTRY                0xE5     // this entry is free in the directory
#define NO_MORE_ENTRY             0x00     // no more entry in the directory
/*******************************************************************************/

/******************** CLuster Index Special Values *****************************/
#define FREE_CLUSTER              0x00000000
#define RESERVED_CLUSTER          0x00000001
#define BAD_CLUSTER               0x0FFFFFF7
#define END_OF_CHAIN_CLUSTER_MIN  0x0ffffff8
#define END_OF_CHAIN_CLUSTER_MAX  0x0fffffff
/*******************************************************************************/

/****  Forward declarations  ****/

struct mapper_s;
struct page_s;
struct vfs_ctx_s;
struct vfs_inode_s;

/*****************************************************************************************
 * This structure defines a FATFS specific context (extension to VFS context). 
 ****************************************************************************************/

typedef struct fatfs_ctx_s
{
    uint32_t          fat_begin_lba;         /*! lba of FAT region                      */
    uint32_t          fat_sectors_count;     /*! number of sectors in FAT region        */
    uint32_t          bytes_per_sector;      /*!                                        */
    uint32_t          bytes_per_cluster;     /*!                                        */
    uint32_t          cluster_begin_lba;     /*! lba of data region                     */
    uint32_t          sectors_per_cluster;   /*! cluster index for root directory       */
    uint32_t          root_dir_cluster;      /*!                                        */
    uint32_t          last_allocated_sector; /*! TODO ???                               */
    uint32_t          last_allocated_index;  /*! TODO ???                               */
    xptr_t            fat_mapper_xp;         /*! FAT mapper (in IO cluster)             */
}
fatfs_ctx_t;

/*****************************************************************************************
 * This structure defines the FATFS specific inode (extension to VFS inode).
 ****************************************************************************************/

typedef struct fatfs_inode_s
{
	uint32_t          first_cluster;         /*! first cluster for this file/dir        */ 
}
fatfs_inode_t;




//////////////////////////////////////////////////////////////////////////////////////////
// These functions are specific to the FATFS, and cannot be called by the VFS.
//////////////////////////////////////////////////////////////////////////////////////////

/*****************************************************************************************
 * This function returns the LBA of the first sector of a FAT cluster.
 * This function can be called by any thread running in any cluster.
 *****************************************************************************************
 * @ ctx	  :	pointer on FATFS context.
 * @ cluster  :	cluster index in FATFS.
 * @ return the lba value.
 ****************************************************************************************/
inline uint32_t fatfs_lba_from_cluster( fatfs_ctx_t * ctx,
                                        uint32_t      cluster );

/*****************************************************************************************
 * This function scan the FAT (File Allocation Table), stored in the FAT mapper,
 * and return the FATFS cluster index for a given page of a given file.
 * It must be called by a thread running in the cluster containing the FAT mapper
 * (can be a local thread or a RPC thread).
 * The FAT is actually an array of uint32_t slots. Each slot in this array contains the 
 * index of another slot in this array, to form one linked list for each file stored on
 * device in the RAMFS file system. This index in the FAT array is also the index of the
 * FATFS cluster on the device. One FATFS cluster is supposed to contain one PPM page. 
 * For a given file, the entry point in the FAT is simply the index of the FATFS cluster 
 * containing the first page of the file. The mapper being a cache, this function 
 * automatically updates the FAT mapper from informations stored on device in case of miss.
 *****************************************************************************************
 * @ mapper	 : local pointer on the FAT mapper.
 * @ first	 : index of the first FATFS cluster allocated to the file.
 * @ page    : index of searched page in the file.
 * @ cluster : [out] pointer on buffer for the found FATFS cluster index.
 * @ return 0 if success / return EIO if FAT mapper miss cannot be solved.
 ****************************************************************************************/
error_t fatfs_get_cluster( struct mapper_s * mapper,
                           uint32_t          first_cluster,
                           uint32_t          searched_page,
                           uint32_t        * cluster );




//////////////////////////////////////////////////////////////////////////////////////////
// These functions are called by the VFS, and must be implemented by all File Systems.
//////////////////////////////////////////////////////////////////////////////////////////

/******************************************************************************************
 * This function initializes the FATFS file system as the root FS.
 * It is executed cooperatively during kernel init by all CP0s in all clusters.
 * The initilisation is made in three phases, separated by synchronisation barrier:     
 * - phase 1 : all CP0s in all clusters allocate memory for the local copy of
 *   the FATFS context.
 * - phase 2 : cluster_0 only creates the root inode descriptor, access the external
 *   device to get information stored on the boot record, and initialises both 
 *   the VFS context, and the FATFS context.
 * - phase 3 : all other clusters initialize their local VFS context and FATFS context
 *   from values contained in cluster_0, using hal_remote_memcpy().
 ******************************************************************************************
 * @ return an extended pointer on the created root inode / return XPTR_NULL if failure.
 *****************************************************************************************/
xptr_t fatfs_init();

/******************************************************************************************
 * This function mount the FATFS on the root FS.
 * TODO not implemented [AG]
 ******************************************************************************************
 * @ parent_inode_xp : extended pointer on the parent inode.
 *****************************************************************************************/
error_t fatfs_mount( xptr_t parent_inode_xp );


/*****************************************************************************************
 * This function  initializes both the VFS context and the FATFS context.
 * Both the VFS context and the FATFS context must have been previously allocated.
 * It access the device to read the boot record, and is supposed to be called only 
 * in cluster_0 (in other clusters these contexts are replicated from values 
 * contained in cluster_0).
 *****************************************************************************************
 * @ vfs_ctx        : local pointer on VFS context for FATFS.
 * @ fatfs_ctx      : local pointer on specific FATFS context.
 * @ root_inode_xp  : extended pointer on VFS root inode.
 ****************************************************************************************/
error_t fatfs_ctx_init( struct vfs_ctx_s   * vfs_ctx,
                        struct fatfs_ctx_s * fatfs_ctx,
                        xptr_t               root_inode_xp );

/*****************************************************************************************
 * This function releases memory dynamically allocated for the FATFS context extension.
 *****************************************************************************************
 * @ vfs_ctx   : local pointer on VFS context.
 ****************************************************************************************/
void fatfs_ctx_destroy( struct vfs_ctx_s * vfs_ctx );



/*****************************************************************************************
 * This function allocates memory for a FATFS inode, initializes it,
 * and link it to the VFS inode.
 *****************************************************************************************
 * @ inode         : local pointer on the VFS inode.
 * @ first_cluster : first cluster index in the FAT32.
 * @ return 0 if success / return ENOMEM if error. 
 ****************************************************************************************/
error_t fatfs_inode_create( struct vfs_inode_s * inode,
                            uint32_t             first_cluster);

/*****************************************************************************************
 * This function releases memory allocated for a FATFS inode.
 *****************************************************************************************
 * @ inode   : local pointer on vfs_inode.
 ****************************************************************************************/
void fatfs_inode_destroy( struct vfs_inode_s * inode );



/*****************************************************************************************
 * This function moves a page from the mapper to the FATFS file system on device.
 * It must be called by a thread running in cluster containing the mapper.
 * The pointer on the mapper and the page index in file are registered
 * in the page descriptor. 
 *****************************************************************************************
 * @ page    : local pointer on page descriptor.
 * @ return 0 if success / return EIO if error. 
 ****************************************************************************************/
error_t fatfs_write_page( struct page_s * page );

/*****************************************************************************************
 * This function moves a page from the FATFS file system on device to the mapper.
 * It must be called by a thread running in cluster containing the mapper.
 * The pointer on the mapper and the page index in file are registered
 * in the page descriptor. 
 *****************************************************************************************
 * @ page    : local pointer on page descriptor.
 * @ return 0 if success / return EIO if error. 
 ****************************************************************************************/
error_t fatfs_read_page( struct page_s * page );




#endif	/* _FATFS_H_ */
