/*
 * dev_nic.c - NIC (Network Controler) generic device API implementation.
 * 
 * Author  Alain Greiner    (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_special.h>
#include <printk.h>
#include <chdev.h>
#include <thread.h>
#include <soclib_nic.h>
#include <dev_nic.h>

/////////////////////////////////////////////////////////////////////////////////////////
// Extern global variables
/////////////////////////////////////////////////////////////////////////////////////////

extern chdev_directory_t  chdev_dir;         // allocated in kernel_init.c

extern chdev_pic_input_t  chdev_pic_input;   // allocated in kernel_init.c

////////////////////////////////////
void dev_nic_init( chdev_t * chdev )
{
    // the local ICU chdev must be initialized before the NIC chdev, because
    // the NIC chdevs initialisation requires allocation of a WTI from local ICU
    xptr_t  icu_xp  = chdev_dir.icu[local_cxy];
    assert( (icu_xp != XPTR_NULL) , __FUNCTION__ , "ICU not initialised before NIC" );

    // get "impl" , "channel" , "is_rx" fields from chdev descriptor
    uint32_t  impl    = chdev->impl;
    uint32_t  is_rx   = chdev->is_rx;
    uint32_t  channel = chdev->channel;

    // set chdev name
    snprintf( chdev->name , 16 , "nic_%d" , chdev->channel );

    // set driver specific fields in chdev descriptor and call driver init function
    if( impl == IMPL_NIC_SOC )
    {
        chdev->cmd = &soclib_nic_cmd;
        chdev->isr = &soclib_nic_isr;
        soclib_nic_init( chdev );
    }
    else
    {
        assert( false , __FUNCTION__ , "undefined NIC device implementation" );
    }

    // get a WTI mailbox from local ICU
    uint32_t wti_id = dev_icu_wti_alloc();

    assert( (wti_id != -1) , __FUNCTION__ , "cannot allocate WTI mailbox" );

    // select a core
    lid_t lid = cluster_select_local_core();

    // enable WTI IRQ and update WTI interrupt vector
    dev_icu_enable_irq( lid , WTI_TYPE , wti_id , chdev );

    // link NIC IRQ to WTI mailbox in PIC component 
    uint32_t irq_id; 
    if( is_rx ) irq_id = chdev_pic_input.nic_rx[channel];
    else        irq_id = chdev_pic_input.nic_tx[channel];
    dev_pic_bind_irq( irq_id , local_cxy , wti_id );

    // create server thread
    thread_t * new_thread;
    error_t    error;

    error = thread_kernel_create( &new_thread,
                                  THREAD_DEV,
                                  &chdev_sequencial_server,
                                  chdev,
                                  lid ); 

    assert( (error == 0) , __FUNCTION__ , "cannot create server thread" );

    // set "server" field in chdev descriptor
    chdev->server = new_thread;
    
    // start server thread
    thread_unblock( XPTR( local_cxy , new_thread ) , THREAD_BLOCKED_GLOBAL );

}  // end dev_nic_init()

///////////////////////////////////
error_t dev_nic_read( pkd_t * pkd )
{
    error_t  error;

    // get pointers on this NIC-RX kernel thread 
    thread_t * thread_ptr = CURRENT_THREAD;
    xptr_t     thread_xp  = XPTR( local_cxy , thread_ptr );

    // get local pointer on core running this kernel thead
    core_t * core = thread_ptr->core;

    nic_dmsg("\n[INFO] %s enters for NIC-RX thread on core %d in cluster %x\n", 
                 __FUNCTION__ , core->lid , local_cxy );

    // get pointer on NIC-RX chdev descriptor
    uint32_t   channel = thread_ptr->dev_channel; 
    xptr_t     dev_xp  = chdev_dir.nic_rx[channel];
    cxy_t      dev_cxy = GET_CXY( dev_xp );
    chdev_t  * dev_ptr = (chdev_t *)GET_PTR( dev_xp );

    assert( (dev_xp != XPTR_NULL) , __FUNCTION__ , "undefined NIC chdev descriptor" );

    assert( (dev_cxy == local_cxy) , __FUNCTION__ , " chdev must be local" );

    // initialize command in thread descriptor
    thread_ptr->command.nic.dev_xp = dev_xp;

    // call driver to test readable
    thread_ptr->command.nic.cmd = NIC_CMD_READABLE;
    dev_ptr->cmd( thread_xp );

    // check error
    error = thread_ptr->command.nic.error;
    if( error ) return error;

    // block and deschedule if queue non readable
    if( thread_ptr->command.nic.status == false )  
    {
        // get NIC-RX IRQ index and type
        uint32_t   irq_type = dev_ptr->irq_type;
        uint32_t   irq_id   = dev_ptr->irq_id;

        // enable NIC-RX IRQ 
        dev_icu_enable_irq( core->lid , irq_type , irq_id , dev_ptr );

        // block on THREAD_BLOCKED I/O condition and deschedule
        thread_block( thread_ptr , THREAD_BLOCKED_IO );
        sched_yield();

        // disable NIC-RX channel IRQ 
        dev_icu_disable_irq( core->lid , irq_type , irq_id );
    }

    // call driver for actual read 
    thread_ptr->command.nic.cmd     = NIC_CMD_READ;
    thread_ptr->command.nic.buffer  = pkd->buffer;
    dev_ptr->cmd( thread_xp );

    // check error
    error = thread_ptr->command.nic.error;
    if( error ) return error;

    // returns packet length   
    pkd->length = thread_ptr->command.nic.length;

    nic_dmsg("\n[INFO] %s exit for NIC-RX thread on core %d in cluster %x\n", 
             __FUNCTION__ , core->lid , local_cxy );

    return 0;

}   // end dev_nic_read()


////////////////////////////////////
error_t dev_nic_write( pkd_t * pkd )
{
    error_t error;

    // get pointers on the NIC-TX kernel tread 
    thread_t * thread_ptr = CURRENT_THREAD;
    xptr_t     thread_xp  = XPTR( local_cxy , thread_ptr );

    // get local pointer on core running this kernel thead
    core_t * core = thread_ptr->core;

    nic_dmsg("\n[INFO] %s enters for NIC-RX thread on core %d in cluster %x\n", 
                 __FUNCTION__ , core->lid , local_cxy );

    // get pointer on NIC-TX chdev descriptor
    uint32_t   channel = thread_ptr->dev_channel; 
    xptr_t     dev_xp  = chdev_dir.nic_tx[channel];
    cxy_t      dev_cxy = GET_CXY( dev_xp );
    chdev_t  * dev_ptr = (chdev_t *)GET_PTR( dev_xp );

    assert ( (dev_xp != XPTR_NULL) , __FUNCTION__ , "undefined NIC chdev descriptor" );

    assert( (dev_cxy == local_cxy) , __FUNCTION__ , " chdev must be local" );

    // initialize command in thread descriptor
    thread_ptr->command.nic.dev_xp = dev_xp;

    // call driver to test writable
    thread_ptr->command.nic.cmd    = NIC_CMD_WRITABLE;
    dev_ptr->cmd( thread_xp );

    // check error
    error = thread_ptr->command.nic.error;
    if( error ) return error;

    // block and deschedule if queue non writable
    if( thread_ptr->command.nic.status == false )  
    {
        // get NIC-TX IRQ index and type
        uint32_t   irq_type = dev_ptr->irq_type;
        uint32_t   irq_id   = dev_ptr->irq_id;

        // enable NIC-TX IRQ 
        dev_icu_enable_irq( core->lid , irq_type , irq_id , dev_ptr );

        // block on THREAD_BLOCKED I/O condition and deschedule
        thread_block( thread_ptr , THREAD_BLOCKED_IO );
        sched_yield();

        // disable NIC-TX IRQ 
        dev_icu_disable_irq( core->lid , irq_type , irq_id );
    }

    // call driver for actual write
    thread_ptr->command.nic.cmd    = NIC_CMD_WRITE;
    thread_ptr->command.nic.buffer = pkd->buffer;
    thread_ptr->command.nic.length = pkd->length;
    dev_ptr->cmd( thread_xp );

    // check error
    error = thread_ptr->command.nic.error;
    if( error ) return error;

    nic_dmsg("\n[INFO] %s exit for NIC-TX thread on core %d in cluster %x\n", 
             __FUNCTION__ , core->lid , local_cxy );

    return 0;
}  // end dev_nic_write()



