/*
 * hal_trap.c - Trap handler
 * 
 * Copyright (c) 2017 Maxime Villard
 * 
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH.; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_kentry.h>
#include <hal_internal.h>

static const char *trap_type[] = {
	[T_PRIVINFLT]= "privileged instruction fault",
	[T_BPTFLT] = "breakpoint trap",
	[T_ARITHTRAP] = "arithmetic trap",
	[T_ASTFLT] = "asynchronous system trap",
	[T_PROTFLT] = "protection fault",
	[T_TRCTRAP] = "trace trap",
	[T_PAGEFLT] = "page fault",
	[T_ALIGNFLT] = "alignment fault",
	[T_DIVIDE] = "integer divide fault",
	[T_NMI] = "non-maskable interrupt",
	[T_OFLOW] = "overflow trap",
	[T_BOUND] = "bounds check fault",
	[T_DNA] = "FPU not available fault",
	[T_DOUBLEFLT] = "double fault",
	[T_FPOPFLT] = "FPU operand fetch fault",
	[T_TSSFLT] = "invalid TSS fault",
	[T_SEGNPFLT] = "segment not present fault",
	[T_STKFLT] = "stack fault",
	[T_MCA] = "machine check fault",
	[T_XMM] = "SSE FP exception",
};
int	trap_types = __arraycount(trap_type);

void x86_printf(char *s, ...);

/*
 * Trap handler.
 */
void hal_trap_entry(struct small_trapframe *tf)
{
	uint64_t trapno = tf->tf_trapno;
	const char *buf;

	if (trapno < trap_types) {
		buf = trap_type[trapno];
	} else {
		buf = "unknown trap";
	}

	x86_printf("\n****** FAULT OCCURRED ******\n");
	x86_printf("%s\n", (char *)buf);
	x86_printf("-> rip = %Z\n", tf->tf_rip);
	x86_printf("-> rsp = %Z\n", tf->tf_rsp);
	x86_printf("-> err = %Z\n", tf->tf_err);
	if (trapno == T_PAGEFLT)
		x86_printf("-> va  = %Z\n", rcr2());
	x86_printf("****** FAULT OCCURRED ******\n\n");

	while (1);
}

/*
 * Timer interrupt
 */
void hal_timer_intr(struct trapframe *tf)
{
	x86_printf("-> got timer: rip=%Z\n", tf->tf_rip);
	return;
}

/* -------------------------------------------------------------------------- */

#define PS2_DATA	0x60

#define PS2_STATUS	0x64
#	define STATUS_OUT_FULL	0x00
#	define STATUS_IN_FULL	0x01
#	define STATUS_SYS_FLAG	0x02

#define PS2_CMD	0x64

/*
 * US scancode table, found on the internet.
 */
unsigned char scancode_US[128] =
{
	 0,     27, '1', '2', '3', '4', '5', '6', '7', '8', '9', '0', '-', '=',
	'\b', '\t', 'q', 'w', 'e', 'r', 't', 'y', 'u', 'i', 'o', 'p', '[', ']',
	'\n',
	 0,	/* 29   - Control */
	'a',   's', 'd', 'f', 'g', 'h', 'j', 'k', 'l', ';', '\'', '`',
	 0,	/* Left shift */
	'\\',  'z', 'x', 'c', 'v', 'b', 'n', 'm', ',', '.', '/',
	 0,	/* Right shift */
	'*',
	 0,	/* Alt */
	' ',	/* Space bar */
	 0,	/* Caps lock */
	 0,	/* 59 - F1 key ... > */
	 0,   0,   0,   0,   0,   0,   0,   0,
	 0,	/* < ... F10 */
	 0,	/* 69 - Num lock*/
	 0,	/* Scroll Lock */
	 0,	/* Home key */
	 0,	/* Up Arrow */
	 0,	/* Page Up */
	'-',
	 0,	/* Left Arrow */
	 0,
	 0,	/* Right Arrow */
	'+',
	 0,	/* 79 - End key*/
	 0,	/* Down Arrow */
	 0,	/* Page Down */
	 0,	/* Insert Key */
	 0,	/* Delete Key */
	 0,   0,   0,
	 0,	/* F11 Key */
	 0,	/* F12 Key */
	 0,	/* All other keys are undefined */
};	

/*
 * Keyboard interrupt (8042 PS/2)
 */
void hal_keyboard_intr(struct trapframe *tf)
{
	uint64_t val;

	do {
		val = in8(PS2_STATUS);
	} while ((val & STATUS_IN_FULL) == 0);

	val = in8(PS2_DATA);

	x86_printf("-> got keyboard: rip=%Z key=%c ", tf->tf_rip, scancode_US[val]);

	if (val & 0x80)
		x86_printf("[released]\n");
	else
		x86_printf("[pressed]\n");

	return;
}

