/*
 * soclib_pic.c - soclib PIC driver definition.
 *
 * Author  Alain Greiner (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-kernel; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _SOCLIB_IOPIC_H_
#define _SOCLIB_IOPIC_H_

#include <hal_types.h>

/******************************************************************************************
 *  PIC register offsets (per input IRQ)
 *****************************************************************************************/

#define IOPIC_ADDRESS           0
#define IOPIC_EXTEND            1
#define IOPIC_STATUS            2
#define IOPIC_MASK              3
#define IOPIC_SPAN              4

/******************************************************************************************
 * This blocking function desactivates all input IRQs in PIC controler.
 * It must be called by a local thread.
 ******************************************************************************************
 * @ chdev    : pointer on PIC chdev descriptor.
 *****************************************************************************************/
void   soclib_pic_init( chdev_t * chdev );

/******************************************************************************************
 * This blocking function returns the status for a given input IRQ in the remote
 * PIC controler. It can be called by any thread running on any cluster.
 ******************************************************************************************
 * @ dev_xp    : extended pointer on the PIC device descriptor.
 * @ irq_id    : input IRQ index.
 * @ status    : pointer on local buffer for returned status.
 *****************************************************************************************/
void   soclib_pic_get_status( xptr_t     dev_xp,
                              uint32_t   irq_id,
                              uint32_t * status );

/******************************************************************************************
 * This blocking function unmask an input IRQ in a remote PIC controler, and bind it
 * with a WTI mailbox, by registering the WTI mailbox extended pointer.
 * It can be called by any thread running on any cluster.
 ******************************************************************************************
 * @ dev_xp    : extended pointer on the PIC device descriptor.
 * @ irq_id    : input IRQ index.
 * @ xp_wti    : extended pointer on the WTI mailbox.
 *****************************************************************************************/
void   soclib_pic_bind_irq( xptr_t    dev_xp,
                            uint32_t  irq_id,
                            xptr_t    wti_xp );

/******************************************************************************************
 * This blocking function mask an input IRQ in a remote PIC controler.
 * It can be called by any thread running on any cluster.
 ******************************************************************************************
 * @ dev_xp    : extended pointer on the PIC device descriptor.
 * @ irq_id    : input IRQ index. 
 *****************************************************************************************/
void   soclib_pic_unbind_irq( xptr_t    dev_xp,
                              uint32_t  irq_id );


#endif	/* _SOCLIB_IOPIC_H_ */
