/*
 * ramfs.h  RAMFS file system API definition.
 *
 * Authors   Mohamed Lamine Karaoui (2015)
 *           Alain Greiner (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _RAMFS_H_
#define _RAMFS_H_

/****  Forward declarations  ****/

struct vfs_inode_s;
struct page_s;

/*****************************************************************************************
 * This structure defines a RAMFS specific context extension. 
 ****************************************************************************************/

typedef struct ramfs_ctx_s
{
    intptr_t          base;                                                                 
    rwlock_t          size;
}
ramfs_ctx_t;

/*****************************************************************************************
 * This structure defines the RAMFS inode specific extension (versus the VFS inode).
 ****************************************************************************************/

typedef struct ramfs_inode_s
{
    struct vfs_inode_s * vfs_inode;   /*! local pointer on VFS inode                    */
}
ramfs_inode_t;




/*****************************************************************************************
 * This function allocates memory for a RAMFS inode, and link it to the VFS inode.
 *****************************************************************************************
 * @ inode   : local pointer on vfs_inode.
 * @ return 0 if success / return ENOMEM if error. 
 ****************************************************************************************/
error_t ramfs_inode_create( struct vfs_inode_s * inode );

/*****************************************************************************************
 * This function releases memory allocated for a RAMFS inode.
 *****************************************************************************************
 * @ inode   : local pointer on vfs_inode.
 ****************************************************************************************/
void ramfs_inode_destroy( struct vfs_inode_s * inode );

/*****************************************************************************************
 * This function moves a page from the mapper to the RAMFS file system.
 * It must be called by a thread running in cluster containing the mapper.
 * The pointer on the mapper and the page index in file are supposed to be registered
 * in the page descriptor. 
******************************************************************************************
 * @ page    : local pointer on source page descriptor.
 * @ return 0 if success / return EIO if error. 
 ****************************************************************************************/
error_t ramfs_write_page( struct page_s  * page );

/*****************************************************************************************
 * This function moves a page from the RAMFS file system to the mapper.
 * It must be called by a thread running in cluster containing the mapper.
 * The pointer on the mapper and the page index in file are supposed to be registered
 * in the page descriptor. 
 *****************************************************************************************
 * @ page    : local pointer on destination page descriptor.
 * @ return 0 if success / return EIO if error. 
 ****************************************************************************************/
error_t ramfs_read_page( struct page_s  * page );


   
#endif	/* _RAMFS_H_ */
