/*
 * vfs.h - Virtual File System definition.
 *
 * Author  Mohamed Lamine Karaoui (2015)
 *         Alain Greiner (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _VFS_H_
#define _VFS_H_

#include <almos_config.h>
#include <hal_types.h>
#include <hal_atomic.h>
#include <remote_rwlock.h>
#include <remote_spinlock.h>
#include <spinlock.h>
#include <list.h>
#include <xlist.h>
#include <slist.h>
#include <bits.h>
#include <xhtab.h>
#include <errno.h>
#include <metafs.h>

#include <fatfs.h>
#include <ramfs.h>

/****  Forward declarations  ***/

struct vfs_inode_s;
struct vfs_dentry_t;
struct vfs_ctx_t;
struct vfs_file_ref_s;
struct vfs_file_s;

struct vfs_inode_op_s;
struct vfs_dentry_op_s;
struct vfs_file_op_s;
struct vfs_ctx_op_s;

struct vfs_lookup_cmd_s;
struct vfs_lookup_rsp_s;

struct mapper_s;
struct process_s;
struct device_s;
struct vseg_s;

/*********************************************************************************************
 * This defines the various Flags arguments for an open() or opendir() system call.
 ********************************************************************************************/

#define VFS_O_APPEND         0x00080000
#define VFS_O_RDONLY         0x00100000
#define VFS_O_WRONLY         0x00200000
#define VFS_O_RDWR           0x00300000
#define VFS_O_CREATE         0x00400000
#define VFS_O_EXCL           0x00800000
#define VFS_O_TRUNC          0x01000000
#define VFS_O_SYNC	         0x08000000

/*********************************************************************************************
 * This defines the various types of command for an lseek() system call.
 ********************************************************************************************/

#define VFS_SEEK_SET         0
#define VFS_SEEK_CUR         1
#define VFS_SEEK_END         2

/*********************************************************************************************
 * TODO : the following flags were defined in the vfs-params.h file...
 *        ... and must be documented. [AG]
 ********************************************************************************************/


//////////////////////////////////////
///    keep these flags compact    ///
//////////////////////////////////////
#define VFS_REGFILE          0x0000000
#define VFS_DIR              0x0000001
#define VFS_FIFO             0x0000002
#define VFS_DEV_CHR          0x0000004
#define VFS_DEV_BLK          0x0000008
#define VFS_DEV              0x000000C
#define VFS_SOCK             0x0000010
#define VFS_SYMLNK           0x0000020
//////////////////////////////////////

#define VFS_RD_ONLY          0x0000040
#define VFS_SYS              0x0000080
#define VFS_ARCHIVE          0x0000100
#define VFS_PIPE             0x0000200

#define VFS_IFMT	     0x0170000
#define VFS_IFSOCK	     0x0140000
#define VFS_IFLNK	     0x0120000
#define VFS_IFREG	     0x0100000
#define VFS_IFBLK	     0x0060000
#define VFS_IFDIR	     0x0040000
#define VFS_IFCHR	     0x0020000
#define VFS_IFIFO	     0x0010000

#define VFS_ISUID	     0x0004000
#define VFS_ISGID	     0x0002000
#define VFS_ISVTX	     0x0001000

#define VFS_IRWXU            0x0000700
#define VFS_IRUSR            0x0000400
#define VFS_IWUSR            0x0000200
#define VFS_IXUSR            0x0000100
#define VFS_IRWXG            0x0000070
#define VFS_IRGRP            0x0000040
#define VFS_IWGRP            0x0000020
#define VFS_IXGRP            0x0000010
#define VFS_IRWXO            0x0000007
#define VFS_IROTH            0x0000004
#define VFS_IWOTH            0x0000002
#define VFS_IXOTH            0x0000001

#define VFS_IREAD            VFS_IRUSR
#define VFS_IWRITE           VFS_IWUSR
#define VFS_IEXEC            VFS_IXUSR

#define VFS_SET(state,flag)    (state) |= (flag)
#define VFS_IS(state,flag)     (state) & (flag)
#define VFS_CLEAR(state,flag)  (state) &= ~(flag)


/* Lookup flags */
#define VFS_LOOKUP_FILE		0x1
#define VFS_LOOKUP_LAST		0x2
#define VFS_LOOKUP_OPEN		0x4
#define VFS_LOOKUP_RESTART	0x8
#define VFS_LOOKUP_RETRY	0x10
#define VFS_LOOKUP_PARENT	0x20
#define VFS_LOOKUP_HELD		0x40

/* Context flags*/
#define VFS_FS_LOCAL		0x1
#define VFS_FS_USE_MAPPER	0x2




/******************************************************************************************
 * This structure define a VFS inode.
 * It contains an extended pointer on the parent dentry, and (for directory only)
 * an hash table xhtab refering all children dentries.
 * The <parent> inode is unique for a directory (not hard links for directories). 
 * For a file, the parent field points to the first dentry who created this inode.
 * Syncronisation:
 * - the main_lock (spinlock) is used during the inode tree traversal or for inode
 *   modification (add/remove children).
 * - the data_lock (rwlock) is used during read/write accesses to the data stored
 *   in the mapper. 
 * - the mapper lock (rwlock) is only used during the radix tree traversal to return
 *   to return the relevant page for red/write.
 *****************************************************************************************/

typedef enum   
{
    INODE_TYPE_NORMAL,                 /*! file or directory                            */
    INODE_TYPE_PIPE,                   /*! POSIX pipe                                   */
    INODE_TYPE_SOCKET,                 /*! POSIX socket                                 */
    INODE_TYPE_DEV,                    /*! peripheral channel                           */
}
vfs_inode_type_t;

typedef enum 
{
    INODE_ATTR_DIRTY  = 0x01,
    INODE_ATTR_DIR    = 0x02,
    INODE_ATTR_INLOAD = 0x04,
    INODE_ATTR_NEW    = 0x08,
}
vfs_inode_attr_t;

typedef struct vfs_inode_s
{
	struct vfs_ctx_s      * ctx;         /*! Rlocal pointer on FS context                */
    uint32_t                gc;          /*! generation counter                          */
	uint32_t                inum;        /*! inode identifier (unique in file system)    */
	uint32_t                attr;        /*! inode attributes (see above)                */
	uint32_t                type;        /*! inode type (see above)                      */
	uint32_t                size;        /*! number of bytes                             */
	uint32_t                links;       /*! number of alias dentry                      */
	uid_t                   uid;         /*! user owner identifier                       */
	gid_t                   gid;         /*! group owner identifier                      */
    uint32_t                mode;        /*! access mode                                 */
	uint32_t	            refcount;    /*! reference counter (all pointers)            */
	xptr_t                  parent_xp;   /*! extended pointer on parent dentry           */
	xhtab_t                 children;    /*! embedded htab of dir entries (for dir type) */
	remote_rwlock_t         data_lock;   /*! protect read/write to data and to size      */
	remote_spinlock_t       main_lock;   /*! protect inode tree traversal and modifs     */
	xlist_entry_t           xlist;       /*! member of set of inodes in same cluster     */
	xlist_entry_t           wait_root;   /*! root of threads waiting on this inode       */
	struct vfs_inode_op_s * op;          /*! TODO ???                                    */
	struct mapper_s       * mapper;      /*! associated file cache                       */
	void                  * extend;      /*! FS specific inode extension                 */
}
vfs_inode_t;

/******************************************************************************************
 * This structure defines a directory entry.
 * A dentry contains the name of a remote file/dir, an extended pointer on the 
 * inode representing this file/dir, and a local pointer on the inode representing 
 * the parent directory.
 *****************************************************************************************/

typedef struct vfs_dentry_s
{
    struct vfs_ctx_s       * ctx;        /*! local pointer on FS context                 */
	char                     name[CONFIG_VFS_MAX_NAME_LENGTH];
	uint32_t                 length;     /*! name length (bytes)                         */
	uint32_t                 refcount;   /*! reference counter (all pointers)            */
    struct vfs_inode_s     * parent;     /*! local pointer on parent inode               */
    xptr_t                   child_xp;   /*! extended pointer on child inode             */
    xlist_entry_t            xlist;      /*! member of xlist of dentries with same key   */
	struct vfs_dentry_op_s * op;         /*! TODO                                        */
	void                   * extend;     /*! FS specific extension                       */
}
vfs_dentry_t;

/******************************************************************************************
 * This structure describes an open file/directory for a given process.
 * It is not replicated, and is dynamically allocated in the cluster that contains
 * the inode, when a thread makes an open() or opendir() system call.
 *****************************************************************************************/


typedef enum
{
    FD_ATTR_READ_ENABLE  = 0x01,       /*! read access possible                         */
    FD_ATTR_WRITE_ENABLE = 0x02,       /*! write access possible                        */
    FD_ATTR_APPEND       = 0x04,       /*! append on each write                         */
    FD_ATTR_CLOSE_EXEC   = 0x08,       /*! close file on exec                           */
    FD_ATTR_SYNC         = 0x10,       /*! synchronise FS on each write                 */
    FD_ATTR_IS_DIR       = 0x20,       /*! this is a directory                          */
}
vfs_fd_attr_t;

typedef struct vfs_file_s
{
	uint32_t                gc;         /*! generation counter                           */
	uint32_t                type;       /*! see above                                    */
	uint32_t                attr;       /*! see above                                    */
	uint32_t                offset;     /*! seek position in file                        */
	uint32_t                refcount;   /*! all pointers on this file                    */
	remote_rwlock_t       lock;       /*! protect offset modifications                 */
	struct mapper_s       * mapper;     /*! associated file cache                        */
	struct vfs_inode_s    * inode;      /*! local pointer on associated inode            */
	struct vfs_ctx_s      * ctx;        /*! file system features                         */
	struct vfs_file_op_s  * op;         /*! local set of function pointers               */
}
vfs_file_t;

/******************************************************************************************
 * This structure defines informations common to all inodes and dentries
 * of a given file system. As it is declared a global variable in the kdata segment,
 * it is replicated in all clusters and handled as private by each OS intance.
 *****************************************************************************************/

typedef enum
{
	FS_TYPE_SYSFS = 0,
	FS_TYPE_DEVFS = 1,
	FS_TYPE_FATFS = 2,
	FS_TYPE_RAMFS = 3,
 
	FS_TYPES_NR   = 4,
}
vfs_types_t;

typedef enum
{
    CTX_ATTR_READ_ONLY    = 0x01,       /*! read access possible                         */
    CTX_ATTR_SYNC         = 0x10,       /*! synchronise FS on each write                 */
}
vfs_ctx_attr_t;

typedef struct vfs_ctx_s
{
	uint32_t	              type;          /*! File System type                        */
	uint32_t	              attr;          /*! global attributes for all files in FS   */
	uint32_t                  count;         /*! number of clusters                      */
	uint32_t                  blksize;       /*! cluster size                            */
    xptr_t                    ioc_xp;        /*! extended pointer on IOC device          */
	xptr_t                    root_xp;       /*! extended pointer on root inode          */

    spinlock_t                lock;          /*! lock protecting inum allocator          */
	BITMAP( inum , CONFIG_VFS_MAX_INODES );  /*! inum allocator                          */ 

    void                    * extend;        /*! FS specific context extension           */
}
vfs_ctx_t;

/******************************************************************************************
 * This structure define the informations associated to a given file descriptor,
 * that are returned by the vfs_stat() function.
 *****************************************************************************************/

typedef struct vfs_stat_s 
{
	uint32_t    dev;        /*! ID of device containing file                             */
	uint32_t    ino;        /*! inode number                                             */
	uint32_t    mode;       /*! protection                                               */
	uint32_t    nlink;      /*! number of hard links                                     */
	uint32_t    uid;        /*! user ID of owner                                         */
	uint32_t    gid;        /*! group ID of owner                                        */
	uint32_t    rdev;       /*! device ID (if special file)                              */
	uint64_t    size;       /*! total size, in bytes                                     */
	uint32_t    blksize;    /*! blocksize for file system I/O                            */
	uint32_t    blocks;     /*! number of 512B blocks allocated                          */
	uint64_t    atime;      /*! time of last access                                      */
	uint64_t    mtime;      /*! time of last modification                                */
	uint64_t    ctime;      /*! time of last status change                               */
}
vfs_stat_t;






/****************************************************************************************** 
 * This structure defines the set of operations that can be done on a VFS context. 
 * TODO A quoi cela sert-il ? [AG]
 *****************************************************************************************/

typedef error_t (vfs_create_context_t)  ( vfs_ctx_t * context );
typedef error_t (vfs_destroy_context_t) ( vfs_ctx_t * context );
typedef error_t (vfs_read_root_t)       ( vfs_ctx_t * context , vfs_inode_t * root );
typedef error_t (vfs_reply_root_t)      ( vfs_ctx_t * context , vfs_inode_t * root );
typedef error_t (vfs_write_root_t)      ( vfs_ctx_t * context , vfs_inode_t * root );

struct vfs_ctx_op_s
{
	vfs_create_context_t  * create;      /*! allocate memory and initialize a context   */
	vfs_destroy_context_t * destroy;     /*! release memory allocated to a context      */
	vfs_read_root_t       * read_root;   /*! TODO                                       */
	vfs_reply_root_t      * repli_root;  /*! TODO                                       */
	vfs_write_root_t      * write_root;  /*! TODO                                       */
}
vfs_ctx_op_t;

/****************************************************************************************** 
 * This structure defines the set of operations that can be done on a VFS inode. 
 * TODO A quoi cela sert-il ? [AG]
 *****************************************************************************************/

typedef error_t (vfs_init_inode_t)    ( vfs_inode_t * inode );
typedef error_t (vfs_create_inode_t)  ( vfs_inode_t * parent , vfs_dentry_t  * dentry );
typedef error_t (vfs_lookup_inode_t)  ( vfs_inode_t * parent , vfs_dentry_t  * dentry );
typedef error_t (vfs_write_inode_t)   ( vfs_inode_t * inode );
typedef error_t (vfs_release_inode_t) ( vfs_inode_t * inode );
typedef error_t (vfs_unlink_inode_t)  ( vfs_inode_t * parent , vfs_dentry_t  * dentry , uint32_t flags );
typedef error_t (vfs_stat_inode_t)    ( vfs_inode_t * inode );
typedef error_t (vfs_trunc_inode_t)   ( vfs_inode_t * inode );
typedef error_t (vfs_delete_inode_t)  ( vfs_inode_t * inode );

typedef struct vfs_inode_op_s
{
	vfs_init_inode_t    * init;     /*! initialise inode from scratch                    */
	vfs_create_inode_t  * create;   /*! allocate memory for one inode                    */
	vfs_lookup_inode_t  * lookup;   /*! get one directory entry by name                  */ 
	vfs_write_inode_t   * write;    /*! update the device from the inode                 */
	vfs_release_inode_t * release;  /*! reset one inode and release associated objects   */
	vfs_unlink_inode_t  * unlink;   /*! unlink a directory entry from parent inode       */
	vfs_delete_inode_t  * delete;   /*! release memory allocated to inode when count = 0 */
	vfs_stat_inode_t    * stat;     /*! TODO                                             */
	vfs_trunc_inode_t   * trunc;    /*! change the size of a file                        */
}
vfs_inode_op_t;

/****************************************************************************************** 
 * This structure defines the set of operations that can be done on a VFS dentry.
 * TODO A quoi cela sert-il ? [AG]
 *****************************************************************************************/

typedef error_t (vfs_compare_dentry_t) ( char * first , char * second );

typedef struct vfs_dentry_op_s
{
	vfs_compare_dentry_t * compare;
}
vfs_dentry_op_t;

/****************************************************************************************** 
 * This structure defines the set of operations that can be done on a VFS file.  
 * TODO A quoi cela sert-il ? [AG]
 *****************************************************************************************/

typedef error_t (vfs_open_file_t)    ( vfs_file_t * file , void * priv );
typedef error_t (vfs_read_file_t)    ( vfs_file_t * file , char * buffer );
typedef error_t (vfs_write_file_t)   ( vfs_file_t * file , char * buffer );
typedef error_t (vfs_lseek_file_t)   ( vfs_file_t * file );
typedef error_t (vfs_close_file_t)   ( vfs_file_t * file );
typedef error_t (vfs_release_file_t) ( vfs_file_t * file );
typedef error_t (vfs_read_dir_t)     ( vfs_file_t * file );
typedef error_t (vfs_mmap_file_t)    ( vfs_file_t * file , struct vseg_s * vseg );
typedef error_t (vfs_munmap_file_t)  ( vfs_file_t * file , struct vseg_s * vseg );

typedef struct vfs_file_op_s
{
	vfs_open_file_t    * open;
	vfs_read_file_t    * read;
	vfs_write_file_t   * write;
	vfs_lseek_file_t   * lseek;
	vfs_read_dir_t     * readdir;
	vfs_close_file_t   * close;
	vfs_release_file_t * release;
	vfs_mmap_file_t    * mmap;
	vfs_munmap_file_t  * munmap;
}
vfs_file_op_t;






/*****************************************************************************************/
/******************** VFS global functions ***********************************************/
/*****************************************************************************************/ 

/****************************************************************************************** 
 * This function initializes the Virtual File System.     
 *****************************************************************************************/
void vfs_init();

/****************************************************************************************** 
 * This function mount a given file system type for a given process.     
 *****************************************************************************************/
error_t vfs_mount_fs_root( struct device_s  * device,
			               uint32_t           fs_type,
			               struct process_s * process );


/*****************************************************************************************/
/******************* FS Context related functions ****************************************/
/*****************************************************************************************/

/****************************************************************************************** 
 * This function allocates an inode identifier from the local cluster inum allocator.
 * The inum respects a fixed format:
 * - the 16 MSB bits contain the cluster identifier : cxy
 * - the 16 LSB bits contains the local inode identifier  : lid
 ******************************************************************************************
 * @ ctx      : local pointer on file system context.
 * @ inum     : [ou] buffer for allocated inode identifier.
 * @ return 0 if success / return non-zero if error.
 *****************************************************************************************/
error_t vfs_ctx_inum_alloc( vfs_ctx_t * ctx,
                            uint32_t  * inum );

/****************************************************************************************** 
 * This function release an inode identifier.                                  
 ******************************************************************************************
 * @ ctx      : local pointer on file system context.
 * @ inum     : released inode identifier.
 *****************************************************************************************/
void vfs_ctx_inum_release( vfs_ctx_t * ctx,
                           uint32_t    inum );



/*****************************************************************************************/
/************************ File Descriptor related functions ******************************/
/*****************************************************************************************/

/****************************************************************************************** 
 * This function allocates memory and initializes a new local file descriptor.
 * It must be executed in the owner cluster containing the inode.
 * If the client thread is not running in the owner cluster, it must use the
 * rpc_vfs_file_create_client() function.
 ****************************************************************************************** 
 * @ inode_xp : extended pointer on associated inode.
 * @ type     : file descriptor type.
 * @ attr     : file descriptor attributes.
 * @ file_xp  : [out] buffer for extended pointer on created file descriptor.
 * @ return 0 if success / return ENOMEM if error.
 *****************************************************************************************/
error_t vfs_file_create( xptr_t     inode_xp,
                         uint32_t   type,
                         uint32_t   attr,
                         xptr_t   * file_xp );  

/****************************************************************************************** 
 * This function releases memory allocated to a local file descriptor.
 * It must be executed by a thread running in the cluster containing the inode,
 * and the file refcount must be zero.
 * If the client thread is not running in the owner cluster, it must use the
 * rpc_vfs_file_destroy_client() function.
 ****************************************************************************************** 
 * @ file_xp  : extended pointer on file descriptor.
 *****************************************************************************************/
void vfs_file_destroy( xptr_t  file_xp );  


/****************************************************************************************** 
 * These functions increment (resp. decrement) the count field in a remote file
 * descriptor, using a remote_atomic access. 
 *****************************************************************************************/
void vfs_file_count_up  ( xptr_t   file_xp );
void vfs_file_count_down( xptr_t   file_xp );



/*****************************************************************************************/
/********************* Inode related functions *******************************************/
/*****************************************************************************************/

/****************************************************************************************** 
 * This function allocates memory from local cluster for an inode descriptor and the 
 * associated mapper. It initialise these descriptors from arguments values. 
 * The parent dentry must have been previously created.
 * If the client thread is not running in the cluster containing this inode,
 * it must use the rpc_vfs_inode_create_client() function.
 ******************************************************************************************
 * @ dentry_xp  : extended pointer on associated dentry (in parent inode cluster).
 * @ type       : file system type.
 * @ attr       : inode attributes.
 * @ mode       : inode access mode.
 * @ uid        : user owner ID.
 * @ gid        : group owner ID.
 * @ inode_xp   : [out] buffer for extended pointer on created inode.
 * # return 0 if success / return ENOMEM or EINVAL if error.
 *****************************************************************************************/
error_t vfs_inode_create( xptr_t      dentry_xp,
                          uint32_t    type,
                          uint32_t    attr,
                          uint32_t    mode,
                          uid_t       uid,
                          gid_t       gid,
                          xptr_t    * inode_xp );

/****************************************************************************************** 
 * This function releases memory allocated to an inode descriptor.
 * It must be executed by a thread running in the cluster containing the inode,
 * and the inode refcount must be zero.
 * If the client thread is not running in the owner cluster, it must use the
 * rpc_vfs_inode_destroy_client() function.
 ****************************************************************************************** 
 * @ inode  : local pointer on inode descriptor.
 *****************************************************************************************/
void vfs_inode_destroy( vfs_inode_t *  inode );  

/****************************************************************************************** 
 * This function atomically increment the inode refcount.
 * It can be called by any thread running in any cluster.
 *****************************************************************************************/
void vfs_inode_remote_up( xptr_t  inode_xp );

/****************************************************************************************** 
 * This function atomically decrement the inode refcount. 
 * It can be called by any thread running in any cluster.
 *****************************************************************************************/
void vfs_inode_remote_down( xptr_t  inode_xp );

/****************************************************************************************** 
 * This function returns the <size> of a file/dir from a remote inode,
 * taking the remote_rwlock protecting <size> in READ_MODE.
 *****************************************************************************************
 * @ inode_xp  : extended pointer on the remote inode.
 * @ return the current size.
 *****************************************************************************************/
uint32_t vfs_inode_get_size( xptr_t inode_xp );

/****************************************************************************************** 
 * This function set the <size> of a file/dir to a remote inode,
 * taking the remote_rwlock protecting <size> in WRITE_MODE.
 *****************************************************************************************
 * @ inode_xp  : extended pointer on the remote inode.
 * @ size      : value to be written.
 *****************************************************************************************/
void vfs_inode_set_size( xptr_t   inode_xp,
                         uint32_t size );

/****************************************************************************************** 
 * This function takes the main lock of a remote inode.
 * This lock protect all inode fiels, including the children dentries.
 *****************************************************************************************
 * @ inode_xp  : extended pointer on the remote inode.
 *****************************************************************************************/
void vfs_inode_remote_lock( xptr_t inode_xp );

/****************************************************************************************** 
 * This function releases the main lock of a remote inode.
 * This lock protect all inode fiels, including the children dentries.
 *****************************************************************************************
 * @ inode_xp  : extended pointer on the remote inode.
 *****************************************************************************************/
void vfs_inode_remote_unlock( xptr_t inode_xp );




/****************************************************************************************** 
 * This function TODO                                                          
 *****************************************************************************************/
error_t vfs_inode_hold( vfs_inode_t * inode,
                        uint32_t      gc );

/****************************************************************************************** 
 * This function TODO                                                          
 *****************************************************************************************/
error_t vfs_inode_trunc( vfs_inode_t * inode );

/****************************************************************************************** 
 * This function TODO                                                          
 *****************************************************************************************/
error_t vfs_inode_link( vfs_inode_t * inode,
                        uint32_t      igc );

/****************************************************************************************** 
 * This function TODO                                                          
 *****************************************************************************************/
error_t vfs_inode_unlink( vfs_inode_t * inode );

/****************************************************************************************** 
 * This function TODO                                                          
 *****************************************************************************************/
error_t vfs_inode_stat( vfs_inode_t * inode,
                        uint32_t      inum );

/****************************************************************************************** 
 * This function TODO                                                          
 *****************************************************************************************/
error_t vfs_icache_del( vfs_inode_t * inode );


/****************************************************************************************** 
 * This function TODO  Pourquoi 2 arguments ?
 *****************************************************************************************/
error_t vfs_stat_inode( vfs_inode_t * inode,
                        uint32_t      inum );


/*****************************************************************************************/
/***************** Dentry related functions **********************************************/
/*****************************************************************************************/

/****************************************************************************************** 
 * This function allocates memory from local cluster for a dentry descriptor, 
 * initialises it from  arguments values, and returns the extended pointer on dentry.
 * The inode field is not initialized, because the inode does not exist yet.
 * If the client thread is not running in the target cluster for this inode,
 * it must use the rpc_dentry_create_client() function.
 ******************************************************************************************
 * @ type       : file system type.
 * @ name       : directory entry file/dir name.
 * @ parent     : local pointer on parent inode.
 * @ dentry_xp  : [out] buffer for extended pointer on created inode.
 * @ return 0 if success / return ENOMEM or EINVAL if error.
 *****************************************************************************************/
error_t vfs_dentry_create( uint32_t      type,
                           char        * name,
                           vfs_inode_t * parent,
                           xptr_t      * dentry_xp );
 
/****************************************************************************************** 
 * This function releases memory allocated to a dentry descriptor.
 * It must be executed by a thread running in the cluster containing the dentry,
 * and the dentry refcount must be zero.
 * If the client thread is not running in the owner cluster, it must use the
 * rpc_dentry_destroy_client() function.
 ****************************************************************************************** 
 * @ dentry  : local pointer on dentry descriptor.
 *****************************************************************************************/
void vfs_dentry_destroy( vfs_dentry_t *  dentry );  

/****************************************************************************************** 
 * This function atomically increment the dentry refcount.                                
 * It can be called by any thread running in any cluster.
 *****************************************************************************************/
void vfs_dentry_remote_up( xptr_t dentry_xp );

/****************************************************************************************** 
 * This function atomically decrement the dentry refcount.                                
 * It can be called by any thread running in any cluster.
 *****************************************************************************************/
void vfs_dentry_remote_down( xptr_t dentry_xp );


/*****************************************************************************************/
/*                 Inode-Tree related functions                                          */ 
/*****************************************************************************************/ 

/****************************************************************************************** 
 * This function returns in a kernel buffer allocated by the caller function,
 * the pathname of a file/dir identified by an extended pointer on the inode. 
 * It traverse the Inode Tree from the target node to the root.
 * It can be called by any thread running in any cluster.
 ******************************************************************************************
 * @ inode_xp    : pointer on inode descriptor.
 * @ buffer      : kernel buffer for pathname (must be allocated by caller).
 * @ size        : max number of characters in buffer.
 * @ return 0 if success / return EINVAL if buffer too small.
 *****************************************************************************************/
error_t vfs_get_path( xptr_t    inode_xp,
                      char    * buffer,
                      uint32_t  max_size );

/******************************************************************************************
 * This function takes a pathname (absolute or relative to cwd) and returns an extended 
 * pointer on the associated inode, and an extended pointer on the inode context.
 * If a given name in the path is not found in the inode tree, it try to load the missing
 * dentry/inode couple, from informations found in the parent directory.
 * If this directory entry does not exist on device, it returns an error.
 ******************************************************************************************
 * @ cwd_xp     : extended pointer on current directory (for relative path).
 * @ pathname   : path in kernel space (can be relative or absolute).
 * @ client_uid : client thread user ID (for checking).
 * @ client_gid : client thread group ID (for checking).
 * @ inode_xp   : [out] buffer for extended pointer on inode.
 * @ ctx_xp     : [out] buffer for extended pointer on inode context.
 * @ return 0 if success / ENOENT if inode not found / EACCES if permissopn denied
 *****************************************************************************************/
error_t vfs_lookup( xptr_t       cwd_xp,
                    char       * pathname,
                    uint32_t     client_uid,
                    uint32_t     client_gid,
					xptr_t     * inode_xp,
                    xptr_t     * ctx_xp );

/******************************************************************************************
 * This function creates a new couple dentry/inode, and insert it in the Inode-Tree.
 * It can be executed by any thread running in any cluster, as this function
 * uses the rpc_dentry_create_client() and rpc_inode_create client() if required.
 * - The dentry is created in the cluster containing the existing <parent_xp> inode.
 * - the child inode and its associated mapper are created in a randomly selected cluster.
 * - The dentry name is defined by the <name> argument.
 ******************************************************************************************
 * @ type       : new inode type
 * @ parent_xp  : extended pointer on parent inode.
 * @ name       : new directory entry name.
 * @ child_xp   : [out] buffer for extended pointer on child inode.
 * @ return 0 if success / ENOENT if entry not found in parent directory
 *****************************************************************************************/
error_t vfs_add_child_in_parent( uint32_t type,
                                 xptr_t   parent_xp,
                                 char   * name,   
                                 xptr_t * child_xp );

/******************************************************************************************
 * TODO
 ******************************************************************************************
 * @ child_xp   : extended pointer on removed inode.
 *****************************************************************************************/
error_t vfs_remove_child_from_parent( xptr_t child_xp );

/*****************************************************************************************/
/************************ File related functions *****************************************/
/*****************************************************************************************/

/****************************************************************************************** 
 * This function allocates a vfs_file_t structure in the cluster containing the inode
 * associated to the requested file, initializes it, and returns an extended pointer 
 * on this remote open file descriptor. 
 * The pathname can be relative to current directory or absolute.
 * If the inode does not exist in the inode cache, it try to find file on the mounted
 * device, and creates an inode on a pseudo randomly selected cluster if found.
 * It the requested file does not exist on device, it creates a new inode if the
 * O_CREAT flag is set and return an error otherwise. 
 ****************************************************************************************** 
 * @ cwd_xp    : extended pointer on current directory file descriptor (for relative path).
 * @ path      : file pathname (absolute or relative to current directory).
 * @ flags     : O_RDONLY, O_WRONLY, O_CREATE etc...
 * @ file_xp   : [out] buffer for extended pointer on created remote file descriptor.
 * @ return 0 if success / return non-zero if error.
 *****************************************************************************************/
error_t vfs_open( xptr_t     cwd_xp,
		          char     * path,
		          uint32_t   flags,
		          xptr_t   * file_xp );

/****************************************************************************************** 
 * This function moves <size> bytes from the file identified by the open file descriptor
 * <file_xp> to the local kernel <buffer> , taken into account the offset in <file_xp>.
 ****************************************************************************************** 
 * @ file_xp   : extended pointer on the remote open file descriptor.
 * @ buffer    : local pointer on destination kernel buffer.
 * @ size      : requested number of bytes from offset.
 * @ returns number of bytes actually transfered / 0 if EOF / -1 if error.
 *****************************************************************************************/
uint32_t vfs_read( xptr_t   file_xp, 
                   void   * buffer,
                   uint32_t size );

/****************************************************************************************** 
 * This function moves <size> bytes from the local kernel <buffer> to the open file 
 * descriptor <file_xp>, taken into account the offset defined in <file_xp>.
 ****************************************************************************************** 
 * @ file_xp   : extended pointer on the remote open file descriptor.
 * @ buffer    : local pointer on source kernel buffer.
 * @ size      : requested number of bytes to be written from offset.
 * @ returns number of bytes actually transfered / -1 if error.
 *****************************************************************************************/
uint32_t vfs_write( xptr_t   file_xp,
                    void   * buffer,
                    uint32_t size );

/****************************************************************************************** 
 * This function set a new value in the offset of the open file descriptor <file_xp>.
 * This value depends on the <whence> argument:
 * - if VFS_SEEK_SET, new value is <offset>
 * - if VFS_SEEK_CUR, new value is current_offset + <offset>
 * - if VFS_SEEK_END, new value is file_size + <offset>
 ****************************************************************************************** 
 * @ file_xp   : extended pointer on the remote open file descriptor.
 * @ offset    : local pointer on source kernel buffer.
 * @ whence    : VFS_SEEK_SET / VFS_SEEK_CUR / VFS_SEEK_END.
 * @ new_offset : [out] buffer for new offset value.
 * @ returns 0 if success / -1 if error.
 *****************************************************************************************/
error_t vfs_lseek( xptr_t     file_xp,
                   uint32_t   offset,
                   uint32_t   whence, 
                   uint32_t * new_offset );

/****************************************************************************************** 
 * This function close an open file descriptor.                              
 ******************************************************************************************
 * @ file_xp     : extended pointer on the file descriptor.
 * @ refcount    : number of references after close.
 * @ return 0 if success / return -1 if error
 *****************************************************************************************/
error_t vfs_close( xptr_t     file_xp, 
                   uint32_t * refcount );

/****************************************************************************************** 
 * This function remove from the file system a directory entry identified by its pathname.
 * The pathname can be relative to current directory or absolute.
 ******************************************************************************************
 * @ cwd_xp   : extended pointer on the current directory file descriptor.
 * @ path     : pathname (absolute or relative to current directory).
 *****************************************************************************************/
error_t vfs_unlink( xptr_t   cwd_xp,
                    char   * path );

/****************************************************************************************** 
 * This function returns in the <ustat> structure various informations on the file TODO
 *****************************************************************************************/
error_t vfs_stat( xptr_t       file_xp,
                  vfs_stat_t * ustat );




/*****************************************************************************************/
/************************ Directory related functions ************************************/
/*****************************************************************************************/

/****************************************************************************************** 
 * This function TODO                                                          
 *****************************************************************************************/
error_t vfs_opendir( xptr_t      cwd_xp,
                     char      * path,
                     uint32_t    flags,
                     xptr_t      file_xp );

/****************************************************************************************** 
 * This function TODO                                                          
 *    fat32_readdir need cleaning
 *****************************************************************************************/
error_t vfs_readdir( xptr_t            file_xp,
                     char            * path );

/****************************************************************************************** 
 * This function TODO                                                          
 *****************************************************************************************/
error_t vfs_mkdir( xptr_t            file_xp,
                   char            * path,
                   uint32_t          mode );

/****************************************************************************************** 
 * This function remove a directory from the file system.
 *****************************************************************************************/
error_t vfs_rmdir( xptr_t            file_xp,
                   char            * path );

/****************************************************************************************** 
 * This function TODO                                                          
 *****************************************************************************************/
error_t vfs_chdir( char            * pathname,
                   xptr_t            cwd_xp );

/****************************************************************************************** 
 * This function TODO                                                          
 *****************************************************************************************/
error_t vfs_chmod( char            * pathname,
                   vfs_file_t      * cwd,
                   uint32_t          mode );

/****************************************************************************************** 
 * This function TODO                                                          
 *****************************************************************************************/
error_t vfs_closedir( xptr_t       file_xp,
                      uint32_t   * refcount );




/*****************************************************************************************/
/*******************  Pipe related functions *********************************************/
/*****************************************************************************************/

/****************************************************************************************** 
 * This function TODO ???                                                         
 *****************************************************************************************/
error_t vfs_pipe( xptr_t pipefd[2] );

/****************************************************************************************** 
 * This function TODO                                                          
 *****************************************************************************************/
error_t vfs_mkfifo( xptr_t            cwd_xp,
                    char            * path,
                    uint32_t          mode );




#endif	/* _VFS_H_ */
