/*
 * vfs.h - Virtual File System definition.
 *
 * Author  Mohamed Lamine Karaoui (2014,2015)
 *         Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _VFS_H_
#define _VFS_H_

#include <kernel_config.h>
#include <hal_types.h>
#include <hal_atomic.h>
#include <remote_rwlock.h>
#include <remote_spinlock.h>
#include <spinlock.h>
#include <list.h>
#include <xlist.h>
#include <slist.h>
#include <bits.h>
#include <xhtab.h>
#include <errno.h>
#include <metafs.h>

#include <fatfs.h>
#include <ramfs.h>
#include <devfs.h>

/****  Forward declarations  ***/

struct vfs_inode_s;
struct vfs_dentry_t;
struct vfs_ctx_t;
struct vfs_file_ref_s;
struct vfs_file_s;

struct vfs_inode_op_s;
struct vfs_dentry_op_s;
struct vfs_file_op_s;
struct vfs_ctx_op_s;

struct vfs_lookup_cmd_s;
struct vfs_lookup_rsp_s;

struct mapper_s;
struct process_s;
struct device_s;
struct vseg_s;
struct page_s;


/******************************************************************************************
 * These flags are used to define the working mode of the vfs_lookup() function.  
 *****************************************************************************************/

#define VFS_LOOKUP_DIR 	    0x01     /* the searched inode is a directory                */
#define VFS_LOOKUP_OPEN		0x02     /* the search is for an open/opendir                */
#define VFS_LOOKUP_PARENT	0x04     /* return the parent inode (not the inode itself)   */
#define VFS_LOOKUP_CREATE   0x10     /* file must be created if missing                  */
#define VFS_LOOKUP_EXCL     0x20     /* file cannot previously exist                     */    

/******************************************************************************************
 * This define the masks for the POSIX access rights to inodes
 *****************************************************************************************/

#define VFS_ISUID	         0x0004000
#define VFS_ISGID	         0x0002000
#define VFS_ISVTX	         0x0001000

#define VFS_IRWXU            0x0000700
#define VFS_IRUSR            0x0000400
#define VFS_IWUSR            0x0000200
#define VFS_IXUSR            0x0000100

#define VFS_IRWXG            0x0000070
#define VFS_IRGRP            0x0000040
#define VFS_IWGRP            0x0000020
#define VFS_IXGRP            0x0000010

#define VFS_IRWXO            0x0000007
#define VFS_IROTH            0x0000004
#define VFS_IWOTH            0x0000002
#define VFS_IXOTH            0x0000001

#define VFS_IREAD            VFS_IRUSR
#define VFS_IWRITE           VFS_IWUSR
#define VFS_IEXEC            VFS_IXUSR


/******************************************************************************************
 * This structure defines informations common to all inodes and dentries
 * of a given file system. As it is declared a global variable in the kdata segment,
 * it is replicated in all clusters and handled as private by each OS intance.
 *****************************************************************************************/

typedef enum
{
	FS_TYPE_DEVFS = 0,
	FS_TYPE_FATFS = 1,
	FS_TYPE_RAMFS = 2,
 
	FS_TYPES_NR   = 3,
}
vfs_fs_type_t;

typedef enum
{
    CTX_ATTR_READ_ONLY    = 0x01,            /*! write access prohibited                 */
    CTX_ATTR_SYNC         = 0x10,            /*! synchronise FS on each write            */
}
vfs_ctx_attr_t;

typedef struct vfs_ctx_s
{
	vfs_fs_type_t  type;                     /*! File System type                        */
	uint32_t	   attr;                     /*! global attributes for all files in FS   */
	uint32_t       count;                    /*! total number of clusters on device      */
	uint32_t       blksize;                  /*! device cluster size (bytes)             */
	xptr_t         root_xp;                  /*! extended pointer on root inode          */
    spinlock_t     lock;                     /*! lock protecting inum allocator          */
    uint32_t       bitmap[BITMAP_SIZE(CONFIG_VFS_MAX_INODES)];  /* inum allocator        */
    void         * extend;                   /*! FS specific context extension           */
}
vfs_ctx_t;

/******************************************************************************************
 * This structure define a VFS inode.
 * It contains an extended pointer on the parent dentry, and (for directory only)
 * an hash table xhtab refering all children dentries.
 * The <parent> inode is unique for a directory (not hard links for directories). 
 * For a file, the parent field points to the first dentry who created this inode.
 * Syncrhonisation:
 * - the main_lock (spinlock) is used during the inode tree traversal or for inode
 *   modification (add/remove children).
 * - the data_lock (rwlock) is used during read/write accesses to the data stored
 *   in the mapper. 
 * - the mapper lock (rwlock) is only used during the radix tree traversal to return
 *   the relevant page for read/write.
 *****************************************************************************************/

/* this enum define the VFS inode types values */

typedef enum   
{
    INODE_TYPE_FILE    =     0x001,      /*! file                                        */
    INODE_TYPE_DIR     =     0x002,      /*! directory                                   */
    INODE_TYPE_FIFO    =     0x004,      /*! POSIX named pipe                            */
    INODE_TYPE_PIPE    =     0x008,      /*! POSIX anonymous pipe                        */
    INODE_TYPE_SOCKET  =     0x010,      /*! POSIX socket                                */
    INODE_TYPE_DEV     =     0x020,      /*! character peripheral channel                */
    INODE_TYPE_SYML    =     0x080,      /*! symbolic link                               */
}
vfs_inode_type_t;

/* this enum define the VFS inode attributes values */

typedef enum 
{
    INODE_ATTR_DIRTY   =     0x01,       /* modified versus the value on device          */ 
    INODE_ATTR_INLOAD  =     0x04,       /* loading from device in progress              */
    INODE_ATTR_NEW     =     0x08,       /* not saved on device yet                      */
}
vfs_inode_attr_t;

typedef struct vfs_inode_s
{
	struct vfs_ctx_s      * ctx;         /*! local pointer on FS context                 */
    uint32_t                gc;          /*! generation counter                          */
	uint32_t                inum;        /*! inode identifier (unique in file system)    */
	uint32_t                attr;        /*! inode attributes (see above)                */
	vfs_inode_type_t        type;        /*! inode type (see above)                      */
	uint32_t                size;        /*! number of bytes                             */
	uint32_t                links;       /*! number of alias dentry                      */
	uid_t                   uid;         /*! user owner identifier                       */
	gid_t                   gid;         /*! group owner identifier                      */
    uint32_t                rights;      /*! access rights                               */
	uint32_t	            refcount;    /*! reference counter (all pointers)            */
	xptr_t                  parent_xp;   /*! extended pointer on parent dentry           */
	xhtab_t                 children;    /*! embedded xhtab of vfs_dentry_t              */
	remote_rwlock_t         data_lock;   /*! protect read/write to data and to size      */
	remote_spinlock_t       main_lock;   /*! protect inode tree traversal and modifs     */
	list_entry_t            list;        /*! member of set of inodes in same cluster     */
	xlist_entry_t           wait_root;   /*! root of threads waiting on this inode       */
	struct vfs_inode_op_s * op;          /*! TODO ???                                    */
	struct mapper_s       * mapper;      /*! associated file cache                       */
	void                  * extend;      /*! FS specific inode extension                 */
}
vfs_inode_t;

/******************************************************************************************
 * This structure defines a directory entry.
 * A dentry contains the name of a remote file/dir, an extended pointer on the 
 * inode representing this file/dir, and a local pointer on the inode representing 
 * the parent directory.
 *****************************************************************************************/

typedef struct vfs_dentry_s
{
    struct vfs_ctx_s       * ctx;        /*! local pointer on FS context                 */
	char                     name[CONFIG_VFS_MAX_NAME_LENGTH];
	uint32_t                 length;     /*! name length (bytes)                         */
	uint32_t                 refcount;   /*! reference counter (all pointers)            */
    struct vfs_inode_s     * parent;     /*! local pointer on parent inode               */
    xptr_t                   child_xp;   /*! extended pointer on child inode             */
    xlist_entry_t            xlist;      /*! member of xlist of dentries with same key   */
	struct vfs_dentry_op_s * op;         /*! TODO                                        */
	void                   * extend;     /*! FS specific extension                       */
}
vfs_dentry_t;

/******************************************************************************************
 * This file structure describes an open file/directory for a given process.
 * It is not replicated, and is dynamically allocated in the cluster that contains
 * the inode, when a thread makes an open() or opendir() system call.
 * It cannot exist a file structure without an inode structure.
 * Aa the fd_array (containing extended pointers on the open file descriptors)
 * is replicated in all process descriptors, we need a references counter.
 *****************************************************************************************/

typedef enum
{
    FD_ATTR_READ_ENABLE    = 0x01,     /*! read access possible                         */
    FD_ATTR_WRITE_ENABLE   = 0x02,     /*! write access possible                        */
    FD_ATTR_APPEND         = 0x04,     /*! append on each write                         */
    FD_ATTR_CLOSE_EXEC     = 0x08,     /*! close file on exec                           */
    FD_ATTR_SYNC           = 0x10,     /*! synchronise FS on each write                 */
    FD_ATTR_IS_DIR         = 0x20,     /*! this is a directory                          */
}
vfs_file_attr_t;

typedef struct vfs_file_s
{
	struct vfs_ctx_s      * ctx;        /*! local pointer on FS context.                 */
	uint32_t                gc;         /*! generation counter                           */
	vfs_file_attr_t         attr;       /*! file attributes bit vector (see above)       */
	vfs_inode_type_t        type;       /*! same type as inode                           */
	uint32_t                offset;     /*! seek position in file                        */
	uint32_t                refcount;   /*! all pointers on this file descriptor         */
	remote_rwlock_t         lock;       /*! protect offset modifications                 */
	struct mapper_s       * mapper;     /*! associated file cache                        */
	struct vfs_inode_s    * inode;      /*! local pointer on associated inode            */
	void                  * extend;     /*! FS specific extension                        */
}
vfs_file_t;

/******************************************************************************************
 * This structure define the informations associated to a file descriptor,
 * returned to user space by the stat() system call.
 *****************************************************************************************/

typedef struct vfs_stat_s 
{
	uint32_t    dev;        /*! ID of device containing file                             */
	uint32_t    ino;        /*! inode number                                             */
	uint32_t    mode;       /*! protection                                               */
	uint32_t    nlink;      /*! number of hard links                                     */
	uint32_t    uid;        /*! user ID of owner                                         */
	uint32_t    gid;        /*! group ID of owner                                        */
	uint32_t    rdev;       /*! device ID (if special file)                              */
	uint64_t    size;       /*! total size, in bytes                                     */
	uint32_t    blksize;    /*! blocksize for file system I/O                            */
	uint32_t    blocks;     /*! number of 512B blocks allocated                          */
	uint64_t    atime;      /*! time of last access                                      */
	uint64_t    mtime;      /*! time of last modification                                */
	uint64_t    ctime;      /*! time of last status change                               */
}
vfs_stat_t;

/*********************************************************************************************
 * This structure defines the information associated to a directory entry,
 * returned to user space by the readdir() system call.
 ********************************************************************************************/

typedef struct vfs_dirent_s
{
    uint32_t    inum;                               /*! inode identifier                    */
    uint32_t    type;                               /*! inode type                          */
    char        name[CONFIG_VFS_MAX_NAME_LENGTH];   /*! dentry name                         */
}
vfs_dirent_t;



/*****************************************************************************************/
/******************** VFS global functions ***********************************************/
/*****************************************************************************************/ 


/****************************************************************************************** 
 * This function mount a given file system type for a given process TODO.     
 *****************************************************************************************/
error_t vfs_mount_fs_root( struct device_s  * device,
			               uint32_t           fs_type,
			               struct process_s * process );


/*****************************************************************************************/
/******************* FS Context related functions ****************************************/
/*****************************************************************************************/

/****************************************************************************************** 
 * This function allocates an inode identifier from the local cluster inum allocator.
 * The inum respects a fixed format:
 * - the 16 MSB bits contain the cluster identifier : cxy
 * - the 16 LSB bits contains the local inode identifier  : lid
 ******************************************************************************************
 * @ ctx      : local pointer on file system context.
 * @ inum     : [ou] buffer for allocated inode identifier.
 * @ return 0 if success / return non-zero if error.
 *****************************************************************************************/
error_t vfs_ctx_inum_alloc( vfs_ctx_t * ctx,
                            uint32_t  * inum );

/****************************************************************************************** 
 * This function release an inode identifier.                                  
 ******************************************************************************************
 * @ ctx      : local pointer on file system context.
 * @ inum     : released inode identifier.
 *****************************************************************************************/
void vfs_ctx_inum_release( vfs_ctx_t * ctx,
                           uint32_t    inum );




/*****************************************************************************************/
/********************* Inode related functions *******************************************/
/*****************************************************************************************/

/****************************************************************************************** 
 * This function allocates memory from local cluster for an inode descriptor and the 
 * associated mapper. It initialise these descriptors from arguments values. 
 * The parent dentry must have been previously created.
 * If the client thread is not running in the cluster containing this inode,
 * it must use the rpc_vfs_inode_create_client() function.
 ******************************************************************************************
 * @ dentry_xp  : extended pointer on associated dentry (in parent inode cluster).
 * @ fs_type    : file system type.
 * @ inode_type : inode type.
 * @ attr       : inode attributes.
 * @ rights     : inode access rights.
 * @ uid        : user owner ID.
 * @ gid        : group owner ID.
 * @ inode_xp   : [out] buffer for extended pointer on created inode.
 * # return 0 if success / return ENOMEM or EINVAL if error.
 *****************************************************************************************/
error_t vfs_inode_create( xptr_t            dentry_xp,
                          vfs_fs_type_t     fs_type,
                          vfs_inode_type_t  inode_type,
                          uint32_t          attr,
                          uint32_t          rights,
                          uid_t             uid,
                          gid_t             gid,
                          xptr_t          * inode_xp );

/****************************************************************************************** 
 * This function releases memory allocated to an inode descriptor.
 * It must be executed by a thread running in the cluster containing the inode,
 * and the inode refcount must be zero.
 * If the client thread is not running in the owner cluster, it must use the
 * rpc_vfs_inode_destroy_client() function.
 ****************************************************************************************** 
 * @ inode  : local pointer on inode descriptor.
 *****************************************************************************************/
void vfs_inode_destroy( vfs_inode_t *  inode );  

/****************************************************************************************** 
 * This function atomically increment/decrement the inode refcount.
 * It can be called by any thread running in any cluster.
 *****************************************************************************************/
void vfs_inode_remote_up( xptr_t  inode_xp );
void vfs_inode_remote_down( xptr_t  inode_xp );

/****************************************************************************************** 
 * This function returns the <size> of a file/dir from a remote inode,
 * taking the remote_rwlock protecting <size> in READ_MODE.
 *****************************************************************************************
 * @ inode_xp  : extended pointer on the remote inode.
 * @ return the current size.
 *****************************************************************************************/
uint32_t vfs_inode_get_size( xptr_t inode_xp );

/****************************************************************************************** 
 * This function set the <size> of a file/dir to a remote inode,
 * taking the remote_rwlock protecting <size> in WRITE_MODE.
 *****************************************************************************************
 * @ inode_xp  : extended pointer on the remote inode.
 * @ size      : value to be written.
 *****************************************************************************************/
void vfs_inode_set_size( xptr_t   inode_xp,
                         uint32_t size );

/****************************************************************************************** 
 * This function takes the main lock of a remote inode.
 * This lock protect all inode fiels, including the children dentries.
 *****************************************************************************************
 * @ inode_xp  : extended pointer on the remote inode.
 *****************************************************************************************/
void vfs_inode_lock( xptr_t inode_xp );

/****************************************************************************************** 
 * This function releases the main lock of a remote inode.
 * This lock protect all inode fiels, including the children dentries.
 *****************************************************************************************
 * @ inode_xp  : extended pointer on the remote inode.
 *****************************************************************************************/
void vfs_inode_unlock( xptr_t inode_xp );

/****************************************************************************************** 
 * This debug function returns the current owner of the inode main lock.
 *****************************************************************************************
 * @ inode_xp  : extended pointer on the remote inode.
 * @ return extended pointer on owner thread / return XPTR_NULL if lock not taken.
 *****************************************************************************************/
xptr_t vfs_inode_owner( xptr_t inode_xp );

/****************************************************************************************** 
 * This function TODO                                                          
 *****************************************************************************************/
error_t vfs_inode_trunc( vfs_inode_t * inode );

/****************************************************************************************** 
 * This function TODO                                                          
 *****************************************************************************************/
error_t vfs_inode_link( vfs_inode_t * inode,
                        uint32_t      igc );

/****************************************************************************************** 
 * This function TODO                                                          
 *****************************************************************************************/
error_t vfs_inode_unlink( vfs_inode_t * inode );

/****************************************************************************************** 
 * This function TODO                                                          
 *****************************************************************************************/
error_t vfs_inode_stat( vfs_inode_t * inode,
                        uint32_t      inum );

/****************************************************************************************** 
 * This function TODO                                                          
 *****************************************************************************************/
error_t vfs_icache_del( vfs_inode_t * inode );


/****************************************************************************************** 
 * This function TODO  Pourquoi 2 arguments ?
 *****************************************************************************************/
error_t vfs_stat_inode( vfs_inode_t * inode,
                        uint32_t      inum );


/*****************************************************************************************/
/***************** Dentry related functions **********************************************/
/*****************************************************************************************/

/****************************************************************************************** 
 * This function allocates memory from local cluster for a dentry descriptor, 
 * initialises it from  arguments values, and returns the extended pointer on dentry.
 * The inode field is not initialized, because the inode does not exist yet.
 * If the client thread is not running in the target cluster for this inode,
 * it must use the rpc_dentry_create_client() function.
 ******************************************************************************************
 * @ fs_type    : file system type.
 * @ name       : directory entry file/dir name.
 * @ parent     : local pointer on parent inode.
 * @ dentry_xp  : [out] buffer for extended pointer on created dentry.
 * @ return 0 if success / return ENOMEM or EINVAL if error.
 *****************************************************************************************/
error_t vfs_dentry_create( vfs_fs_type_t   fs_type,
                           char          * name,
                           vfs_inode_t   * parent,
                           xptr_t        * dentry_xp );
 
/****************************************************************************************** 
 * This function releases memory allocated to a dentry descriptor.
 * It must be executed by a thread running in the cluster containing the dentry,
 * and the dentry refcount must be zero.
 * If the client thread is not running in the owner cluster, it must use the
 * rpc_dentry_destroy_client() function.
 ****************************************************************************************** 
 * @ dentry  : local pointer on dentry descriptor.
 *****************************************************************************************/
void vfs_dentry_destroy( vfs_dentry_t *  dentry );  

/****************************************************************************************** 
 * These functions atomically increment/decrement the dentry refcount.
 * It can be called by any thread running in any cluster.
 *****************************************************************************************/
void vfs_dentry_remote_up( xptr_t dentry_xp );
void vfs_dentry_remote_down( xptr_t dentry_xp );


/*****************************************************************************************/
/************************ File descriptor related functions ******************************/
/*****************************************************************************************/

/****************************************************************************************** 
 * This function allocates memory and initializes a new local file descriptor.
 * It must be executed in the cluster containing the inode.
 * If the client thread is not running in the owner cluster, it must use the
 * rpc_vfs_file_create_client() function.
 ****************************************************************************************** 
 * @ inode    : local pointer on associated inode.
 * @ attr     : file descriptor attributes.
 * @ file_xp  : [out] buffer for extended pointer on created file descriptor.
 * @ return 0 if success / return ENOMEM if error.
 *****************************************************************************************/
error_t vfs_file_create( vfs_inode_t * inode,
                         uint32_t      attr,
                         xptr_t      * file_xp );  

/****************************************************************************************** 
 * This function releases memory allocated to a local file descriptor.
 * It must be executed by a thread running in the cluster containing the inode,
 * and the file refcount must be zero.
 * If the client thread is not running in the owner cluster, it must use the
 * rpc_vfs_file_destroy_client() function.
 ****************************************************************************************** 
 * @ file  : local pointer on file descriptor.
 *****************************************************************************************/
void vfs_file_destroy( vfs_file_t *  file );  

/****************************************************************************************** 
 * These functions increment (resp. decrement) the count field in a remote file
 * descriptor, using a remote_atomic access. 
 *****************************************************************************************/
void vfs_file_count_up  ( xptr_t   file_xp );
void vfs_file_count_down( xptr_t   file_xp );



/*****************************************************************************************/
/******************* Inode-Tree related functions ****************************************/ 
/*****************************************************************************************/ 

/****************************************************************************************** 
 * This function returns in a kernel buffer allocated by the caller function,
 * the pathname of a file/dir identified by an extended pointer on the inode. 
 * It traverse the Inode Tree from the target node to the root.
 * It can be called by any thread running in any cluster.
 ******************************************************************************************
 * @ inode_xp    : pointer on inode descriptor.
 * @ buffer      : kernel buffer for pathname (must be allocated by caller).
 * @ size        : max number of characters in buffer.
 * @ return 0 if success / return EINVAL if buffer too small.
 *****************************************************************************************/
error_t vfs_get_path( xptr_t    inode_xp,
                      char    * buffer,
                      uint32_t  max_size );

/******************************************************************************************
 * This function takes a pathname (absolute or relative to cwd) and returns an extended 
 * pointer on the associated inode.
 * - If a given name in the path is not found in the inode tree, it try to load the missing
 *   dentry/inode couple, from informations found in the parent directory.
 * - If this directory entry does not exist on device, it returns an error.
 * - If the the file identified by the pathname does not exist on device but the
 *   flag CREATE is set, the inode is created.
 * - If the the file identified by the pathname exist on device but both flags EXCL
 *   and CREATE are set, an error is returned.
 ******************************************************************************************
 * @ cwd_xp      : extended pointer on current directory (for relative path).
 * @ pathname    : path in kernel space (can be relative or absolute).
 * @ lookup_mode : flags defining the working mode (defined above in this file).
 * @ inode_xp    : [out] buffer for extended pointer on inode.
 * @ return 0 if success / ENOENT if inode not found , EACCES if permissopn denied,
 *                        EAGAIN if a new complete lookup must be made
 *****************************************************************************************/
error_t vfs_lookup( xptr_t             cwd_xp,
                    char             * pathname,
                    uint32_t           lookup_mode,
					xptr_t           * inode_xp );

/******************************************************************************************
 * This function creates a new couple dentry/inode, and insert it in the Inode-Tree.
 * It can be executed by any thread running in any cluster, as this function
 * uses the rpc_dentry_create_client() and rpc_inode_create client() if required.
 * - The dentry is created in the cluster containing the existing <parent_xp> inode.
 * - the child inode and its associated mapper are created in a randomly selected cluster.
 * - The new dentry name is defined by the <name> argument.
 * - The new inode and the parent inode can have different FS types.
 ******************************************************************************************
 * @ inode_type : new inode type
 * @ fs_type    : new inode FS type.
 * @ parent_xp  : extended pointer on parent inode.
 * @ name       : new directory entry name.
 * @ child_xp   : [out] buffer for extended pointer on child inode.
 * @ return 0 if success / ENOENT if entry not found in parent directory
 *****************************************************************************************/
error_t vfs_add_child_in_parent( vfs_inode_type_t   inode_type,
                                 vfs_fs_type_t      fs_type,
                                 xptr_t             parent_xp,
                                 char             * name,   
                                 xptr_t           * child_xp );

/******************************************************************************************
 * This function removes a couple dentry/inode from the Inode-Tree, and remove it from
 * the external device.
 * TODO                   
 ******************************************************************************************
 * @ child_xp   : extended pointer on removed inode.
 *****************************************************************************************/
error_t vfs_remove_child_from_parent( xptr_t child_xp );






/*****************************************************************************************/
/****************** File access API ******************************************************/
/*****************************************************************************************/

/****************************************************************************************** 
 * This function allocates a vfs_file_t structure in the cluster containing the inode
 * associated to the file identified by <cwd_xp> & <path>.
 * It initializes it, register it in the reference process fd_array, and returns both 
 * the extended pointer on the remote file descriptor, and the index in the fd_array. 
 * The pathname can be relative to current directory or absolute.
 * If the inode does not exist in the inode cache, it try to find the file on the mounted
 * device, and creates an inode on a pseudo randomly selected cluster if found.
 * It the requested file does not exist on device, it creates a new inode if the
 * O_CREAT flag is set and return an error otherwise. 
 ****************************************************************************************** 
 * @ cwd_xp      : extended pointer on current working directory file descriptor.
 * @ path        : file pathname (absolute or relative to current directory).
 * @ flags       : defined above 
 * @ mode        : access rights (as defined by chmod)
 * @ file_xp     : [out] buffer for extended pointer on created remote file descriptor.
 * @ file_id     : [out] buffer for created file descriptor index in reference fd_array.
 * @ return 0 if success / return non-zero if error.
 *****************************************************************************************/
error_t vfs_open( xptr_t     cwd_xp,
		          char     * path,
		          uint32_t   flags,
                  uint32_t   mode,
		          xptr_t   * file_xp,
		          uint32_t * file_id );

/****************************************************************************************** 
 * This function moves <size> bytes between the file identified by the open file descriptor
 * <file_xp> and a local kernel <buffer> , taken into account the offset in <file_xp>.
 * The transfer direction is defined by the <to_buffer> argument.
 ******************************************************************************************
 * @ to_buffer : mapper -> buffer if true / buffer->mapper if false. 
 * @ file_xp   : extended pointer on the remote file descriptor.
 * @ buffer    : local pointer on local kernel buffer.
 * @ size      : requested number of bytes from offset.
 * @ returns number of bytes actually transfered / -1 if error.
 *****************************************************************************************/
error_t vfs_move( bool_t   to_buffer,
                  xptr_t   file_xp, 
                  void   * buffer,
                  uint32_t size );

/****************************************************************************************** 
 * This function set a new value in the offset of the open file descriptor <file_xp>.
 * This value depends on the <whence> argument:
 * - if VFS_SEEK_SET, new value is <offset>
 * - if VFS_SEEK_CUR, new value is current_offset + <offset>
 * - if VFS_SEEK_END, new value is file_size + <offset>
 ****************************************************************************************** 
 * @ file_xp   : extended pointer on the remote open file descriptor.
 * @ offset    : local pointer on source kernel buffer.
 * @ whence    : VFS_SEEK_SET / VFS_SEEK_CUR / VFS_SEEK_END.
 * @ new_offset : [out] buffer for new offset value.
 * @ returns 0 if success / -1 if error.
 *****************************************************************************************/
error_t vfs_lseek( xptr_t     file_xp,
                   uint32_t   offset,
                   uint32_t   whence, 
                   uint32_t * new_offset );

/****************************************************************************************** 
 * This function close an open file descriptor: 
 * 1) All entries in fd_array copies are directly cancelled by the calling thread,
 *    using remote accesses. 
 * 2) The memory allocated to file descriptor in cluster containing the inode is released.
 *    It requires a RPC if cluster containing the file descriptor is remote.
 ******************************************************************************************
 * @ file_xp     : extended pointer on the file descriptor.
 * @ file_id     : file descriptor index in fd_array.
 * @ returns 0 if success / -1 if error.
 *****************************************************************************************/
error_t vfs_close( xptr_t    file_xp,
                   uint32_t  file_id );

/****************************************************************************************** 
 * This function remove from the file system a directory entry identified by the
 * <cwd_xp> & <path> arguments.
 ******************************************************************************************
 * @ cwd_xp   : extended pointer on the current working directory file descriptor.
 * @ path     : pathname (absolute or relative to current directory).
 * @ returns 0 if success / -1 if error.
 *****************************************************************************************/
error_t vfs_unlink( xptr_t   cwd_xp,
                    char   * path );

/****************************************************************************************** 
 * This function returns, in the structure pointed by the <k_stat> kernel pointer,
 * various informations on the file descriptor identified by the <file_xp> argument.
 * TODO not implemented yet...
 ******************************************************************************************
 * @ file_xp    : extended pointer on the file descriptor of the searched directory .
 * @ k_dirent   : local pointer on the dirent_t structure in kernel space.
 * @ returns 0 if success / -1 if error.
 *****************************************************************************************/
error_t vfs_stat( xptr_t       file_xp,
                  vfs_stat_t * k_stat );

/****************************************************************************************** 
 * This function returns, in the structure pointed by the <k_dirent> kernel pointer, 
 * various infos on the directory entry currently pointed by the <file_xp> file descriptor.
 * TODO not implemented yet...
 ******************************************************************************************
 * @ file_xp    : extended pointer on the file descriptor of the searched directory .
 * @ k_dirent   : local pointer on the dirent_t structure in kernel space.
 * @ returns 0 if success / -1 if error.
 *****************************************************************************************/
error_t vfs_readdir( xptr_t         file_xp,
                     vfs_dirent_t * k_dirent );

/****************************************************************************************** 
 * This function  creates a new inode and associated dentry  for the directory defined
 * by the <cwd_xp> & <path> arguments.
 * TODO not implemented yet...
 ******************************************************************************************
 * @ cwd_xp   : extended pointer on the current working directory file descriptor.
 * @ path     : pathname (absolute or relative to current directory).                     
 * @ mode     : access rights (as defined by chmod)
 * @ returns 0 if success / -1 if error.
 *****************************************************************************************/
error_t vfs_mkdir( xptr_t     cwd_xp,
                   char     * path,
                   uint32_t   mode );

/****************************************************************************************** 
 * This function remove a directory identified by the <cwd_xp / path> arguments
 * from the file system.
 * TODO not implemented yet...
 ******************************************************************************************
 * @ cwd_xp   : extended pointer on the current working directory file descriptor.
 * @ path     : pathname (absolute or relative to current directory).                     
 * @ returns 0 if success / -1 if error.
 *****************************************************************************************/
error_t vfs_rmdir( xptr_t   cwd_xp,
                   char   * path );

/****************************************************************************************** 
 * This function makes the directory identified by <cwd_xp / path> arguments to become
 * the working directory for the calling process.
 ****************************************************************************************** 
 * @ cwd_xp      : extended pointer on current directory file descriptor (relative path).
 * @ path        : file pathname (absolute or relative to current directory).
 * return 0 if success / -1 if error.
 *****************************************************************************************/
error_t vfs_chdir( xptr_t   cwd_xp,
                   char   * path );

/****************************************************************************************** 
 * This function change the access rigths for the file identified by the <cwd_xp / path>
 * arguments. The new access rights are defined by the <mode> argument value.
 ****************************************************************************************** 
 * @ cwd_xp      : extended pointer on current directory file descriptor (relative path).
 * @ path        : file pathname (absolute or relative to current directory).
 * @ mode        : access rights new value.
 * return 0 if success / -1 if error.
 *****************************************************************************************/
error_t vfs_chmod( xptr_t        cwd_xp,
                   char        * path,
                   uint32_t      mode );

/****************************************************************************************** 
 * This function creates a named FIFO file.
 * TODO not implemented yet                                                          
 ****************************************************************************************** 
 * @ path        : FIFO pathname (absolute or relative to current directory).
 * @ cwd_xp      : extended pointer on the current working directory file descriptor.
 * @ mode        : access rights new value.
 *****************************************************************************************/
error_t vfs_mkfifo( xptr_t       cwd_xp,
                    char       * path,
                    uint32_t     mode );


/*****************************************************************************************/
/****************** Mapper access API ****************************************************/
/*****************************************************************************************/

/******************************************************************************************
 * This function makes an I/O operation to move one page from VFS to a given mapper,
 * in case of MISS on the mapper cache.
 * Depending on the file system type, it calls the proper, FS specific function.
 * It must be executed by a thread running in the cluster containing the mapper.
 * The mapper pointer is obtained from the page descriptor.
 * It takes the mapper lock before launching the IO operation.
 ******************************************************************************************
 * @ page   : local pointer on the page descriptor. 
 * @ returns 0 if success / return EINVAL if it cannot access the external device.
 *****************************************************************************************/
error_t vfs_move_page_to_mapper( struct page_s * page );

/******************************************************************************************
 * This function makes an I/0 operation to move one page from a given mapper to VFS,
 * when a dirty page in the mapper cache must be updated in the File System.
 * Depending on the file system type, it calls the proper, FS specific function.
 * It must be executed by a thread running in the cluster containing the mapper.
 * The mapper pointer is obtained from the page descriptor.
 * It takes the mapper lock before launching the IO operation.
 * It does nothing if the page is not dirty. If the page is dirty, it clear the page
 * dirty bit, and remove the page from the PPM dirty list. 
 ******************************************************************************************
 * @ page   : local pointer on the page descriptor. 
 * @ returns 0 if success / return EINVAL if it cannot access the external device.
 *****************************************************************************************/
error_t vfs_move_page_from_mapper( struct page_s * page );       






/////////////////////////////////////////////////////////////////////////////////////////// 
// These typedef define the FS specific operations that must be implemented by any 
// specific file system to be supported by the ALMOS_MKH VFS.
// These typedef are not actually used, and are only defined for documentation
/////////////////////////////////////////////////////////////////////////////////////////// 

typedef error_t (fs_init_t)          ( xptr_t vfs_root_xp );

typedef error_t (fs_inode_extend_t)  ( struct vfs_inode_s * inode,
                                       void               * extend );
 
typedef void    (fs_inode_release_t) ( struct vfs_inode_s * inode );

typedef error_t (fs_write_page_t)    ( struct page_s * page );

typedef error_t (fs_read_page_t)     ( struct page_s * page );








/* deprecated [AG]

typedef error_t (lookup_inode_t)  ( vfs_inode_t  * parent , 
                                    vfs_dentry_t * dentry );

typedef error_t (write_inode_t)   ( vfs_inode_t  * inode );

typedef error_t (release_inode_t) ( vfs_inode_t  * inode );

typedef error_t (unlink_inode_t)  ( vfs_inode_t  * parent,
                                    vfs_dentry_t * dentry,
                                    uint32_t       flags );

typedef error_t (stat_inode_t)    ( vfs_inode_t  * inode );

typedef error_t (trunc_inode_t)   ( vfs_inode_t  * inode );

typedef error_t (delete_inode_t)  ( vfs_inode_t  * inode );

typedef struct vfs_inode_op_s
{
	init_inode_t    * init;   
	create_inode_t  * create;  
	lookup_inode_t  * lookup;  
	write_inode_t   * write; 
	release_inode_t * release;
	unlink_inode_t  * unlink; 
	delete_inode_t  * delete; 
	stat_inode_t    * stat;
	trunc_inode_t   * trunc;    // change the size of a file 
}
vfs_inode_op_t;

 ****************************************************************************************** 
 * These typedef define the set of FS specific operations on a VFS DENTRY descriptor.
 * They must be implemented by any specific file system to be supported by ALMOS_MKH.
 * This code is not actually used, and is only defined for documentation
 ******************************************************************************************


typedef error_t (vfs_compare_dentry_t) ( char * first , char * second );

typedef struct vfs_dentry_op_s
{
	vfs_compare_dentry_t * compare;
}
vfs_dentry_op_t;


 ****************************************************************************************** 
 * These typedef define the set of FS specific operations on FILE descriptors
 * They must be implemented by any specific file system to be supported by ALMOS_MKH.
 * This code is not actually used, and is only defined for documentation
 ******************************************************************************************


typedef error_t (open_file_t   ) ( vfs_file_t * file,
                                   void       * extend );

typedef error_t (read_file_t   ) ( vfs_file_t * file,
                                   char       * buffer,
                                   uint32_t     count );

typedef error_t (write_file_t  ) ( vfs_file_t * file,
                                   char       * buffer,
                                   uint32_t     count );

typedef error_t (lseek_file_t  ) ( vfs_file_t * file );

typedef error_t (close_file_t  ) ( vfs_file_t * file );

typedef error_t (release_file_t) ( vfs_file_t * file );

typedef error_t (read_dir_t    ) ( vfs_file_t * file );

typedef error_t (mmap_file_t   ) ( vfs_file_t    * file ,
                                   struct vseg_s * vseg );

typedef error_t (munmap_file_t ) ( vfs_file_t    * file,
                                   struct vseg_s * vseg );

typedef struct vfs_file_op_s
{
	open_file_t    * open;
	read_file_t    * read;
	write_file_t   * write;
	lseek_file_t   * lseek;
	read_dir_t     * readdir;
	close_file_t   * close;
	release_file_t * release;
	mmap_file_t    * mmap;
	munmap_file_t  * munmap;
}
vfs_file_op_t;

*/

#endif	/* _VFS_H_ */
