/*
 * ramfs.h  RAMFS file system API definition.
 *
 * Authors   Mohamed Lamine Karaoui (2014,2015)
 *           Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _RAMFS_H_
#define _RAMFS_H_

///////////////////////////////////////////////////////////////////////////////////////////
// The RAMFS File System Rdoes not uses any external device to store data.
// It stores the dynamically created files and directories in the VFS mappers.
// The ramfs_read_page() and ramfs_write_page() functions should never be used.
// The RAMFS cannot be used as the root FS.
// There is no RAMFS context extension, and no RAMFS inode extension.
///////////////////////////////////////////////////////////////////////////////////////////


/****  Forward declarations  ****/


//////////////////////////////////////////////////////////////////////////////////////////
// These functions are called by the VFS, and must be implemented by all FS.
//////////////////////////////////////////////////////////////////////////////////////////

/******************************************************************************************
 * This function does not exist, as the RAMFS cannot be the root FS.
 *****************************************************************************************/
xptr_t ramfs_init();

/******************************************************************************************
 * This function mount a RAMFS on a given inode of the root FS.
 * It actually creates a new VFS dentry in the cluster containing the parent inode,
 * and create a new VFS inode in another cluster.
 ******************************************************************************************
 * @ parent_inode_xp : extended pointer on the parent inode.
 * @ ramfs_root_name : RAMFS root directory name.
 *****************************************************************************************/
error_t ramfs_mount( xptr_t   parent_inode_xp,
                     char   * ramfs_root_name );

/*****************************************************************************************
 * This function initializes all fields of the VFS context.
 * No extra memory is allocated for a RAMFS context.
 ****************************************************************************************/
error_t ramfs_ctx_init( struct vfs_ctx_s * vfs_ctx,
                        xptr_t             root_inode_xp );

/*****************************************************************************************
 * This function does not exist for a RAMFS context, as there is no RAMFS context.
 ****************************************************************************************/
error_t ramfs_ctx_destroy();

/*****************************************************************************************
 * This function does not exist, as the RAMFS does not use a RAMFS inode extension.
 ****************************************************************************************/
error_t ramfs_inode_create( struct vfs_inode_s * inode );

/*****************************************************************************************
 * This function does not exist, as the RAMFS does not use a RAMFS inode extension.
 ****************************************************************************************/
void ramfs_inode_destroy( struct vfs_inode_s * inode );

/*****************************************************************************************
 * This function does nothing for the RAMFS File System.
 ****************************************************************************************/
error_t ramfs_write_page( struct page_s * page );

/*****************************************************************************************
 * This function does not exist for the RAMFS File System.
 ****************************************************************************************/
error_t ramfs_read_page( struct page_s * page );


#endif	/* _RAMFS_H_ */
