/*
 * scheduler.h - Core scheduler definition.
 * 
 * Author    Alain Greiner (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _SCHEDULER_H_
#define _SCHEDULER_H_

#include <hal_types.h>
#include <list.h>
#include <spinlock.h>

/****  Forward declarations  ****/

struct core_s;
struct thread_s;

/***********************************************************************************************
 * This structure define the scheduler associated to a given core.
 * WARNING : the idle thread is executed when there is no runable thread in the list 
 * of attached threads, but is NOT part of the list of attached threads.
 **********************************************************************************************/

typedef struct scheduler_s
{
    spinlock_t        lock;         /*! readlock protecting lists of threads                  */
    uint16_t          u_threads_nr; /*! total numbre of attached user threads                 */
    uint16_t          k_threads_nr; /*! total number of attached kernel threads               */
    list_entry_t      u_root;       /*! root of list of user threads for this scheduler       */
    list_entry_t      k_root;       /*! root of list of kernel threads for this scheduler     */
    list_entry_t    * u_last;       /*! pointer on list_entry for last executed kernel thread */
    list_entry_t    * k_last;       /*! pointer on list entry for last executed user thread   */
    struct thread_s * idle;         /*! pointer on idle thread                                */
    struct thread_s * current;      /*! pointer on current running thread                     */
}
scheduler_t;

/***********************************************************************************************
 *  This function initialises the scheduler for a given core.
 **********************************************************************************************/  
void sched_init( struct core_s * core );

/***********************************************************************************************
 * This function register a new thread in a given core scheduler.
 ***********************************************************************************************
 * @ core    : local pointer on the core descriptor.
 * @ thread  : local pointer on the thread descriptor.
 **********************************************************************************************/  
void sched_register_thread( struct core_s   * core,
                            struct thread_s * thread );

/*********************************************************************************************** 
 *  This function removes a thread from the set of threads attached to a given core.
 ***********************************************************************************************
 * @ thread  : local pointer on the thread descriptor.
 **********************************************************************************************/  
void sched_remove_thread( struct thread_s * thread );

/*********************************************************************************************** 
 * This function handles pending signals for all registered threads, and tries to make 
 * a context switch for the core running the calling thread.
 * - If there is a runable thread (other than the current thread or the idle thread),
 *   the calling thread is descheduled, but its state is not modified.
 * - If there is no other runable thread, the calling thread continues execution.
 * - If there is no runable thread, the idle thread is executed.
 **********************************************************************************************/
void sched_yield();

/*********************************************************************************************** 
 * This function handles pending signals for all registered threads, and make 
 * a context switch to the thread defined by the <thread> argument.
 * If the selected thread is not attached to the same core as the calling thread,
 * or is blocked, it causes a kernel panic.
 ***********************************************************************************************
 * @ new   : local pointer on the thread to run.
 **********************************************************************************************/
void sched_switch_to( struct thread_s * new );

/***********************************************************************************************
 * This function scan all threads attached to a given core scheduler, and executes
 * the relevant actions for pending signals, such as the THREAD_SIG_KILL signal.
 ***********************************************************************************************
 * @ core    : local pointer on the core descriptor.
 **********************************************************************************************/
void sched_handle_signals( struct core_s * core );

/*********************************************************************************************** 
 * This function is used by the scheduler of a given core to actually kill a thread that has 
 * the SIG_KILL signal set (following a thread_exit() or a thread_kill() event).
 * - It checks that the thread has released all locks => panic otherwise...
 * - It detach the thread from the local process descriptor.
 * - It removes the thread from the scheduler.
 * - It release physical memory allocated for thread descriptor.
 ***********************************************************************************************
 * @ thread  : local pointer on the thread descriptor.
 **********************************************************************************************/
void sched_kill_thread( struct thread_s * thread );

/***********************************************************************************************
 * This function does NOT modify the scheduler state.
 * It just select a thread in the list of attached threads, implementing the following policy: 
 * 1) it scan the list of kernel threads, from the next thread after the last executed one,
 *    and returns the first runnable found (can be the current thread).
 * 2) if no kernel thread found, it scan the list of user thread, from the next thread after
 *    the last executed one, and returns the first runable found (can be the current thread).
 * 3) if no runable thread found, it returns the idle thread.
 ***********************************************************************************************
 * @ core    : local pointer on the core descriptor.
 * @ returns pointer on selected thread descriptor
 **********************************************************************************************/
struct thread_s * sched_select( struct core_s * core );

/*********************************************************************************************** 
 * This function scan the list of kernel threads to find an idle (blocked) RPC thread.
 ***********************************************************************************************
 * @ core    : local pointer on the core descriptor.
 * @ returns pointer on RPC thread descriptor / returns NULL if no idle RPC thread.
 **********************************************************************************************/
struct thread_s * sched_get_rpc_thead( struct core_s * core );



#endif	/* _SCHEDULER_H_ */
