/*
 * sys_getcwd.c - get process current work directory
 * 
 * Author    Alain Greiner (2016,2017)
 *
 * Copyright (c)  UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_types.h>
#include <hal_uspace.h>
#include <hal_special.h>
#include <errno.h>
#include <vfs.h>
#include <vmm.h>
#include <process.h>
#include <thread.h>
#include <printk.h>

/* TODO: user page(s) need to be locked  [AG] */

////////////////////////////////
int sys_getcwd ( char     * buf, 
                 uint32_t   nbytes )
{
	error_t    error;
    paddr_t    paddr;
    char       kbuf[CONFIG_VFS_MAX_PATH_LENGTH];
  
	thread_t  * this    = CURRENT_THREAD;
    process_t * process = this->process;

    // check buffer size
	if( nbytes < CONFIG_VFS_MAX_PATH_LENGTH )
	{
        printk("\n[ERROR] in %s : buffer too small\n", __FUNCTION__ );
		this->errno = ERANGE;
        return -1;
	}

    // check buffer in user space
    error = vmm_v2p_translate( false , buf , &paddr );

	if( error )
	{
        printk("\n[ERROR] in %s : user buffer unmapped\n", __FUNCTION__ );
		this->errno = EFAULT;
        return -1;
	}

    // get reference process cluster and local pointer
    xptr_t      ref_xp  = process->ref_xp;
    cxy_t       ref_cxy = GET_CXY( ref_xp );
    process_t * ref_ptr = (process_t *)GET_PTR( ref_xp );

    // get CWD lock in read mode
	remote_rwlock_rd_lock( XPTR( ref_cxy , &ref_ptr->cwd_lock ) );

    // call relevant VFS function
	error = vfs_get_path( XPTR( ref_cxy , &ref_ptr->vfs_cwd_xp ) ,
                          kbuf , CONFIG_VFS_MAX_PATH_LENGTH );

    // release CWD lock
	remote_rwlock_rd_unlock( XPTR( ref_cxy , &ref_ptr->cwd_lock ) );

    // copy kernel buffer to user space
    hal_copy_to_uspace( buf , kbuf , CONFIG_VFS_MAX_PATH_LENGTH );

    hal_fence();

	return 0;

}  // end sys_getcwd()
