/*
 * semaphore.h - POSIX semaphore related system calls definition.
 * 
 * Author    Ghassan Almaless (2008,2009,2010,2011,2012)
 *           Alain Greiner    (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _SEMAPHORE_H_
#define _SEMAPHORE_H_

/*********************************************************************************************
 * This defines the user sem_t structure
 ********************************************************************************************/

typedef unsigned long sem_t;

/*********************************************************************************************
 * This function initializes a semaphore, defined as a global variable in user process.
 *********************************************************************************************
 * @ sem     : pointer on semaphore in user space.
 * @ pshared : inter-process semaphore if non-zero. Not supported => must be zero.
 * @ value   : semaphore initial value.
 * @ returns 0 if success / returns -1 if failure.
 ********************************************************************************************/
int sem_init( sem_t      * sem,
              int          pshared,
              unsigned int value );

/*********************************************************************************************
 * This function destroy a semaphore defined as a global variable in user process.
 *********************************************************************************************
 * @ sem     : pointer on semaphore in user space.
 * @ returns 0 if success / returns -1 if failure.
 ********************************************************************************************/
int sem_destroy( sem_t * sem );

/*********************************************************************************************
 * This function returns in the value buffer the current semaphore current value.
 *********************************************************************************************
 * @ sem     : pointer on semaphore in user space.
 * @ value   : pointer on buffer for returned value in user space.
 * @ returns 0 if success / returns -1 if failure.
 ********************************************************************************************/
int sem_getvalue(sem_t *sem, int *value);

/*********************************************************************************************
 * This blocking function returns only when the semaphore has been obtained (or if error).
 *********************************************************************************************
 * @ sem     : pointer on semaphore in user space.
 * @ returns 0 if success / returns -1 if failure.
 ********************************************************************************************/
int sem_wait(sem_t *sem);

/*********************************************************************************************
 * This function releases a semaphore obtained by a previous sem_wait().
 *********************************************************************************************
 * @ sem     : pointer on semaphore in user space.
 * @ returns 0 if success / returns -1 if failure.
 ********************************************************************************************/
int sem_post(sem_t *sem);

#endif	/* _SEMAPHORE_H_ */
