/*
 * ppm.h - Per-cluster Physical Pages Manager Interface
 *
 * Authors  Ghassan Almaless (2008,2009,2010,2011,2012)
 *          Alain Greiner    (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-kernel is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-kernel is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-kernel; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _PPM_H_
#define _PPM_H_

#include <hal_types.h>
#include <list.h>
#include <spinlock.h>
#include <boot_info.h>
#include <page.h>


/*****************************************************************************************
 * This structure defines the Physical Memory Manager in a cluster.
 * In all clusters, the physical memory bank starts at local physical address 0.
 * The size of this local physical memory is defined by the <pages_nr> field in the
 * boot_info structure. It is split in three parts:
 * - the "kernel_code" section contains the kernel code, loaded by the boot-loader.
 *   It starts at PPN = 0 and the size is defined by the <pages_offset> field in the
 *   boot_info structure.
 * - the "pages_tbl" section contains the physical page descriptors array. It starts
 *   at PPN = pages_offset, and it contains one entry per small physical page in cluster.
 *   It is created and initialized by the hal_ppm_create() function. "the
 * - The "kernel_heap" section contains all physical pages that are are not in the
 *   in the kernel_code and pages_tbl sections, and that have not been reserved by the
 *   architecture specific bootloader. The reserved pages are defined in the boot_info
 *   structure.
 *
 * The main service provided by the PMM is the dynamic allocation of physical pages
 * from the "kernel_heap" section.
 * This low-level allocator implements the buddy algorithm: an allocated block is
 * an integer number n of 4 Kbytes pages, and n (called order) is a power of 2.
 ****************************************************************************************/

typedef struct ppm_s
{
	spinlock_t     free_lock;               /*! lock protecting free_pages[] lists      */
	list_entry_t   free_pages_root[CONFIG_PPM_MAX_ORDER];  /*! roots of free lists      */
	uint32_t       free_pages_nr[CONFIG_PPM_MAX_ORDER];    /*! numbers of free pages    */
	page_t       * pages_tbl;               /*! pointer on page descriptors array       */
	uint32_t       pages_nr;                /*! total number of small physical page     */
    spinlock_t     dirty_lock;              /*! lock protecting the dirty pages list    */
    list_entry_t   dirty_root;              /*! root of dirty pages list                */
    void         * vaddr_base;              /*! pointer on local physical memory base   */
}
ppm_t;

/*****************************************************************************************
 * This is the low-level physical pages allocation function.
 * It allocates N contiguous physical pages. N is a power of 2.
 * In normal use, you don't need to call it directly, as the recommended way to get
 * physical pages is to call the generic allocator defined in kmem.h.
 *****************************************************************************************
 * @ order        : ln2( number of 4 Kbytes pages)
 * @ returns a pointer on the page descriptor if success / NULL otherwise
 **************************************************************************************à))**/
page_t * ppm_alloc_pages( uint32_t order );

/*****************************************************************************************
 * This is the low-level physical pages release function. It takes the lock protecting
 * the free_list before register the released page in the relevant free_list.
 * In normal use, you do not need to call it directly, as the recommended way to free
 * physical pages is to call the generic allocator defined in kmem.h.
 *****************************************************************************************
 * @ page         : pointer to the page descriptor to be released
 ****************************************************************************************/
void ppm_free_pages( page_t * page );

/*****************************************************************************************
 * This function does the same as the ppm_free_page() function, without taking the lock.
 * It is used by the hal_ppm_init() function to initialize the pages_tbl[] array, when
 * there is no concurrent access issue.
 *****************************************************************************************
 * @ page         : pointer to the page descriptor to be released
 ****************************************************************************************/
void ppm_free_pages_nolock( page_t * page );

/*****************************************************************************************
 * This function check if a page descriptor pointer is valid.
 *****************************************************************************************
 * @ page         : pointer on a page descriptor
 * @ returns true if valid / false otherwise.
 ****************************************************************************************/
inline bool_t ppm_page_is_valid( page_t * page );

/*****************************************************************************************
 * Get the page virtual address from the page descriptor pointer.
 *****************************************************************************************
 * @ page         : pointer to page descriptor
 * @ returns virtual address of page itself.
 ****************************************************************************************/
inline void* ppm_page2vaddr( page_t * page );

/*****************************************************************************************
 * Get the page descriptor pointer from the page virtual address.
 *****************************************************************************************
 * @ vaddr        : page virtual address
 * @ returns pointer on page descriptor
 ****************************************************************************************/
inline page_t * ppm_vaddr2page( void * vaddr );

/*****************************************************************************************
 * Get the PPN from the page descriptor pointer.
 *****************************************************************************************
 * @ page         : pointer to page descriptor
 * @ returns physical page number
 ****************************************************************************************/
inline ppn_t ppm_page2ppn( page_t * page );

/*****************************************************************************************
 * Get the page descriptor pointer from the PPN.
 *****************************************************************************************
 * @ ppn          : physical page number
 * @ returns pointer on page descriptor
 ****************************************************************************************/
inline page_t * ppm_ppn2page( ppn_t ppn );

/*****************************************************************************************
 * Get the page virtual address from the PPN.
 *****************************************************************************************
 * @ ppn          : physical page number
 * @ returns page virtual address.
 ****************************************************************************************/
inline void* ppm_ppn2vaddr( ppn_t ppn );

/*****************************************************************************************
 * Get the PPN from the page virtual address.
 *****************************************************************************************
 * @ vaddr        : page virtual address
 * @ returns physical page number.
 ****************************************************************************************/
inline ppn_t ppm_vaddr2ppn( void * base );

/*****************************************************************************************
 * This function prints the PPM allocator status.
 *****************************************************************************************
 * @ ppm      : pointer on PPM allocator.
 * @ string   : define context of display.
 ****************************************************************************************/
void ppm_print( ppm_t * ppm,
                char  * string );

/*****************************************************************************************
 * This function checks PPM allocator consistency.
 *****************************************************************************************
 * @ ppm      : pointer on PPM allocator.
 * @ return 0 if PPM is OK / return -1 if PPM not consistent.
 ****************************************************************************************/
error_t ppm_assert_order( ppm_t * ppm );

#endif	/* _PPM_H_ */
