/*
 * syscalls.h - kernel services definition
 * 
 * Author     Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _SYSCALLS_H_
#define _SYSCALLS_H_

#include <hal_types.h>
#include <time.h>

/*****   Forward declarations  *****/

struct thread_s;                  // defined in thread.h
struct pthread_attr_s;            // defined in thread.h
struct vfs_stat_s;                // defined in vfs.h
struct vfs_dirent_s;              // defined in vfs.h
struct mmap_attr_s;               // defined in vmm.h

/******************************************************************************************
 * This enum defines the mnemonics for the syscall indexes.
 * It must be kept consistent with the array defined in do_syscalls.c
 *****************************************************************************************/
enum
{
	SYS_THREAD_EXIT    = 0,
	SYS_MMAP           = 1,
	SYS_THREAD_CREATE  = 2,
	SYS_THREAD_JOIN    = 3,
	SYS_THREAD_DETACH  = 4,
	SYS_THREAD_YIELD   = 5,
	SYS_SEM            = 6,
	SYS_CONDVAR        = 7,
	SYS_BARRIER        = 8,
	SYS_MUTEX          = 9,

	SYS_SLEEP          = 10,
	SYS_WAKEUP         = 11,
	SYS_OPEN           = 12,
	SYS_CREAT          = 13,
	SYS_READ           = 14,
	SYS_WRITE          = 15,
	SYS_LSEEK          = 16,
	SYS_CLOSE          = 17,
	SYS_UNLINK         = 18,   
	SYS_PIPE           = 19,

	SYS_CHDIR          = 20,
	SYS_MKDIR          = 21,
	SYS_MKFIFO         = 22,    
	SYS_OPENDIR        = 23,
	SYS_READDIR        = 24,
	SYS_CLOSEDIR       = 25,
	SYS_GETCWD         = 26,
	SYS_CLOCK          = 27,
	SYS_ALARM          = 28,   
	SYS_RMDIR          = 29,

	SYS_UTLS           = 30,  
	SYS_CHMOD          = 31,
	SYS_SIGNAL         = 32,
	SYS_TIMEOFDAY      = 33,
	SYS_KILL           = 34,
	SYS_GETPID         = 35,
	SYS_FORK           = 36,
	SYS_EXEC           = 37,
	SYS_STAT           = 38,     
	SYS_TRACE          = 39,
        
	SYSCALLS_NR        = 40,
};


/********************************************************************************************/
/********************    system calls    ****************************************************/ 
/********************************************************************************************/

/********************************************************************************************* 
 * [0] This function terminates the execution of the calling user thread value, 
 * and makes the exit_value pointer available to any successful pthread_join() with the 
 * terminating thread.
 *********************************************************************************************
 * @ exit_vallue  : [out] pointer to to the parrent thread if attached.
 * @ return 0 if success / return -1 if failure.
 ********************************************************************************************/
int sys_thread_exit( void * exit_value );

/********************************************************************************************* 
 * [1] This function map physical memory (or a file) in the calling thread virtual space.
 * The <attr> argument is a pointer on a structure containing arguments, defined in vmm.h.
 * TODO not implemented yet...
 *********************************************************************************************
 * @ attr       : pointer on attributes structure.
 * @ return 0 if success / return -1 if failure.
 ********************************************************************************************/
int sys_mmap( struct mmap_attr_s * attr );

/********************************************************************************************* 
 * [2] This function creates a new user thread. The <user_attr> argument is a pointer
 * on astructure containing the thread attributes, defined in thread.h file.
 *********************************************************************************************
 * @ new_thread  : [out] local pointer on created thread descriptor.
 * @ user_attr   : [in]  pointer on thread attributes structure.
 * @ start_func  : [in]  pointer on start function.
 * @ start_args  : [in]  pointer on start function arguments.
 * @ return 0 if success / return -1 if failure.
 ********************************************************************************************/
int sys_thread_create( struct thread_s        * new_thread,
                       struct pthread_attr_s  * user_attr,
                       void                   * start_func,
                       void                   * start_args );

/********************************************************************************************* 
 * [3] This blocking function suspend execution of the calling thread until completion 
 * of another target thread identified by the <trdid> argument.
 * If the <exit_value> argument is not NULL, the value passed to pthread_exit() by the 
 * target thread is stored in the location referenced by exit_value.
 *********************************************************************************************
 * @ trdid     : [in]  target thread identifier.
 * @ thread    : [out] buffer for exit_value returned by target thread. 
 * @ return 0 if success / return -1 if failure.
 ********************************************************************************************/
int sys_thread_join( trdid_t    trdid,
                     void    ** exit_value );

/********************************************************************************************* 
 * [4] This function detach a joinable thread.
 *********************************************************************************************
 * @ trdid   : thread identifier.
 * @ return 0 if success / return -1 if failure.
 ********************************************************************************************/
int sys_thread_detach( trdid_t  trdid );

/********************************************************************************************* 
 * [5] This function calls the scheduler for the core running the calling thread.
 *********************************************************************************************
 * @ return always 0. 
 ********************************************************************************************/
int sys_thread_yield();

/********************************************************************************************* 
 * [6] This function implement all operations on a POSIX unnamed semaphore,
 * that can be shared by threads running in different clusters.
 * The kernel structure representing a remote semaphore is in the remote_sem.h file,
 * and the code implementing the operations is in the remore_sem.c file.
 *********************************************************************************************
 * @ vaddr     : semaphore virtual address in user space == identifier.
 * @ operation : SEM_INIT / SEM_DESTROY / SEM_GETVALUE / SEM_POST / SEM_WAIT.
 * @ value     : pointer on in/out argument in user space.
 * @ return 0 if success / return -1 if failure.
 ********************************************************************************************/
int sys_sem( void       * vaddr,
             uint32_t     operation,
             uint32_t   * value );

typedef enum
{
	SEM_INIT,
	SEM_DESTROY,
	SEM_GETVALUE,
	SEM_WAIT,
	SEM_POST,
} 
sem_operation_t;

/********************************************************************************************* 
 * [7] This function implement all operations on a POSIX condition variable.
 * The kernel structure representing a cond_var is defined in the remote_cv.h file,
 * The code implementing the operations is defined in the remote_cv.c file.
 *********************************************************************************************
 * @ vaddr     : condvar virtual address in user space == identifier.
 * @ operation : operation type (see below).
 * @ attr      : mutex virtual address in user space == identifier.
 * @ return 0 if success / return -1 if failure.
 ********************************************************************************************/
int sys_condvar( void     * condvar,
                 uint32_t   operation,
                 void     * mutex );

typedef enum
{
	CONDVAR_INIT,
	CONDVAR_DESTROY,
    CONDVAR_WAIT,
    CONDVAR_SIGNAL,
    CONDVAR_BROADCAST,
} 
condvar_operation_t;

/********************************************************************************************* 
 * [8] This function implement all operations on a POSIX barrier.
 * The kernel structure representing a barrier is defined in the remote_barrier.h file.
 * The code implementting the operations is defined in the remote_barrier.c file.
 *********************************************************************************************
 * @ vaddr     : barrier virtual address in user space == identifier.
 * @ operation : BARRIER_INIT / BARRIER_DESTROY / BARRIER_WAIT.
 * @ count     : number of expected threads (only used by BARRIER_INIT operation).
 * @ return 0 if success / return -1 if failure.
 ********************************************************************************************/
int sys_barrier( void     * vaddr,
                 uint32_t   operation,
                 uint32_t   count );

typedef enum
{
	BARRIER_INIT,
	BARRIER_DESTROY,
	BARRIER_WAIT,
} 
barrier_operation_t;

/********************************************************************************************* 
 * [9] This function implement all operations on a POSIX mutex.
 * The kernel structure representing a barrier is defined in the remote_barrier.h file.
 * The code implementting the operations is defined in the remote_barrier.c file.
 *********************************************************************************************
 * @ vaddr     : mutex virtual address in user space == identifier.
 * @ operation : MUTEX_INIT / MUTEX_DESTROY / MUTEX_LOCK / MUTEX_UNLOCK
 * @ attr      : mutex attributes (non supported yet => must be 0).
 * @ return 0 if success / return -1 if failure.
 ********************************************************************************************/
int sys_mutex( void     * vaddr,
               uint32_t   operation,
               uint32_t   count );

typedef enum
{
	MUTEX_INIT,
	MUTEX_DESTROY,
	MUTEX_LOCK,
	MUTEX_UNLOCK,
} 
mutex_operation_t;

/********************************************************************************************* 
 * [10] This function block the calling thread on the THREAD_BLOCKED_GLOBAL condition,
 * and deschedule.
 *********************************************************************************************
 * @ return 0 if success / returns -1 if failure.
 ********************************************************************************************/
int sys_thread_sleep();

/********************************************************************************************* 
 * [11] This function unblock the thread identified by its <trdid> from the 
 * THREAD_BLOCKED_GLOBAL condition.
 *********************************************************************************************
 * @ trdid  : target thread identifier.
 * @ return 0 if success / return -1 if failure.
 ********************************************************************************************/
int sys_thread_wakeup();

/********************************************************************************************* 
 * [12] This function open or create a file.
 *********************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ flags      : bit vector attributes (see below).
 * @ mode       : access rights.
 * @ return file descriptor index in fd_array if success / return -1 if failure.
 ********************************************************************************************/
int sys_open( char    * pathname,
              uint32_t  flags,
              uint32_t  mode );

typedef enum
{
    O_RDONLY   = 0x0010000,    /*! open file in read-only mode                              */
    O_WRONLY   = 0x0020000,    /*! open file in write-only mode                             */
    O_RDWR     = 0x0030000,    /*! open file in read/write mode                             */
    O_NONBLOCK = 0x0040000,    /*! do not block if data non available                       */
    O_APPEND   = 0x0080000,    /*! append on each write                                     */
    O_CREAT    = 0x0100000,    /*! create file if it does not exist                         */
    O_TRUNC    = 0x0200000,    /*! file length is forced to 0                               */
    O_EXCL     = 0x0400000,    /*! error if VFS_O_CREAT and file exist                      */
    O_SYNC	   = 0x0800000,    /*! synchronize File System on each write                    */
    O_CLOEXEC  = 0x1000000,    /*! set the close-on-exec flag in file descriptor            */
    O_DIR      = 0x2000000,    /*! new file descriptor is for a directory                   */
}
open_attributes_t;

/********************************************************************************************* 
 * [13] This function creates a new file as specified by the arguments.
 * This function is obsolete, you should use open() with 0_CREATE.
 *********************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ mode       : access rights.
 ********************************************************************************************/
int sys_creat( char     * pathname,
               uint32_t   mode );

/********************************************************************************************* 
 * [14] This function read bytes from an open file identified by its file descriptor.
 * This file can be a regular file or character oriented device.
 *********************************************************************************************
 * @ file_id  : open file index in fd_array.
 * @ buf      : buffer virtual address in user space.
 * @ count    : number of bytes.
 * @ return number of bytes actually read if success / returns -1 if failure.
 ********************************************************************************************/
int sys_read( uint32_t   file_id,
              void     * buf,
              uint32_t   count );

/********************************************************************************************* 
 * [15] This function writes bytes to an open file identified by its file descriptor.
 * This file can be a regular file or character oriented device.
 *********************************************************************************************
 * @ file_id  : open file index in fd_array.
 * @ buf      : buffer virtual address in user space.
 * @ count    : number of bytes.
 * @ return number of bytes actually written if success / returns -1 if failure.
 ********************************************************************************************/
int sys_write( uint32_t   file_id,
               void     * buf,
               uint32_t   count );

/********************************************************************************************* 
 * [16] This function epositions the offset of the file descriptor identified by <file_id>,
 * according to the operation type defined by the <whence> and <offset> arguments.
 *********************************************************************************************
 * @ file_id  : open file index in fd_array.
 * @ offset   : buffer virtual address in user space.
 * @ whence   : operation type (see below).
 * @ return 0 if success / returns -1 if failure.
 ********************************************************************************************/
int sys_lseek( xptr_t    file_id,
               uint32_t  offset,
               uint32_t  whence );

typedef enum
{
    SEEK_SET  = 0,     /*! new_offset <= offset                                             */
    SEEK_CUR  = 1,     /*! new_offset <= current_offset + offset                            */
    SEEK_END  = 2,     /*! new_iffset <= current_size + offset                              */
}
lseek_operation_t;

/********************************************************************************************* 
 * [17] This function release the memory allocated for the file descriptor identified by 
 * the <file_id> argument, and remove the fd array_entry in all copies of the process 
 * descriptor.
 *********************************************************************************************
  file_id   : file descriptor index in fd_array.
 * @ return 0 if success / returns -1 if failure.
 ********************************************************************************************/
int sys_close( uint32_t file_id );

/********************************************************************************************* 
 * [18] This function removes a directory entry identified by the <pathname> from its
 * directory, and decrement the link count of the file referenced by the link.
 * If the link count reduces to zero, and no process has the file open, then all resources
 * associated with the file are reclaimed.  If one or more process have the file open when
 * the last link is removed, the link is removed, but the removal of the file is delayed 
 * until all references to it have been closed.
 *********************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ return 0 if success / returns -1 if failure.
 ********************************************************************************************/
int sys_unlink( char * pathname );

/********************************************************************************************* 
 * [19] This function creates in the calling thread cluster an unnamed pipe, and two
 * (read and write) file descriptors.
 *********************************************************************************************
 * @ file_id[0] : [out] read only file descriptor index.
 * @ file_id[1] : [out] write only file descriptor index.
 * @ return 0 if success / return -1 if failure.
 ********************************************************************************************/
int sys_pipe( uint32_t file_id[2] );

/********************************************************************************************* 
 * [20] This function change the current working directory in reference process descriptor.
 *********************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ return 0 if success / returns -1 if failure.
 ********************************************************************************************/
int sys_chdir( char * pathname );

/********************************************************************************************* 
 * [21] This function creates a new directory in file system.
 *********************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ mode       : access rights (as defined in chmod).
 * @ return 0 if success / returns -1 if failure.
 ********************************************************************************************/
int sys_mkdir( char      pathname,
               uint32_t  mode );

/********************************************************************************************* 
 * [22] This function creates a named FIFO file in the calling thread cluster.
 * The associated read and write file descriptors mut be be  explicitely created
 * using the sy_open() function.
 *********************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ mode       : access rights (as defined in chmod).
 * @ return 0 if success / returns -1 if failure.
 ********************************************************************************************/
int sys_mkfifo( char     * pathname,
                uint32_t   mode );

/********************************************************************************************* 
 * [23] This function open a directory, that must exist in the file system.
 *********************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ return file descriptor index in fd_array if success / return -1 if failure.
 ********************************************************************************************/
int sys_opendir( char * pathname );

/********************************************************************************************* 
 * [24] This function returns in the structure pointed by the <dirent> argument various
 * information about an entry of the directory identified by the <file_id> argument.
 *********************************************************************************************
 * @ file_id   : file descriptor index of the searched directory.
 * @ dirent    : pointer on a dirent structure in user space.
 * @ return 0 if success / returns -1 if failure.
 ********************************************************************************************/
int sys_readdir( uint32_t              file_id,
                 struct vfs_dirent_s * dirent );

/********************************************************************************************* 
 * [25] This function close the file descriptor previouly open by the opendir() function.
 *********************************************************************************************
 * @ file_id   : file descriptor index of the searched directory.
 * @ return 0 if success / returns -1 if failure.
 ********************************************************************************************/
int sys_closedir( uint32_t file_id );

/********************************************************************************************* 
 * [26] This function returns the pathname of the current working directory.
 *********************************************************************************************
 * buf     : buffer addres in user space.
 * nbytes  : user buffer size in bytes.
 * @ return 0 if success / returns -1 if failure.
 ********************************************************************************************/
int sys_getcwd( char     * buf,
                uint32_t   nbytes );

/********************************************************************************************* 
 * [27] This function returns in a 64 bits user buffer the calling core cycles count.
 * It uses both the hardware register and the core descriptor cycles count tos take
 * into account a possible harware register overflow  in 32 bits architectures.
 *********************************************************************************************
 * cyles    : [out] address of buffer in user space.
 ********************************************************************************************/
int sys_clock( uint64_t * cycles );

/********************************************************************************************* 
 * [28] This function forces the calling thread to sleep, for a fixed number of cycles.
 *********************************************************************************************
 * cycles   : number of cycles.
 ********************************************************************************************/
int sys_alarm( uint32_t cycles );

/********************************************************************************************* 
 * [29] This undefined function does nothing.
 *********************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ return 0 if success / returns -1 if failure.
 ********************************************************************************************/
int sys_rmdir( char * pathname ); 

/********************************************************************************************* 
 * [30] This function implement the operations related to User Thread Local Storage.
 * It is actually implemented as an uint32_t variable in the thread descriptor.
 *********************************************************************************************
 * @ operation  : UTLS operation type as defined below.
 * @ value      : argument value for the UTLS_SET operation.
 * @ return value for the UTLS_GET and UTLS_GET_ERRNO / return -1 if failure.
 ********************************************************************************************/
int sys_utls( uint32_t operation,
              uint32_t value );

typedef enum
{
    UTLS_SET       = 1,
    UTLS_GET       = 2,
    UTLS_GET_ERRNO = 3,
}
utls_operation_t;

/********************************************************************************************* 
 * [31] This function change the acces rights for the file/dir identified by the
 * pathname argument.
 *********************************************************************************************
 * @ pathname   : pathname (can be relative or absolute).
 * @ rights     : acces rights.
 * @ return 0 if success / returns -1 if failure.
 ********************************************************************************************/
int sys_chmod( char     * pathname,
               uint32_t   rights );

/********************************************************************************************* 
 * [32] This function associate a specific signal handler to a given signal type.
 * Tee handlers for the SIGKILL and SIGSTOP signals cannot be redefined.
 ********************************************************************************************* 
 * @ sig_id    : index defining signal type (from 1 to 31).
 * @ handler   : pointer on fonction implementing the specific handler.
 * @ return 0 if success / returns -1 if failure.
 ********************************************************************************************/
int sys_signal( uint32_t   sig_id,
                void     * handler );

/********************************************************************************************* 
 * [33] This function returns in the structure <tv>, defined in the time.h file,
 * the current time (in seconds & micro-seconds).
 * It is computed from the calling core descriptor.
 * The timezone is not supported.
 *********************************************************************************************
 * @ tv      : pointer on the timeval structure.
 * @ tz      : pointer on the timezone structure : must be NULL.       
 * @ return 0 if success / returns -1 if failure.
 ********************************************************************************************/
int sys_timeofday( struct timeval  * tv,
                   struct timezone * tz );

/********************************************************************************************* 
 * [34] This function implements the "kill" system call.
 * It register the signal defined by the <sig_id> argument in all thread descriptors 
 * of a target process identified by the <pid> argument. This is done in all clusters
 * containing threads for the target process.
 * It can be executed by any thread running in any cluster, as this function uses
 * remote access to traverse the list of process copies in the owner cluster,
 * and the RPC_SIGNAL_RISE to signal the remote threads. 
 ********************************************************************************************* 
 * @ pid      : target process identifier.
 * @ sig_id   : index defining the signal type (from 1 to 31). 
 * @ return 0 if success / returns -1 if failure.
 ********************************************************************************************/
int sys_kill( pid_t    pid,
              uint32_t sig_id );

/********************************************************************************************* 
 * [35] This function implements the "getpid" system call.
 ********************************************************************************************* 
 * @ returns the PID for the calling thread.
 ********************************************************************************************/
int sys_getpid();

/********************************************************************************************* 
 * [36] This function implement the "fork" system call.
 * The calling process descriptor (parent process), and the associated thread descriptor are
 * replicated in the same cluster as the calling thread, but the new process (child process)
 * is registered in another target cluster, that is the new process owner. 
 * The child process and the associated main thread will be migrated to the target cluster
 * later, when the child process makes an "exec" or any other system call. 
 * The target cluster depends on the "fork_user" flag and "fork_cxy" variable that can be
 * stored in the calling thread descriptor by the specific fork_place() system call.
 * If not, the sys_fork() function makes a query to the DQDT to select the target cluster. 
 ********************************************************************************************* 
 * @ returns child process PID if success / returns -1 if failure
 ********************************************************************************************/
int sys_fork();

/********************************************************************************************* 
 * [37] This function implement the "exec" system call.
 * It is executed in the client cluster, but the new process descriptor and main thread
 * must be created in a server cluster, that is generally another cluster. 
 * - if the server_cluster is the client cluster, call directly the process_make_exec()
 *   function to create a new process, and launch a new thread in local cluster.
 * - if the target_cluster is remote, call rpc_process_exec_client() to execute the
 *   process_make_exec() on the remote cluster.
 * In both case this function build an exec_info_t structure containing all informations
 * required to build the new process descriptor and the associated thread.
 * Finally, the calling process and thread are deleted.
 ********************************************************************************************* 
 * @ filename : string pointer on .elf filename (virtual pointer in user space)
 * @ argv     : array of strings on process arguments (virtual pointers in user space)
 * @ envp     : array of strings on Renvironment variables (virtual pointers in user space)
 * @ returns O if success / returns -1 if failure.
 ********************************************************************************************/
int sys_exec( char  * filename,
              char ** argv,
              char ** envp );

/********************************************************************************************* 
 * [38] This function  returns in the <stat> structure, defined in the vfs.h file,
 * various informations on the file/directory identified by the <file_id> argument.
 ********************************************************************************************* 
 * @ file_id   : file descriptor index in fd_array.
 * @ stat      : pointer on the stat structure.
 * @ returns O if success / returns -1 if failure.
 ********************************************************************************************/
int sys_stat( uint32_t            file_id,
              struct vfs_stat_s * stat );

/********************************************************************************************* 
 * [39] This function is used to activate / desactivate Rthe trace for a thread 
 * identified by the <trdid> and <pid> arguments.
 * It can be called by any other thread.
 ********************************************************************************************* 
 * @ operation  : operation type as defined below.
 * @ pid        : process identifier.
 * @ trdid      : thread identifier.
 * @ returns O if success / returns -1 if failure.
 ********************************************************************************************/
int sys_trace( uint32_t operation,
               pid_t    pid, 
               uint32_t trdid );

typedef enum
{
    TRACE_ON       = 0,
    TRACE_OFF      = 1,
}
trace_operation_t;


#endif  // _SYSCALLS_H_
