/*
 * sys_condvar.c - Access a POSIX condvar.
 * 
 * Author    Alain Greiner  (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_special.h>
#include <errno.h>
#include <thread.h>
#include <printk.h>
#include <vmm.h>
#include <syscalls.h>
#include <remote_condvar.h>

////////////////////////////////////////
int sys_condvar( void         * condvar,
                 uint32_t       operation,
                 void         * mutex )
{
	error_t    error;
    paddr_t    paddr;
 
    thread_t * this = CURRENT_THREAD;

    // check condvar in user vspace
	error = vmm_v2p_translate( false , condvar , &paddr );
	if( error )
    {
        printk("\n[ERROR] in %s : illegal condvar virtual address = %x\n",
               __FUNCTION__ , (intptr_t)condvar );
        this->errno = error;
        return -1;
    }

    // execute requested operation
	switch( operation )
	{
        //////////////////
        case CONDVAR_INIT:
        {
            error = remote_condvar_create( (intptr_t)condvar );
    
		    if( error )
            {
                printk("\n[ERROR] in %s : cannot create condvar = %x\n",
                       __FUNCTION__ , (intptr_t)condvar );
                this->errno = error;
                return -1;
            }
		    break;
        }
        //////////////////
	    case CONDVAR_WAIT:
        {
            // check mutex in user vspace
	        error = vmm_v2p_translate( false , mutex , &paddr );

	        if( error )
            {
                printk("\n[ERROR] in %s : illegal condvar virtual address = %x\n",
                       __FUNCTION__ , (intptr_t)condvar );
                this->errno = error;
                return -1;
            }

            xptr_t condvar_xp = remote_condvar_from_ident( (intptr_t)condvar );

            if( condvar_xp == XPTR_NULL )     // user error
            {
                printk("\n[ERROR] in %s : condvar %x not registered\n",
                       __FUNCTION__ , (intptr_t)condvar );
                this->errno = EINVAL;
                return -1;
            }
    
            xptr_t mutex_xp = remote_condvar_from_ident( (intptr_t)condvar );
            if( mutex_xp == XPTR_NULL )     // user error
            {
                printk("\n[ERROR] in %s : mutex %x not registered\n",
                       __FUNCTION__ , (intptr_t)condvar );
                this->errno = EINVAL;
                return -1;
            }

            remote_condvar_wait( condvar_xp , mutex_xp );

            break;
        }
        ////////////////////
	    case CONDVAR_SIGNAL:
        {
            xptr_t condvar_xp = remote_condvar_from_ident( (intptr_t)condvar );

            if( condvar_xp == XPTR_NULL )     // user error
            {
                printk("\n[ERROR] in %s : condvar %x not registered\n",
                       __FUNCTION__ , (intptr_t)condvar );
                this->errno = EINVAL;
                return -1;
            }

            remote_condvar_signal( condvar_xp ); 
            
            break;
        }
        ///////////////////////
	    case CONDVAR_BROADCAST:
        {
            xptr_t condvar_xp = remote_condvar_from_ident( (intptr_t)condvar );

            if( condvar_xp == XPTR_NULL )     // user error
            {
                printk("\n[ERROR] in %s : condvar %x not registered\n",
                       __FUNCTION__ , (intptr_t)condvar );
                this->errno = EINVAL;
                return -1;
            }

            remote_condvar_broadcast( condvar_xp );

            break;
        }
        /////////////////////
	    case CONDVAR_DESTROY:
        {
            xptr_t condvar_xp = remote_condvar_from_ident( (intptr_t)condvar );

            if( condvar_xp == XPTR_NULL )     // user error
            {
                printk("\n[ERROR] in %s : condvar %x not registered\n",
                       __FUNCTION__ , (intptr_t)condvar );
                this->errno = EINVAL;
                return -1;
            }

            remote_condvar_destroy( condvar_xp ); 

            break;
		}
        /////////
	    default:
        {
            printk("\n[PANIC] in %s : illegal operation type\n", __FUNCTION__ );
            hal_core_sleep();
        }
	}   // end switch

	return 0;

}  // enc sys_condvar()

