/*
 * sys_write.c - write bytes to a file
 * 
 * Author        Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_types.h>
#include <hal_uspace.h>
#include <hal_special.h>
#include <errno.h>
#include <vfs.h>
#include <thread.h>
#include <printk.h>
#include <process.h>

/* TODO: user page(s) need to be locked  [AG] */

//////////////////////////////////
int sys_write( uint32_t   file_id,
               void     * buf,
               uint32_t   count )
{
    error_t      error;
    paddr_t      paddr;
    char         kbuf[CONFIG_VFS_KBUF_SIZE];
	xptr_t       file_xp;                       // remote file extended pointer
    uint32_t     nbytes;                        // number of bytes in one iteration

	thread_t   * this = CURRENT_THREAD;
	process_t  * process = this->process;
 
    // check file_id argument
	if( file_id >= CONFIG_PROCESS_FILE_MAX_NR )
	{
        printk("\n[ERROR] in %s : illegal file descriptor index\n", __FUNCTION__ );
        this->errno = EBADFD;
		return -1;
	}

    // check user buffer in user space
    error = vmm_v2p_translate( false , buf , &paddr );

    if ( error )
    {
        printk("\n[ERROR] in %s : user buffer unmapped = %x\n",
               __FUNCTION__ , (intptr_t)buf );
		this->errno = EINVAL;
		return -1;
    }

    // get extended pointer on remote file descriptor
    file_xp = process_fd_get_xptr( process , file_id );

    if( file_xp == XPTR_NULL )
    {
        printk("\n[ERROR] in %s : undefined file descriptor index = %d\n",
               __FUNCTION__ , file_id );
		this->errno = EBADFD;
		return -1;
    }

    // get file descriptor cluster and local pointer
    vfs_file_t * file_ptr = (vfs_file_t *)GET_PTR( file_xp );
    cxy_t        file_cxy = GET_CXY( file_xp );

    // check file writable
    uint32_t attr = hal_remote_lw( XPTR( file_cxy , &file_ptr->attr ) );
    if( (attr & FD_ATTR_WRITE_ENABLE) == 0 )
	{
        printk("\n[ERROR] in %s : file %d not writable\n",
               __FUNCTION__ , file_id );
		this->errno = EBADFD;
		return -1;
	}
   
    // transfer at most CONFIG_VFS_KBUF_SIZE bytes per iteration
    while( count )
    {
        if( count <= CONFIG_VFS_KBUF_SIZE ) 
        {
            nbytes = count;
            count  = 0;
        }
        else
        {
            nbytes = CONFIG_VFS_KBUF_SIZE;
            count  = count - CONFIG_VFS_KBUF_SIZE;
        }

        // copy user buffer to kernel buffer
        hal_copy_to_uspace( buf , kbuf , nbytes );

        // transfer nbytes from kernel buffer
        error = vfs_move( false,               // from buffer 
                          true,                // is_user
                          file_xp,
                          kbuf , 
                          nbytes );

        if( error )
        {
            printk("\n[ERROR] in %s cannot read data from file %d\n",
                   __FUNCTION__ , file_id );
            this->errno = error;
            return -1;
        }
    }

    hal_fence();

	return 0;

}  // end sys_write()
