/*
 * sys_mutex.c - Access a POSIX mutex.
 * 
 * Author    Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_special.h>
#include <errno.h>
#include <thread.h>
#include <printk.h>
#include <vmm.h>
#include <syscalls.h>
#include <remote_mutex.h>


/////////////////////////////////
int sys_mutex( void     * vaddr,
               uint32_t   operation,
               uint32_t   attr )
{
	error_t    error;
    paddr_t    paddr;

    thread_t * this = CURRENT_THREAD;

    // check vaddr in user vspace
	error = vmm_v2p_translate( false , vaddr , &paddr );
	if( error )
    {
        printk("\n[ERROR] in %s : illegal virtual address = %x\n",
               __FUNCTION__ , (intptr_t)vaddr );
        this->errno = error;
        return -1;
    }

    // execute requested operation
	switch( operation )
	{   
        //////////////// 
	    case MUTEX_INIT:
        { 
            if( attr != 0 )
            {
                printk("\n[ERROR] in %s : mutex attributes non supported yet\n",
                       __FUNCTION__ );
                this->errno = error;
                return -1;
            }
   
            error = remote_mutex_create( (intptr_t)vaddr );

            if( error )
            {
                printk("\n[ERROR] in %s : cannot create mutex\n",
                       __FUNCTION__ );
                this->errno = error;
                return -1;
            } 
		    break;
		}
        ///////////////////
	    case MUTEX_DESTROY:
        {
            xptr_t mutex_xp = remote_mutex_from_ident( (intptr_t)vaddr );

            if( mutex_xp == XPTR_NULL )     // user error
            {
                printk("\n[ERROR] in %s : mutex %x not registered\n",
                       __FUNCTION__ , (intptr_t)vaddr );
                this->errno = EINVAL;
                return -1;
            }
            else                          // success
            {
                remote_mutex_destroy( mutex_xp ); 
            }
            break;
        }
        ////////////////
	    case MUTEX_LOCK:
        {
            xptr_t mutex_xp = remote_mutex_from_ident( (intptr_t)vaddr );

            if( mutex_xp == XPTR_NULL )     // user error
            {
                printk("\n[ERROR] in %s : mutex %x not registered\n",
                       __FUNCTION__ , (intptr_t)vaddr );
                this->errno = EINVAL;
                return -1;
            }
            else                          // success
            {
                remote_mutex_lock( mutex_xp ); 
            }
            break;
        }
        //////////////////
	    case MUTEX_UNLOCK:
        {
            xptr_t mutex_xp = remote_mutex_from_ident( (intptr_t)vaddr );

            if( mutex_xp == XPTR_NULL )     // user error
            {
                printk("\n[ERROR] in %s : mutex %x not registered\n",
                       __FUNCTION__ , (intptr_t)vaddr );
                this->errno = EINVAL;
                return -1;
            }
            else                          // success
            {
                remote_mutex_unlock( mutex_xp ); 
            }
            break;
        }
        ////////
	    default:
        {
            printk("\n[PANIC] in %s : illegal operation type\n", __FUNCTION__ );
		    hal_core_sleep();
        }
	}

	return 0;

}  // end sys_mutex()

