/*
 * hal_exception.c - implementation of exception handler for TSAR-MIPS32.
 * 
 * Author   Alain Greiner (2016, 2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_irqmask.h>
#include <hal_exception.h>
#include <thread.h>
#include <printk.h>
#include <vmm.h>
#include <errno.h>
#include <scheduler.h>
#include <core.h>
#include <signal.h>
#include <syscalls.h>
#include <do_exception.h>
#include <remote_spinlock.h>
#include <mips32_uzone.h>

//////////////////////////////////////////////////////////////////////////////////////////
//  Extern global variables
//////////////////////////////////////////////////////////////////////////////////////////

extern remote_spinlock_t  exception_lock;  // allocated in the do_exception.c file.

//////////////////////////////////////////////////////////////////////////////////////////
// This enum defines the relevant values for XCODE field in mips32 CP0_CR register.
//////////////////////////////////////////////////////////////////////////////////////////

typedef enum
{
    XCODE_ADEL = 4,        // Illegal address for data load 
    XCODE_ADES = 5,        // Illegal address for data store
    XCODE_IBE  = 6,        // Instruction MMU exception       (can be NON-FATAL)
    XCODE_DBE  = 7,        // Data MMU exception              (can be NON-FATAL)
    XCODE_RI   = 10,       // Reserved instruction exception
    XCODE_CPU  = 11,       // Coprocessor unusable exception  (can be NON-FATAl)
    XCODE_OVR  = 12        // Arithmetic Overflow exception
}
xcode_values_t;

///////////////////////////////////////
void hal_do_exception( thread_t * this, 
                       reg_t    * regs_tbl )
{
	error_t             error;
	uint32_t            excCode;                  // 4 bits XCODE from CP0_CR

    // get 4 bits XCODE from CP0_CR register
	excCode        = (regs_tbl[UZ_CR] >> 2) & 0xF;

	switch(excCode)
	{
        case XCODE_DBE:     // can be non fatal
	    case XCODE_IBE:     // can be non fatal
        {
            // call generic excepton handler for a MMU exception
		    error = do_exception( this , true );
        }
		break;

	    case XCODE_CPU:    // can be non fatal
        {
            if( ((regs_tbl[UZ_CR] >> 28) & 0x3) == 1 )  // unavailable FPU
            {
                // call generic excepton handler for a FPU exception
                error = do_exception( this , false );
            }
            else
            {
		        printk("\n[ERROR] in thread %x / unsupported coprocessor type\n",
		               this->trdid );
		        error = EXCP_USER_ERROR;
            }
        }
		break;

        case XCODE_OVR:   // user fatal error
        {
            printk("\n[ERROR] in thread %x / arithmetic overflow\n",
		           this->trdid );
		    error = EXCP_USER_ERROR;
        }
	    break;

        case XCODE_RI:   // user fatal error
        {
            printk("\n[ERROR] in thread %x / arithmetic overflow\n",
		           this->trdid );
		    error = EXCP_USER_ERROR;
        }
        break;

        case XCODE_ADEL: // user fatal error

        case XCODE_ADES:
        {
            printk("\n[ERROR] in thread %x / illegal address\n",
		           this->trdid );
		    error = EXCP_USER_ERROR;
        }
		break;

        default:
        {
            printk("\n[PANIC] in %s for thread %x / illegal XCODE value\n",
		           __FUNCTION__ , this->trdid );
		    error = EXCP_USER_ERROR;
        }
	}
    
    // analyse error code
	if( error == EXCP_USER_ERROR )      //  user error => kill the user process and return
	{
        hal_exception_dump( this , regs_tbl );
        sys_kill( this->process->pid , SIGKILL );
	}
    else if( error == EXCP_KERNEL_PANIC )   // kernel error => kernel panic
    {
        hal_exception_dump( this , regs_tbl );
        hal_core_sleep();
    }
}  // end hal_do_exception()

/////////////////////////////////////////
void hal_exception_dump( thread_t * this,
                         reg_t    * regs_tbl )
{
    // take the exception_lock located in io_cluster 
    remote_spinlock_lock( XPTR( LOCAL_CLUSTER->io_cxy , &exception_lock ) );

    // dump core registers values

    printk("\n====================================================================\n");
	printk("  thread %x / process %x / core %x / cycle %d\n",
           this->trdid , this->process->pid , this->core->gid , hal_get_cycles() );

	printk("  - Processor State:\n");
  
	printk("    CR:   %x\tEPC:  %x\tSR:   %x\tSP:    %x\n",
		   regs_tbl[UZ_CR], regs_tbl[UZ_EPC], regs_tbl[UZ_SR], regs_tbl[UZ_SP]);

    printk("    at_1  %x\tv0_2  %x\t\tv1_3  %x\ta0_4   %x\ta1_5   %x\n",
   	       regs_tbl[UZ_AT], regs_tbl[UZ_V0], regs_tbl[UZ_V1], regs_tbl[UZ_A0], regs_tbl[UZ_A1]);

    printk("    a2_6  %x\t\ta3_7  %x\tt0_8  %x\tt1_9   %x\tt2_10  %x\n",
		   regs_tbl[UZ_A2],regs_tbl[UZ_A3],regs_tbl[UZ_T0],regs_tbl[UZ_T1],regs_tbl[UZ_T2]);
  
    printk("    t3_11 %x\tt4_12 %x\t\tt5_13 %x\tt6_14  %x\tt7_15  %x\n",
		   regs_tbl[UZ_T3],regs_tbl[UZ_T4],regs_tbl[UZ_T5],regs_tbl[UZ_T6],regs_tbl[UZ_T7]);

    printk("    t8_24 %x\t\tt9_25 %x\tgp_28 %x\tc0_hi  %x\tc0_lo  %x\n",
		   regs_tbl[UZ_T8],regs_tbl[UZ_T9],regs_tbl[UZ_GP],regs_tbl[UZ_HI],regs_tbl[UZ_LO]);

    printk("    s0_16 %x\ts1_17 %x\ts2_18 %x\ts3_19  %x\ts4_20  %x\n",
		   regs_tbl[UZ_S0],regs_tbl[UZ_S1],regs_tbl[UZ_S2],regs_tbl[UZ_S3],regs_tbl[UZ_S4]);
  
    printk("    s5_21 %x\ts6_22 %x\t\ts7_23 %x\ts8_30  %x\tra_31  %x\n\n",
	       regs_tbl[UZ_S5],regs_tbl[UZ_S6],regs_tbl[UZ_S7],regs_tbl[UZ_S8],regs_tbl[UZ_RA]);

    printk("  - Thread State: %x\n"
           "    type = %s / local_locks = %d / remote_locks = %d / blocked = %x\n",
           thread_type_str( this->type ), this->local_locks, this->remote_locks,
           this->blocked );

    // release the exception_lock
    remote_spinlock_unlock( XPTR( LOCAL_CLUSTER->io_cxy , &exception_lock ) );

}  // end hal_exception_dump()

