/*
 * dev_pic.c - PIC (External Interrupt Controler) generic device API implementation.
 *
 * Authors   Alain Greiner  (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTPICLAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_special.h>
#include <chdev.h>
#include <printk.h>
#include <thread.h>
#include <hal_drivers.h>
#include <dev_pic.h>
#include <cluster.h>

/////////////////////////////////////////////////////////////////////////////////////////
// Extern global variables
/////////////////////////////////////////////////////////////////////////////////////////

extern chdev_directory_t  chdev_dir;         // allocated in kernel_init.c

///////////////////////////////////
void dev_pic_init( chdev_t  * pic )
{
    // get implementation
    uint32_t impl = pic->impl;

    // set chdev name
    strcpy( pic->name , "pic" );

    // call the implementation-specific PIC driver
    hal_drivers_pic_init(pic, impl);
}

/////////////////////////////////////////////////
void dev_pic_extend_init( uint32_t * lapic_base )
{
    // get pointer on PIC chdev
    chdev_t * pic_ptr = (chdev_t *)GET_PTR( chdev_dir.pic );
    cxy_t     pic_cxy = GET_CXY( chdev_dir.pic );

    // get pointer on extend_init function
    extend_init_t * f = hal_remote_lpt( XPTR( pic_cxy , &pic_ptr->ext.pic.extend_init ) ); 

    // call relevant driver function
    f( lapic_base );
}

/////////////////////////////////////
void dev_pic_bind_irq( lid_t     lid,
                       chdev_t * src_chdev )
{
    // get pointer on PIC chdev
    chdev_t * pic_ptr = (chdev_t *)GET_PTR( chdev_dir.pic );
    cxy_t     pic_cxy = GET_CXY( chdev_dir.pic );

    // get pointer on bind_irq function
    bind_irq_t * f = hal_remote_lpt( XPTR( pic_cxy , &pic_ptr->ext.pic.bind_irq ) );

    // call relevant driver function
    f( lid , src_chdev );
}

/////////////////////////////////////
void dev_pic_enable_irq( lid_t   lid,
                         xptr_t  src_chdev_xp )
{
    irq_dmsg("\n[INFO] %s : core = [%x,%d] / source_chdev_xp = %l\n",
             __FUNCTION__ , local_cxy , lid , src_chdev_xp );

    // get pointer on PIC chdev
    chdev_t * pic_ptr = (chdev_t *)GET_PTR( chdev_dir.pic );
    cxy_t     pic_cxy = GET_CXY( chdev_dir.pic );

    // get pointer on enable_irq function
    enable_irq_t * f = hal_remote_lpt( XPTR( pic_cxy , &pic_ptr->ext.pic.enable_irq ) );

    // call relevant driver function
    f( lid , src_chdev_xp );
}

//////////////////////////////////////
void dev_pic_disable_irq( lid_t   lid,
                          xptr_t  src_chdev_xp )
{
    irq_dmsg("\n[INFO] %s : core = [%x,%d] / source_chdev_xp = %l\n",
             __FUNCTION__ , local_cxy , lid , src_chdev_xp );

    // get pointer on PIC chdev
    chdev_t * pic_ptr = (chdev_t *)GET_PTR( chdev_dir.pic );
    cxy_t     pic_cxy = GET_CXY( chdev_dir.pic );

    // get pointer on disable_irq function
    disable_irq_t * f = hal_remote_lpt( XPTR( pic_cxy , &pic_ptr->ext.pic.disable_irq ) );

    // call relevant driver function
    f( lid , src_chdev_xp );
}

////////////////////////////////////////////
void dev_pic_enable_timer( uint32_t period )
{
    irq_dmsg("\n[INFO] %s : core = [%x,%d] / period = %d\n",
             __FUNCTION__ , local_cxy , CURRENT_THREAD->core->lid , period );

    // get pointer on PIC chdev
    chdev_t * pic_ptr = (chdev_t *)GET_PTR( chdev_dir.pic );
    cxy_t     pic_cxy = GET_CXY( chdev_dir.pic );

    // get pointer on enable_timer function
    enable_timer_t * f = hal_remote_lpt( XPTR( pic_cxy , &pic_ptr->ext.pic.enable_timer ) );

    // call relevant driver function
    f( period );
}

/////////////////////////
void dev_pic_enable_ipi()
{
    irq_dmsg("\n[INFO] %s : core = [%x,?]\n",
             __FUNCTION__ , local_cxy );

    // get pointer on PIC chdev
    chdev_t * pic_ptr = (chdev_t *)GET_PTR( chdev_dir.pic );
    cxy_t     pic_cxy = GET_CXY( chdev_dir.pic );

    // get pointer on enable_timer function
    enable_ipi_t * f = hal_remote_lpt( XPTR( pic_cxy , &pic_ptr->ext.pic.enable_ipi ) );

    // call relevant driver function
    f();
}

//////////////////////////////////
void dev_pic_send_ipi( cxy_t  cxy,
                       lid_t  lid )
{
    irq_dmsg("\n[INFO] %s : enter / src_core = [%x,%d] / dst_core = [%x,%d] / cycle = %d\n",
             __FUNCTION__, local_cxy, CURRENT_THREAD->core->lid, cxy, lid, hal_time_stamp() );

    // get pointer on PIC chdev
    chdev_t * pic_ptr = (chdev_t *)GET_PTR( chdev_dir.pic );
    cxy_t     pic_cxy = GET_CXY( chdev_dir.pic );

    // get pointer on send_ipi function
    send_ipi_t * f = hal_remote_lpt( XPTR( pic_cxy , &pic_ptr->ext.pic.send_ipi ) );

    // call relevant driver function
    f( cxy , lid );

    irq_dmsg("\n[INFO] %s : exit / src_core = [%x,%d] / dst_core = [%x,%d] / cycle = %d\n",
             __FUNCTION__, local_cxy, CURRENT_THREAD->core->lid, cxy, lid, hal_time_stamp() );
}

