/*
 * process.h - process related management functions
 *
 * Authors  Ghassan Almaless (2008,2009,2010,2011,2012)
 *          Mohamed Lamine Karaoui (2015)
 *          Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _PROCESS_H_
#define _PROCESS_H_

#include <kernel_config.h>
#include <errno.h>
#include <hal_types.h>
#include <list.h>
#include <xlist.h>
#include <bits.h>
#include <spinlock.h>
#include <hal_atomic.h>
#include <vmm.h>
#include <signal.h>
#include <cluster.h>
#include <vfs.h>

/****  Forward declarations  ****/

struct thread_s;

/*********************************************************************************************
 * These macros are used to compose or decompose global process identifier (PID)
 * to or from cluster identifier / local process index (CXY , LPID)
 ********************************************************************************************/

#define LPID_FROM_PID( pid )       (lpid_t)(pid & 0x0000FFFF)
#define CXY_FROM_PID( pid )        (cxy_t)(pid >> 16)
#define PID( cxy , lpid )          (pid_t)((cxy << 16) | lpid )

/*********************************************************************************************
 * This structure defines an array of extended pointers on the open file descriptors
 * for a given process. We use an extended pointer because the open file descriptor
 * is always stored in the same cluster as the inode associated to the file.
 * A free entry in this array contains the XPTR_NULL value.
 * The array size is defined by a the CONFIG_PROCESS_FILE_MAX_NR parameter.
 * All modifications (open/close) in this structure must be done by the reference cluster,
 * and reported in process copies.
 ********************************************************************************************/

typedef struct fd_array_s
{
	remote_spinlock_t lock;                               /*! lock protecting fd_array      */
    uint32_t          current;                            /*! current number of open files  */
	xptr_t            array[CONFIG_PROCESS_FILE_MAX_NR];  /*! xptr on open file descriptors */
}
fd_array_t;

/*********************************************************************************************
 * This structure defines a process descriptor.
 * A process is identified by a unique PID (process identifier):
 * - The PID 16 LSB bits contain the LPID (Local Process Index)
 * - The PID 16 MSB bits contain the owner cluster CXY.
 * In each cluster, the process manager allocates LPID values for the process that are
 * allocated to this cluster.
 * The process descriptor for a PID process is replicated in all clusters containing
 * at least one thread of the PID process, with the following rules :
 *
 * 1) The <pid>, <ppid>, <ref_xp>, <vfs_root_xp>, <vfs_bin_xp>  fields are defined
 *    in all process descriptor copies.
 * 2) The <vfs_cwd_xp> and associated <cwd_lock>, that can be dynamically modified,
 *    are only defined in the reference process descriptor.
 * 2) The <vmm>, containing the list of registered vsegs, and the page table, are only
 *    complete in the reference process cluster, other copies are read-only caches.
 * 3) the <fd_array>, containing extended pointers on the open file descriptors, is only
 *    complete in the reference process cluster, other copies are read-only caches.
 * 4) The <sem_root>, <mutex_root>, <barrier_root>, <condvar_root>, and the associated
 *    <sync_lock>, that are dynamically allocated, are only defined in the reference cluster.
 * 5) The <children_root>, and <children_nr> fields are only defined in the reference
 *    cluster, and are undefined in other clusters.
 * 6) The <brothers_list>, <local_list>, <copies_list>, <th_tbl>, <th_nr>, <th_lock> fields
 *    are defined in all process descriptors copies.
 * 7) The <sig_mgr> field is only defined in the reference cluster. TODO
 ********************************************************************************************/

typedef struct process_s
{
	vmm_t             vmm;              /*! embedded virtual memory manager                 */

	fd_array_t        fd_array;         /*! embedded open file descriptors array            */

	xptr_t            vfs_root_xp;      /*! extended pointer on current VFS root inode      */
	xptr_t            vfs_bin_xp;       /*! extended pointer on .elf file inode             */
	pid_t             pid;              /*! process identifier                              */
	pid_t             ppid;             /*! parent process identifier                       */
    xptr_t            ref_xp;           /*! extended pointer on reference process           */

	xptr_t            vfs_cwd_xp;       /*! extended pointer on current working dir inode   */
	remote_rwlock_t   cwd_lock;         /*! lock protecting working directory changes       */

	xlist_entry_t     children_root;    /*! root of the children process xlist              */
    uint32_t          children_nr;      /*! number of children processes                    */

	xlist_entry_t     brothers_list;    /*! member of list of children of same parent       */
    xlist_entry_t     local_list;       /*! member of list of process in same cluster       */
    xlist_entry_t     copies_list;      /*! member of list of copies of same process        */

	spinlock_t        th_lock;          /*! lock protecting th_tbl[] concurrent access      */
	uint32_t          th_nr;            /*! number of threads in this cluster               */
	struct thread_s * th_tbl[CONFIG_THREAD_MAX_PER_CLUSTER]; /*! pointers on local threads  */

    xlist_entry_t     sem_root;         /*! root of the process semaphore list              */
    xlist_entry_t     mutex_root;       /*! root of the process mutex list                  */
    xlist_entry_t     barrier_root;     /*! root of the process barrier list                */
    xlist_entry_t     condvar_root;     /*! root of the process condvar list                */

    remote_spinlock_t sync_lock;        /*! lock protecting sem,mutex,barrier,condvar lists */

	sig_mgr_t         sig_mgr;          /*! embedded signal manager TODO [AG]               */
}
process_t;

/*********************************************************************************************
 * This structure defines the information required by the process_make_exec() function
 * to create a new reference process descriptor, and the associated main thread.
 ********************************************************************************************/

typedef struct exec_info_s
{
    xptr_t             parent_xp;      /*! extended pointer on parent process descriptor    */

    char               path[CONFIG_VFS_MAX_PATH_LENGTH];   /*!  .elf file path              */

    char            ** args_pointers;  /*! physical base address of array of pointers       */
    char             * args_buf_base;  /*! physical base address of kernel args buffer      */
    uint32_t           args_nr;        /*! actual number of arguments                       */

    char            ** envs_pointers;  /*! physical base address of array of pointers       */
    char             * envs_buf_base;  /*! physical base address of kernel args buffer      */
    char             * envs_buf_free;  /*! physical address of first free slot in envs_buf  */
    uint32_t           envs_nr;        /*! actual number of environment variables           */
}
exec_info_t;

/***************   Process Descriptor Operations    *****************************************/

/*********************************************************************************************
 * This function allocates memory in local cluster for a process descriptor.
 *********************************************************************************************
 * @ returns pointer on process descriptor if success / return NULL if failure
 ********************************************************************************************/
process_t * process_alloc();

/*********************************************************************************************
 * This function releases memory in local cluster for a process descriptor.
 *********************************************************************************************
 * @ process      : pointer on process descriptor to release.
 ********************************************************************************************/
void process_free( process_t * process );

/*********************************************************************************************
 * This function allocates memory and initializes the "process_init" descriptor and the
 * associated "thread_init" descriptor. It should be called once at the end of the kernel
 * initialisation procedure, by the kernel "process_zero".
 * The "process_init" is the first user process, and all other user processes will be forked
 * from this process. The code executed by "process_init" is stored in a .elf file, whose
 * pathname is defined by the CONFIG_PROCESS_INIT_PATH argument. It uses fork/exec syscalls
 * to create the "shell" user process, and various other user daemon processes.
 * Practically, it builds the exec_info structure, registers the stdin / stdout / stderr
 * pseudo-file descriptors and the vfs_root and vfs_cwd in parent process_zero, and calls
 * the generic process_make_exec() function, that makes the real job.
 ********************************************************************************************/
void process_init_create();

/*********************************************************************************************
 * This function initializes a new process descriptor, in the reference cluster.
 * The PID value must have been defined previously by the owner cluster manager.
 * The reference cluster can be different from the owner cluster.
 * It set the pid / ppid / ref_xp fields.
 * It registers this process descriptor in three lists:
 * - the children_list in the parent reference process descriptor.
 * - the local_list, rooted in the reference cluster manager.
 * - the copies_list, rooted in the owner cluster manager.
 * It resets the embedded structures such as the VMM or the file descriptor array.
 *********************************************************************************************
 * @ process      : [in] pointer on process descriptor to initialize.
 * @ pid          : [in] process identifier defined by owner cluster.
 * @ parent_xp    : [in] extended pointer on parent process.
 ********************************************************************************************/
void process_reference_init( process_t * process,
                             pid_t       pid,
                             xptr_t      parent_xp );

/*********************************************************************************************
 * This function initializes a copy process descriptor, in the local cluster,
 * from information defined in the reference remote process descriptor.
 *********************************************************************************************
 * @ process              : [in] local pointer on process descriptor to initialize.
 * @ reference_process_xp : [in] extended pointer on reference process descriptor.
 * @ return 0 if success / return ENOMEM if failure
 ********************************************************************************************/
error_t process_copy_init( process_t * local_process,
                           xptr_t      reference_process_xp );

/*********************************************************************************************
 * This function releases all memory allocated for a process descriptor in the local cluster,
 * including memory allocated for embedded substructures (fd_array, vmm, etc).
 * The local th_tbl[] array must be empty.
 *********************************************************************************************
 * @ process     : pointer on the process descriptor.
 ********************************************************************************************/
void process_destroy( process_t * process );

/*********************************************************************************************
 * This function kills a user process in a given cluster.
 * It can be directly called in the reference cluster, or it can be called through the
 * PROCESS_KILL RPC.
 * - In a first loop, it set the THREAD_SIG_KILL signal to all threads of process.
 * - In a second loop, it wait, for each thread the reset of the THREAD_SIG_KILL signal
 *   by the scheduler, and completes the thread descriptor destruction.
 *********************************************************************************************
 * @ process     : pointer on the process descriptor.
 ********************************************************************************************/
void process_kill( process_t * process );

/*********************************************************************************************
 * This function returns a pointer on the local copy of a process identified by its PID.
 * If this local copy does not exist yet, it is dynamically created, from the reference
 * process descriptor, registered in the global copies_list, and registered in the local_list.
 * This function is used by the thread_user_create() function.
 *********************************************************************************************
 * @ pid     : searched process identifier.
 * @ returns pointer on the local process descriptor if success / returns NULL if failure.
 ********************************************************************************************/
process_t * process_get_local_copy( pid_t pid );

/*********************************************************************************************
 * This function allocates memory and initializes a new user process descriptor,
 * and the associated main thread, from information found in the <exec_info> structure
 * (defined in the process.h file), that must be built by the caller.
 * The new process inherits from the parent process (i) the open file descriptors, (ii) the
 * vfs_root and the vfs_cwd inodes.
 * It accesses the .elf file to get the size of the code and data segments, and initializes
 * the vsegs list in the VMM.
 * It is executed in the local cluster, that becomes both "owner" and "reference".
 * - It can be called by the process_init_create() function to build the "init" process.
 * - It can be called directly by the sys_exec() function in case of local exec.
 * - It can be called through the rpc_process_exec_server() function in case of remote exec.
 *********************************************************************************************
 * @ exec_info   : [in]  pointer on the exec_info structure.
 * @ return 0 if success / return non-zero if error.
 ********************************************************************************************/
error_t process_make_exec( exec_info_t * exec_info );


/********************   Signal Management Operations   **************************************/

/*********************************************************************************************
 * This function TODO [AG]
 ********************************************************************************************/
void process_signal_handler( process_t * process );


/********************   File Management Operations   ****************************************/

/*********************************************************************************************
 * This function initializes all entries of the local fd_array as empty.
 *********************************************************************************************
 * @ process  : pointer on the local process descriptor.
 ********************************************************************************************/
void process_fd_init( process_t * process );

/*********************************************************************************************
 * This function uses as many remote accesses as required, to reset an entry in fd_array[],
 * in all clusters containing a copy. The entry is identified by the <file_id> argument.
 * This function must be executed by a thread running reference cluster, that contains
 * the complete list of process descriptors copies.
 *********************************************************************************************
 * @ process  : pointer on the local process descriptor.
 * @ file_id  : file descriptor index in the fd_array.
 ********************************************************************************************/
void process_fd_remove( process_t * process,
                        uint32_t    file_id );

/*********************************************************************************************
 * This function returns an extended pointer on a file descriptor identified by its index
 * in fd_array. It can be called by any thread running in any cluster.
 * It accesses first the local process descriptor. In case of local miss, it uses remote
 * access to access the reference process descriptor.
 * It updates the local fd_array when the file descriptor exists in reference cluster.
 * The file descriptor refcount is not incremented.
 *********************************************************************************************
 * @ process  : pointer on the local process descriptor.
 * @ file_id  : file descriptor index in the fd_array.
 * @ return extended pointer on file descriptor if success / return XPTR_NULL if not found.
 ********************************************************************************************/
xptr_t process_fd_get_xptr( process_t * process,
                            uint32_t    file_id );

/*********************************************************************************************
 * This function checks the number of open files for a given process.
 * It can be called by any thread in any cluster, because it uses portable remote access
 * primitives to access the reference process descriptor.
 *********************************************************************************************
 * @ returns true if file descriptor array full.
 ********************************************************************************************/
bool_t process_fd_array_full();

/*********************************************************************************************
 * This function allocates a free slot in the fd_array of the reference process,
 * register the <file_xp> argument in the allocated slot, and return the slot index.
 * It can be called by any thread in any cluster, because it uses portable remote access
 * primitives to access the reference process descriptor.
 *********************************************************************************************
 * @ file_xp  : extended pointer on the file descriptor to be registered.
 * @ file_id  : [out] buffer for fd_array slot index.
 * @ return 0 if success / return EMFILE if array full.
 ********************************************************************************************/
error_t process_fd_register( xptr_t      file_xp,
                             uint32_t  * file_id );

/*********************************************************************************************
 * This function copies all non-zero entries from a remote <src_xp> fd_array,
 * embedded in a process descriptor, to another remote <dst_xp> fd_array, embedded
 * in another process descriptor. The calling thread can be running in any cluster.
 * It takes the remote lock protecting the <src_xp> fd_array during the copy.
 * For each involved file descriptor, the refcount is incremented.
 *********************************************************************************************
 * @ dst_xp   : extended pointer on the destination fd_array_t.
 * @ src_xp   : extended pointer on the source fd_array_t.
 ********************************************************************************************/
void process_fd_remote_copy( xptr_t dst_xp,
                             xptr_t src_xp );



/********************   Thread Related Operations   *****************************************/

/*********************************************************************************************
 * This function registers a new thread in the local process descriptor.
 * It checks that there is an available slot in the local th_tbl[] array,
 * allocates a new LTID, and registers the new thread in the th_tbl[].
 * WARNING : the lock protecting the th_tbl[] must be taken by the caller.
 *********************************************************************************************
 * @ process  : pointer on the local process descriptor.
 * @ thread   : pointer on new thread to be registered.
 * @ trdid    : [out] address of buffer for allocated trdid.
 * @ returns 0 if success / returns non zero if no slot available.
 ********************************************************************************************/
error_t process_register_thread( process_t       * process,
                                 struct thread_s * thread,
                                 trdid_t         * trdid );

/*********************************************************************************************
 * This function removes a thread registration from the local process descriptor.
 * WARNING : the lock protecting the th_tbl[] must be taken by the caller.
 *********************************************************************************************
 * @ thread   : local pointer on thread to be removed.
 ********************************************************************************************/
void process_remove_thread( struct thread_s * thread );



#endif	/* _PROCESS_H_ */
