/*
 * rwlock.c - kernel read/write lock synchronization.
 * 
 * Author  Alain Greiner     (2016}
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_types.h>
#include <hal_atomic.h>
#include <hal_special.h>
#include <hal_irqmask.h>
#include <thread.h>
#include <printk.h>
#include <rwlock.h>

///////////////////////////////////////
void rwlock_init( rwlock_t * lock )
{  
	lock->ticket  = 0;
    lock->current = 0;
    lock->count   = 0;
	lock->owner   = NULL;
}

//////////////////////////////////////
void rwlock_rd_lock( rwlock_t * lock )
{
	reg_t               mode;
	uint32_t            ticket;
	thread_t          * this = CURRENT_THREAD;

    // disable IRQs
	hal_disable_irq( &mode );

    // get next free ticket
    ticket = hal_atomic_add( &lock->ticket , 1 );
  
    // poll the current ticket value
	while( lock->current != ticket )
    {
        hal_fixed_delay( CONFIG_RWLOCK_DELAY );
    }

    ////////// From here we have the lock  ////////////

    // increment number of readers
    lock->count++;
    this->local_locks++;

    // consistency
    hal_fence();

    // release the lock to allow several simultaneous readers
    lock->current++;

    // enable IRQs
	hal_restore_irq( mode );

}  // end  rwlock_rd_lock()

////////////////////////////////////////
void rwlock_rd_unlock( rwlock_t * lock )
{
    reg_t      mode;
	thread_t * this = CURRENT_THREAD;

    // disable IRQs
	hal_disable_irq( &mode );
  
    // decrement number of readers
    hal_atomic_add( &lock->count , -1 );
    this->local_locks--;

    // enable IRQs
	hal_restore_irq( mode );
}

//////////////////////////////////////
void rwlock_wr_lock( rwlock_t * lock )
{
	reg_t              mode;
    uint32_t           ticket;
	thread_t         * this = CURRENT_THREAD;

    // disable IRQs
	hal_disable_irq( &mode );
  
    // get next free ticket
    ticket = hal_atomic_add( &lock->ticket , 1 );
  
    // poll the current ticket value
	while( lock->current != ticket )
    {
        hal_fixed_delay( CONFIG_RWLOCK_DELAY );
    } 

    ////////// From here we have the lock  ////////////

    // wait completion of existing read access
    while( lock->count != 0 )
    {
        hal_fixed_delay( CONFIG_RWLOCK_DELAY );
    }

    lock->owner = this;
    this->local_locks++;

    // enable IRQs
	hal_restore_irq( mode );

}  // end rwlock_wr_lock()

////////////////////////////////////////////
void rwlock_wr_unlock( rwlock_t * lock )
{
    reg_t      mode;
	thread_t * this = CURRENT_THREAD;

    // disable IRQs
	hal_disable_irq( &mode );
  
    // release lock
    lock->current++;
    lock->owner = NULL;
    this->local_locks--;
    
    // enable IRQs
	hal_restore_irq( mode );
}

