/*
 * drvdb.c - Dictionary of supported devices and associated drivers
 *
 * Authors Ghassan Almaless (2008,2009,2010,2011,2012)
 *         Alain Greiner (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH.is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH.is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH.; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <types.h>
#include <errno.h>
#include <libk.h>
#include <event.h>
#include <device.h>
#include <driver.h>
#include <drvdb.h>
#include <soclib_xicu.h>
#include <soclib_icu.h>
#include <soclib_tty.h>
#include <soclib_timer.h>
#include <soclib_dma.h>
#include <soclib_block.h>
#include <soclib_fb.h>
#include <soclib_memc.h>
#include <soclib_iopic.h>


///////////////////////////////////////////////////////////////////////////////////////////
// This global variable defines the array of supported devices and associated drivers.
// It must be kept consistant with the enum in file drvdb.h.
///////////////////////////////////////////////////////////////////////////////////////////

static drvdb_entry_t drivers_db[DEV_TYPE_DRVID_NR] = 
{
	{DEV_TYPE_RAM     , "Random Access Memory"           , NULL},
	{DEV_TYPE_TTY     , "Text Terminal Controler"        , &soclib_tty_driver},
    {DEV_TYPE_FBF     , "Frame Buffer Controler"         , &soclib_fbf_driver},
    {DEV_TYPE_IOB     , "I/O Bridge Component"           , &soclib_iob_driver},
	{DEV_TYPE_IOC_BDV , "BDV Block Device Controler"     , &soclib_bdv_driver},
	{DEV_TYPE_IOC_HBA , "HBA Block Device Controler"     , NULL},
    {DEV_TYPE_IOC_SDC , "SDC Block Device Controler"     , NULL},
    {DEV_TYPE_IOC_SPI , "SPI Block Device Controler"     , NULL},
    {DEV_TYPE_IOC_RDK , "RDK Block device Controler"     , NULL},
	{DEV_TYPE_MMC     , "L2 cache Configuration"         , &soclib_mmc_driver},
	{DEV_TYPE_MWR_DMA , "DMA Coprocessor"                , NULL},
	{DEV_TYPE_MWR_GCD , "GCD Coprocessor"                , NULL},
	{DEV_TYPE_MWR_DCT , "DCT Coprocessor"                , NULL},
	{DEV_TYPE_NIC     , "GMII-Compliant NIC Controller"  , NULL},
	{DEV_TYPE_ROM     , "Read Only Memory"               , NULL},
	{DEV_TYPE_CMA     , "Chained buffers DMA Controller" , NULL},
	{DEV_TYPE_TIM     , "Real-Time Timer"                , &soclib_timer_driver},
	{DEV_TYPE_XCU     , "Interrupt Controlier unit"      , &soclib_xicu_driver},
	{DEV_TYPE_PIC     , "HWI to WTI translator"          , &soclib_iopic_driver},
	{DEV_TYPE_NULL    , "End of Table Marker           " , NULL}
};


///////////////////////////////////////////////////
drvdb_entry_t * drvdb_locate_byId( uint32_t drvid )
{
	if(drvid >= DEV_TYPE_RESERVED)
		return NULL;

	return &drivers_db[drvid];
}

//////////////////////////////////////////////
drvdb_entry_t * drvdb_locate_byName( char * name )
{
	register uint32_t i;

	for(i=0; drivers_db[i].id != DEV_TYPE_RESERVED; i++)
		if(!strcmp(drivers_db[i].name, name))
			return &drivers_db[i];

	return NULL;
}

driver_t* drvdb_get_driver(uint32_t drvid)
{
	if(drvid >= DEV_TYPE_RESERVED)
		return NULL;

	return drivers_db[drvid].driver;
}

error_t drvdb_set_driver(uint32_t drvid, driver_t *driver)
{
	if(drvid >= DEV_TYPE_RESERVED)
		return ERANGE;

	drivers_db[drvid].driver = driver;
	return 0;
}
