/*
 * mapper.c - Map memory, file or device in process virtual address space.
 *
 * Authors   Mohamed Lamine Karaoui (2015)
 *           Alain Greiner (2016)
 *
 * Copyright (c)  UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <almos_config.h>
#include <hal_types.h>
#include <hal_special.h>
#include <grdxt.h>
#include <rwlock.h>
#include <printk.h>
#include <thread.h>
#include <core.h>
#include <process.h>
#include <kmem.h>
#include <kcm.h>
#include <page.h>
#include <cluster.h>
#include <vfs.h>
#include <mapper.h>

//////////////////////////
mapper_t * mapper_create()
{
    mapper_t * mapper;
    kmem_req_t req;
    error_t    error;

    // allocate memory for associated mapper
	req.type  = KMEM_MAPPER;
	req.size  = sizeof(mapper_t);
    req.flags = AF_KERNEL | AF_ZERO;
	mapper    = (mapper_t *)kmem_alloc( &req );

    if( mapper == NULL )
    {
        printk("\n[ERROR] in %s : no memory for mapper descriptor\n", __FUNCTION__ );
        return NULL;
    }

    // initialize refcount & inode
	mapper->refcount = 0;
    mapper->inode    = NULL;

    // initialize radix tree
	error = grdxt_init( &mapper->radix,
                        CONFIG_VMM_GRDXT_W1,
                        CONFIG_VMM_GRDXT_W2,
                        CONFIG_VMM_GRDXT_W3 );

    if( error )
    {
        printk("\n[ERROR] in %s : cannot initialize radix tree\n", __FUNCTION__ );
	    req.type  = KMEM_MAPPER;
        req.ptr   = mapper;
        kmem_free( &req );
        return NULL;
    }

    // initialize mapper lock
	rwlock_init(  &mapper->lock );

    // initialize waiting threads xlist (empty)
	xlist_root_init( XPTR( local_cxy , &mapper->wait_root ) );

    // initialize vsegs xlist (empty)
	xlist_root_init( XPTR( local_cxy , &mapper->vsegs_root ) );

    return mapper;

} // end mapper_create()

///////////////////////////////////////////
error_t mapper_destroy( mapper_t * mapper )
{
    page_t   * page;
    uint32_t   found_index = 0;
    uint32_t   start_index = 0;
    kmem_req_t req;
    error_t    error;

    // scan radix three and release all registered pages to PPM
    do
    {
        // get page from radix tree
        page = (page_t *)grdxt_get_first( &mapper->radix , start_index , &found_index );

        if( page != NULL ) 
        {
            // remove page from mapper and release to PPM
	        error = mapper_release_page( mapper , found_index , page );

            if ( error ) return error;

            // update start_key value for next page
            start_index = found_index;
        }
    }
    while( page != NULL );

    // release the memory allocated to radix-tree itself
    grdxt_destroy( &mapper->radix );

    // release memory for mapper descriptor
    req.type = KMEM_MAPPER;
    req.ptr  = mapper;
    kmem_free( &req );

    return 0;
 
}  // end mapper_destroy()


////////////////////////////////////////////
page_t * mapper_get_page( mapper_t * mapper,
                          uint32_t   index )
{
	kmem_req_t    req;
	page_t      * page;
	error_t       error;

    thread_t * this = CURRENT_THREAD;

    // take mapper lock in READ_MODE
    rwlock_rd_lock( &mapper->lock );

    // search page in radix tree
    page = (page_t *)grdxt_lookup( &mapper->radix , index );

    // test if page available in mapper 
    if( ( page == NULL) || page_is_flag( page , PG_INLOAD ) )  // page not available            /
    {
        // release the lock in READ_MODE and take it in WRITE_MODE
        rwlock_rd_unlock( &mapper->lock );
        rwlock_wr_lock( &mapper->lock );

        // second test on missing page because the page status can have been modified
        // by another thread, when passing from READ_MODE to WRITE_MODE.
        // from this point there is no concurrent accesses to mapper.

        page = grdxt_lookup( &mapper->radix , index );

        if ( page == NULL )   // missing page => load it from file system
        {
            // allocate one page from PPM
            req.type  = KMEM_PAGE;
            req.size  = 0;
            req.flags = AF_NONE;
            page = kmem_alloc( &req );
      
            if( page == NULL )
            {
                printk("\n[ERROR] in %s : thread %x cannot allocate a page in cluster %x\n",
                       __FUNCTION__ , this->trdid , local_cxy );
                rwlock_wr_unlock( &mapper->lock );
                return NULL;
            }

            // initialize the page descriptor
            page_init( page );
            page_set_flag( page , PG_INIT );
            page_set_flag( page , PG_INLOAD );
            page_refcount_up( page );
            page->mapper = mapper;
            page->index  = index;

            // insert page in mapper radix tree
            error = grdxt_insert( &mapper->radix, index , page );

            // release mapper lock from WRITE_MODE
            rwlock_wr_unlock( &mapper->lock );

            if( error ) 
            {
                printk("\n[ERROR] in %s : thread %x cannot insert page in mapper\n",
                       __FUNCTION__ , this->trdid );
                mapper_release_page( mapper , index , page );
                page_clear_flag( page , PG_ALL );
                req.ptr  = page;
                req.type = KMEM_PAGE;
                kmem_free(&req);
                return NULL;
            }
        
            // launch I/O operation to load page from file system
            error = mapper_updt_page( mapper , index , page );

            if( error )
            {
                printk("\n[ERROR] in %s : thread %x cannot load page from device\n",
                       __FUNCTION__ , this->trdid );
                mapper_release_page( mapper , index , page );
                page_clear_flag( page , PG_ALL );
                req.ptr  = page;
                req.type = KMEM_PAGE;
                kmem_free( &req );
                return NULL;
            }

            // update the mapper and index fields in page descriptor
            page->mapper = mapper;
            page->index  = index;

            // reset the page INLOAD flag to make the page available to all readers
            page_clear_flag( page , PG_INLOAD );

        }
        else if( page_is_flag( page , PG_INLOAD ) )   // page is loaded by another thread
        {
            // release mapper lock from WRITE_MODE
            rwlock_wr_unlock( &mapper->lock );

            // deschedule to wait load completion
            while( 1 )
            {
                // exit waiting loop when loaded
                if(  page_is_flag( page , PG_INLOAD ) ) break;

                // deschedule 
                sched_yield();
            }
         
        }

        return page;
    }
    else 
    {
         // release lock from READ_MODE
         rwlock_rd_unlock( &mapper->lock );

         return page;
    }
}  // end mapper_get_page()

///////////////////////////////////////////////
error_t mapper_release_page( mapper_t * mapper,
                             uint32_t   index,
                             page_t   * page )
{
    error_t error;

    // lauch IO operation to update page to file system
    error = mapper_sync_page( mapper , index , page );

    if( error )
    {
        printk("\n[ERROR] in %s : cannot update file system\n", __FUNCTION__ );
        return EIO;
    }
        
    // take mapper lock in WRITE_MODE
    rwlock_wr_lock( &mapper->lock );

    // remove physical page from radix tree
	grdxt_remove( &mapper->radix , page->index );

    // release mapper lock from WRITE_MODE
    rwlock_wr_unlock( &mapper->lock );

    // release page to PPM
	kmem_req_t   req;
	req.type  = KMEM_PAGE;
    req.ptr   = page;
    kmem_free( &req );

    return 0;

}  // end mapper_release_page()

////////////////////////////////////////////
error_t mapper_updt_page( mapper_t * mapper,
                          uint32_t   index,
                          page_t   * page )
{
    uint32_t      type;
    vfs_inode_t * inode;
    error_t       error = 0;

    if( page == NULL )
    {
        printk("\n[PANIC] in %s : page pointer is NULL\n", __FUNCTION__ );
        hal_core_sleep();
    }

    if( mapper == NULL )
    {
        printk("\n[PANIC] in %s : no mapper for this page\n", __FUNCTION__ );
        hal_core_sleep();
    }

    // get file system type and inode pointer
    inode = mapper->inode;
    type  = inode->ctx->type;

    // get page lock
    page_lock( page );

    // get mapper lock in WRITE_MODE
    rwlock_wr_lock( &mapper->lock );

    // call proper I/O operation to update file system
    if     ( type == FS_TYPE_FATFS ) error = fatfs_read_page( page );
    else if( type == FS_TYPE_RAMFS ) error = ramfs_read_page( page );
    else
    {
        printk("\n[PANIC] in %s : undefined file system type\n", __FUNCTION__ );
        hal_core_sleep();
    }

    // release mapper lock from WRITE_MODE
    rwlock_wr_unlock( &mapper->lock );

    // release page lock 
    page_unlock( page );

    if( error )
    {
        printk("\n[PANIC] in %s : cannot access file system\n", __FUNCTION__ );
        return EIO;
    }
	
    return 0;
}  // end mapper_updt_page

////////////////////////////////////////////
error_t mapper_sync_page( mapper_t * mapper,
                          uint32_t   index,
                          page_t   * page )
{
    uint32_t      type;
    vfs_inode_t * inode;
    error_t       error = 0;

    if( page == NULL )
    {
        printk("\n[PANIC] in %s : page pointer is NULL\n", __FUNCTION__ );
        hal_core_sleep();
    }

    if( mapper == NULL )
    {
        printk("\n[PANIC] in %s : no mapper for this page\n", __FUNCTION__ );
        hal_core_sleep();
    }

	if( page_is_flag( page , PG_DIRTY ) ) 
	{
        // get file system type and inode pointer
        inode = mapper->inode;
        type  = inode->ctx->type;

        // get page lock
        page_lock( page );

        // get mapper lock in READ_MODE
        rwlock_rd_lock( &mapper->lock );

        // call proper I/O operation to update file system
        if     ( type == FS_TYPE_FATFS ) error = fatfs_write_page( page );
        else if( type == FS_TYPE_RAMFS ) error = ramfs_write_page( page );
        else
        {
            printk("\n[PANIC] in %s : undefined file system type\n", __FUNCTION__ );
            hal_core_sleep();
        }

        // release mapper lock from READ_MODE
        rwlock_rd_unlock( &mapper->lock );

        // release page lock 
        page_unlock( page );

        if( error )
        {
            printk("\n[PANIC] in %s : cannot update file system\n", __FUNCTION__ );
            return EIO;
        }
    
        // clear dirty bit
		page_undo_dirty( page );
     }
	
    return 0;

}  // end mapper_sync_page()

///////////////////////////////////////////////////////////////////////////////////////
// This static function is called by the mapper_move fragments() function.
// It moves one fragment between an user buffer and the kernel mapper.
// Implementation Note: It can require access to one or two pages in mapper: 
//  [max_page_index == min_page_index]     <=>  fragment fit in one mapper page
//  [max_page index == min_page_index + 1] <=>  fragment spread on two mapper pages
///////////////////////////////////////////////////////////////////////////////////////
static error_t mapper_move_one_fragment( mapper_t   * mapper,
                                         bool_t       to_buffer,
                                         fragment_t * fragment )
{
    uint32_t   size;                 // number of bytes in fragment 
    cxy_t      buf_cxy;              // cluster identifier for user buffer
    uint8_t  * buf_ptr;              // local pointer on first byte in user buffer
    
    xptr_t     xp_buf;               // extended pointer on byte in user buffer
    xptr_t     xp_map;               // extended pointer on byte in kernel mapper

    uint32_t   min_file_offset;      // offset of first byte in file 
    uint32_t   max_file_offset;      // offset of last byte in file    

    uint32_t   first_page_index;     // index of first page in mapper
    uint32_t   first_page_offset;    // offset of first byte in first page in mapper    
    uint32_t   first_page_size;      // offset of first byte in first page in mapper    

    uint32_t   second_page_index;    // index of last page in mapper    
    uint32_t   second_page_offset;   // offset of last byte in last page in mapper    
    uint32_t   second_page_size;     // offset of last byte in last page in mapper    
 
    page_t   * page;                 // pointer on one page descriptor in mapper
    uint8_t  * map_ptr;              // local pointer on first byte in mapper

    // get fragment attributes in user buffer
    buf_cxy = fragment->buf_cxy;
    buf_ptr = fragment->buf_ptr;
    size    = fragment->size;

    if( size > CONFIG_PPM_PAGE_SIZE )
    {
        printk("\n[PANIC] in %s : illegal fragment size = %d\n",
               __FUNCTION__ , size );
        return EINVAL;
    }
    
    // compute offsets of first and last bytes in file
    min_file_offset = fragment->file_offset;
    max_file_offset = min_file_offset + size;

    // compute indexes of pages for first and last byte in mapper
    first_page_index  = min_file_offset >> CONFIG_PPM_PAGE_SHIFT;
    second_page_index = max_file_offset >> CONFIG_PPM_PAGE_SHIFT;

    if ( first_page_index == second_page_index )  // only one page in mapper
    {
        // compute offset and size for page in mapper 
        first_page_offset = min_file_offset & (1<<CONFIG_PPM_PAGE_SHIFT);
        first_page_size   = size;

        // get pointer on first page in mapper
        page = mapper_get_page( mapper , first_page_index );

        if ( page == NULL ) return EINVAL;

        // compute pointer on fragment first byte in mapper
        map_ptr = (uint8_t *)ppm_page2base( page ) + first_page_offset;

        // compute extended pointers in mapper and in buffer
        xp_map = XPTR( local_cxy , map_ptr );
        xp_buf = XPTR( buf_cxy , buf_ptr );

        // move fragment
        if( to_buffer )
        {
            hal_remote_memcpy( xp_buf , xp_map , first_page_size );
        }
        else
        {
            page_do_dirty( page );
            hal_remote_memcpy( xp_map , xp_buf , first_page_size );
        }
    }
    else                                        // two pages in mapper
    {
        // compute offset and size for first page in mapper
        first_page_offset = min_file_offset & (1<<CONFIG_PPM_PAGE_SHIFT);
        first_page_size   = CONFIG_PPM_PAGE_SIZE - first_page_offset;

        // get pointer on first page descriptor in mapper
        page = mapper_get_page( mapper , first_page_index );

        if ( page == NULL ) return EINVAL;

        // compute local pointer on first byte in first page in mapper
        map_ptr = (uint8_t *)ppm_page2base(page) + first_page_offset;
    
        // compute extended pointers
        xp_map = XPTR( local_cxy , map_ptr );
        xp_buf = XPTR( buf_cxy , buf_ptr );

        // move fragment to/from first page
        if( to_buffer )
        {
            hal_remote_memcpy( xp_buf , xp_map , first_page_size );
        }
        else
        {
            page_do_dirty( page );
            hal_remote_memcpy( xp_map , xp_buf , first_page_size );
        }

        // compute offset and size for second page in mapper
        second_page_offset = 0;
        second_page_size   = size - first_page_size;  

        // get pointer on second page in mapper
        page = mapper_get_page( mapper , second_page_index );

        if ( page == NULL ) return EINVAL;

        // compute local pointer on first byte in second page in mapper
        map_ptr = (uint8_t *)ppm_page2base( page ) + second_page_offset;
    
        // compute extended pointers
        xp_map = XPTR( local_cxy , map_ptr );
        xp_buf = XPTR( buf_cxy , buf_ptr + first_page_offset );

        // move fragment to/from second page
        if( to_buffer )
        {
            hal_remote_memcpy( xp_buf , xp_map , second_page_size );
        }
        else
        {
            page_do_dirty( page );
            hal_remote_memcpy( xp_map , xp_buf , second_page_size );
        }
    }

    return 0;
}  // end mapper_move_one_fragment()

/////////////////////////////////////////////////
error_t mapper_move_fragments( mapper_t * mapper,
                               bool_t     read,
                               uint32_t   nb_frags,
                               xptr_t     xp_frags )
{
	uint32_t     index;
	error_t      error;
    fragment_t   local_frags[CONFIG_MAPPER_MAX_FRAGMENTS];   // local copy of fragments array
    fragment_t * frags_array;                                // pointer on fragments array

    // check nb_frags
    if( nb_frags > CONFIG_MAPPER_MAX_FRAGMENTS )
    {
        printk("\n[PANIC] in %s : number of fragments cannot be larger than %d\n",
               __FUNCTION__ , CONFIG_MAPPER_MAX_FRAGMENTS );
        return EINVAL;
    }

    // get client cluster and local pointer on fragments array
    cxy_t        client_cxy   = GET_CXY( xp_frags );
    fragment_t * client_frags = (fragment_t *)GET_PTR( xp_frags );

    if ( local_cxy == client_cxy ) // use the local fragments array if possible
    {
        frags_array = client_frags;
    }
    else                           // make a local copy of fragments array 
    {
        hal_remote_memcpy( XPTR( local_cxy , local_frags ) , xp_frags ,
                           sizeof(fragment_t) * nb_frags );
        frags_array = local_frags;
    }

    // loop on fragments
    for( index = 0 ; index < nb_frags ; index ++ )
    {
        error = mapper_move_one_fragment( mapper , read , &frags_array[index] );
        if ( error ) return error;
    }

    return 0;

}  // end mapper_move_fragments()


