/*
 * hal_types.h - common kernel types for I86 64 bits
 *
 * Author    Alain Greiner (2016)           
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef HAL_TYPES_H_
#define HAL_TYPES_H_

#include <almos_config.h>
#include <stdint.h>

#ifndef NULL
#define NULL (void*)        0
#endif

/**************************************************************************
 *                      Exact-width integer types.  
 **************************************************************************/

typedef   signed           char     int8_t;
typedef unsigned           char    uint8_t;

typedef   signed           int      int16_t;
typedef unsigned           int     uint16_t;

typedef   signed long      int      int32_t;
typedef unsigned long      int     uint32_t;

typedef   signed long long int      int64_t;
typedef unsigned long long int     uint64_t;

/***************************************************************************
 * Pthread related types 
 **************************************************************************/

typedef uint32_t      pthread_t;
typedef uint32_t      pthread_mutexattr_t;
typedef uint32_t      pthread_barrier_t;
typedef uint32_t      pthread_barrierattr_t;
typedef uint32_t      sem_t;
typedef uint32_t      pthread_cond_t;
typedef uint32_t      pthread_condattr_t;
typedef uint32_t      pthread_rwlock_t;
typedef uint32_t      pthread_rwlockattr_t;
typedef uint32_t      pthread_key_t;

/***************************************************************************
 * Boolean type and macros
 **************************************************************************/

typedef uint32_t      bool_t;

#define false         0
#define true          1

/***************************************************************************
 * Kernel error type : signed
 **************************************************************************/

typedef int32_t       error_t;

/***************************************************************************
 * Time related types
 **************************************************************************/

typedef uint64_t      clock_t;

/***************************************************************************
 * Processus identifier type / fixed format 
 * 16 MSB bits = cluster XY coordinates
 * 16 LSB bits = local process identifier
 **************************************************************************/

typedef uint32_t      pid_t;   

/***************************************************************************
 * Local Process index  
 **************************************************************************/

typedef uint16_t      lpid_t;

/***************************************************************************
 * Thread  identifier type / fixed format 
 * 16 MSB = cluster XY identifier
 * 16 LSB = local thread index
 **************************************************************************/

typedef uint32_t      trdid_t;

/***************************************************************************
 * Local Thread index   
 **************************************************************************/

typedef uint16_t      ltid_t;

/***************************************************************************
 * User identifier type
 **************************************************************************/

typedef uint32_t      uid_t;

/***************************************************************************
 * CPU identifier types
 **************************************************************************/

typedef uint32_t      lid_t;     // local index in cluster
typedef uint32_t      gid_t;     // global identifier

/***************************************************************************
 * File Descriptor Index in File Descriptor Array
 **************************************************************************/

typedef uint32_t      fdid_t;

/***************************************************************************
 * This structure defines a single 32 bits integer alone in a cache line. 
 **************************************************************************/

typedef struct cache_line_s
{
  union
  {
    uint32_t values[CONFIG_CACHE_LINE_LENGTH];
    uint32_t value;
  };
}
__attribute__((packed)) cacheline_t;

#define CACHELINE_ALIGNED __attribute__((aligned(CONFIG_CACHE_LINE_SIZE)))

/***************************************************************************
 *  Address types and macros        !!! hardware dependant !!!
 ***************************************************************************
 * An extended pointer is a 64 bits integer, structured in two fields :
 * - cxy : cluster identifier.
 * - ptr : pointer in the virtual space of a single cluster.
 *
 * In Intel 64 bits, the kernel virtual space has 256 Tbytes per cluster
 * - the cxy field occupies bits[63:48]
 * - the ptr field occupies bits[47:0]
 ***************************************************************************
 * A physical address is a 64 bits integer, structured in two fields :
 * - cxy : cluster identifier.
 * - lpa : local physical address inside cluster.
 *
 * In INTEL 64 bits, the physical space has 256 Gbytes per cluster.
 * - the cxy field occupies bits[63:48]
 * - the lpa field occupies bits[47:0]
 **************************************************************************/

typedef uint64_t               reg_t;         // core register

typedef uint64_t               xptr_t;         // extended pointer

typedef uint16_t               cxy_t;          // cluster identifier

typedef uint64_t               paddr_t;        // global physical address (64 bits)

typedef uint64_t               lpa_t;          // local physical address

typedef uint64_t               intptr_t;       // local pointer stored as integer

typedef uint64_t               ppn_t;          // Physical Page Number

typedef uint64_t               vpn_t;          // Virtual Page number

#define XPTR_NULL              0

#define PTR_MASK               ((1ULL)<<47)

#define GET_CXY(xp)            ((cxy_t)((xp) >> 48))

#define GET_PTR(xp)            ((void*)((xp) & PTR_MASK))

#define XPTR(cxy,ptr)          (((uint64_t)(cxy) << 48) | (((uint64_t)(ptr)) & PTR_MASK))

#define LPA_MASK               ((1ULL)<<47)

#define CXY_FROM_PADDR(paddr)  ((cxy_t)((paddr) >> 48)) 

#define LPA_FROM_PADDR(paddr)  (lpa_t)((paddr & LPA_MASK)

#define PADDR(cxy,lad)         (((uint64_t)(cxy) << 48) | (((uint64_t)(ptr)) & LPA_MASK))



#endif	/* HAL_TYPES_H_ */
