/*
 * do_syscall.c - architecture independant entry-point for system calls.
 * 
 * Author    Alain Greiner (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_irqmask.h>
#include <do_syscall.h>
#include <errno.h>
#include <thread.h>
#include <printk.h>
#include <syscalls.h>

/////////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////////
static inline int sys_undefined()
{
    printk("\n[PANIC] in %s : undefined system call\n", __FUNCTION__ );
    hal_core_sleep();
    return 0;
}

/////////////////////////////////////////////////////////////////////////////////////////////
// This array of pointers define the kernel functions implementing the syscalls.
// It must be kept consistent with the enum in syscalls.h
/////////////////////////////////////////////////////////////////////////////////////////////

typedef int (*sys_func_t) ();

static const sys_func_t syscall_tbl[SYSCALLS_NR] = 
{
    sys_thread_exit,        // 0
    sys_mmap,               // 1
    sys_thread_create,      // 2
    sys_thread_join,        // 3
    sys_thread_detach,      // 4
    sys_thread_yield,       // 5
    sys_sem,                // 6
    sys_condvar,            // 7
    sys_barrier,            // 8
    sys_mutex,              // 9
    sys_thread_sleep,       // 10
    sys_thread_wakeup,      // 11
    sys_open,               // 12
    sys_creat,              // 13
    sys_read,               // 14
    sys_write,              // 15
    sys_lseek,              // 16
    sys_close,              // 17
    sys_unlink,             // 18
    sys_pipe,               // 19
    sys_chdir,              // 20
    sys_mkdir,              // 21
    sys_mkfifo,             // 22
    sys_opendir,            // 23
    sys_readdir,            // 24
    sys_closedir,           // 25
    sys_getcwd,             // 26
    sys_clock,              // 27
    sys_alarm,              // 28
    sys_rmdir,              // 29
    sys_utls,               // 30
    sys_chmod,              // 31
    sys_signal,             // 32 
    sys_timeofday,          // 33
    sys_kill,               // 34
    sys_getpid,             // 35
    sys_fork,               // 36
    sys_exec,               // 37
    sys_stat,               // 38
    sys_trace,              // 39
};

//////////////////////////////////
reg_t do_syscall( thread_t * this,
                  reg_t      arg0,
		          reg_t      arg1,
		          reg_t      arg2,
		          reg_t      arg3,
		          reg_t      service_num )
{
	int           error = 0;
	
    // update user time 
	thread_user_time_update( this );

    // enable IRQs
	hal_enable_irq( NULL );
  
    // check syscall index
	if( service_num >= SYSCALLS_NR )
	{
		printk("\n[ERROR] in %s : Undefined syscall %d, for thread %x\n",
		       __FUNCTION__ , service_num , this );

		this->errno = ENOSYS;
	    hal_disable_irq(NULL);
		return ENOSYS;;
	}

	syscall_dmsg("\n[INFO] %s : pid = %x / trdid = %x / service #%d\n"
                 "         arg0 = %x / arg1 = %x / arg2 = %x / arg3 = %x\n",
		         __FUNCTION__ , this->process->pid , this->trdid , service_num ,
		         arg0 , arg1 , arg2 , arg3 );

    // reset errno
	this->errno = 0;

    // call relevant kernel function
	error = syscall_tbl[service_num] ( arg0 , arg1 , arg2 , arg3 );

    // handle pending signals for the calling thread
    thread_signals_handle( this );

	// disable IRQs
	hal_disable_irq( NULL );

    // update kernel time
	thread_kernel_time_update( this );

	return error;
}
