/*
 * hal_multiboot.h - Multiboot-specific values and structures
 *
 * Copyright (c) 2017 Maxime Villard
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

/*
 * Multiboot header structure.
 */
#define MULTIBOOT_HEADER_MAGIC		0x1BADB002
#define MULTIBOOT_HEADER_MODS_ALIGNED	0x00000001
#define MULTIBOOT_HEADER_WANT_MEMORY	0x00000002
#define MULTIBOOT_HEADER_HAS_VBE	0x00000004
#define MULTIBOOT_HEADER_HAS_ADDR	0x00010000

#ifndef x86_ASM
struct multiboot_header {
	uint32_t	mh_magic;
	uint32_t	mh_flags;
	uint32_t	mh_checksum;

	/* Valid if mh_flags sets MULTIBOOT_HEADER_HAS_ADDR. */
	paddr_t		mh_header_addr;
	paddr_t		mh_load_addr;
	paddr_t		mh_load_end_addr;
	paddr_t		mh_bss_end_addr;
	paddr_t		mh_entry_addr;

	/* Valid if mh_flags sets MULTIBOOT_HEADER_HAS_VBE. */
	uint32_t	mh_mode_type;
	uint32_t	mh_width;
	uint32_t	mh_height;
	uint32_t	mh_depth;
};
#endif

/*
 * Multiboot information structure.
 */
#define MULTIBOOT_INFO_MAGIC		0x2BADB002
#define MULTIBOOT_INFO_HAS_MEMORY	0x00000001
#define MULTIBOOT_INFO_HAS_BOOT_DEVICE	0x00000002
#define MULTIBOOT_INFO_HAS_CMDLINE	0x00000004
#define MULTIBOOT_INFO_HAS_MODS		0x00000008
#define MULTIBOOT_INFO_HAS_AOUT_SYMS	0x00000010
#define MULTIBOOT_INFO_HAS_ELF_SYMS	0x00000020
#define MULTIBOOT_INFO_HAS_MMAP		0x00000040
#define MULTIBOOT_INFO_HAS_DRIVES	0x00000080
#define MULTIBOOT_INFO_HAS_CONFIG_TABLE	0x00000100
#define MULTIBOOT_INFO_HAS_LOADER_NAME	0x00000200
#define MULTIBOOT_INFO_HAS_APM_TABLE	0x00000400
#define MULTIBOOT_INFO_HAS_VBE		0x00000800

#define MULTIBOOT_INFO_SIZE		92

/* Offsets into the structure */
#define MB_MI_FLAGS		0
#define MB_MI_CMDLINE		16
#define MB_MI_MMAP_LENGTH	44
#define MB_MI_MMAP_ADDR		48
#define MB_MI_LOADER_NAME	64

#ifndef x86_ASM
typedef uint32_t mb_paddr_t;

struct multiboot_info {
	uint32_t	mi_flags;

	/* Valid if mi_flags sets MULTIBOOT_INFO_HAS_MEMORY. */
	uint32_t	mi_mem_lower;
	uint32_t	mi_mem_upper;

	/* Valid if mi_flags sets MULTIBOOT_INFO_HAS_BOOT_DEVICE. */
	uint8_t		mi_boot_device_part3;
	uint8_t		mi_boot_device_part2;
	uint8_t		mi_boot_device_part1;
	uint8_t		mi_boot_device_drive;

	/* Valid if mi_flags sets MULTIBOOT_INFO_HAS_CMDLINE. */
	mb_paddr_t	mi_cmdline;

	/* Valid if mi_flags sets MULTIBOOT_INFO_HAS_MODS. */
	uint32_t	mi_mods_count;
	mb_paddr_t	mi_mods_addr;

	/* Valid if mi_flags sets MULTIBOOT_INFO_HAS_{AOUT,ELF}_SYMS. */
	uint32_t	mi_elfshdr_num;
	uint32_t	mi_elfshdr_size;
	mb_paddr_t	mi_elfshdr_addr;
	uint32_t	mi_elfshdr_shndx;

	/* Valid if mi_flags sets MULTIBOOT_INFO_HAS_MMAP. */
	uint32_t	mi_mmap_length;
	mb_paddr_t	mi_mmap_addr;

	/* Valid if mi_flags sets MULTIBOOT_INFO_HAS_DRIVES. */
	uint32_t	mi_drives_length;
	mb_paddr_t	mi_drives_addr;

	/* Valid if mi_flags sets MULTIBOOT_INFO_HAS_CONFIG_TABLE. */
	mb_paddr_t	unused_mi_config_table;

	/* Valid if mi_flags sets MULTIBOOT_INFO_HAS_LOADER_NAME. */
	mb_paddr_t	mi_loader_name;

	/* Valid if mi_flags sets MULTIBOOT_INFO_HAS_APM. */
	mb_paddr_t	unused_mi_apm_table;

	/* Valid if mi_flags sets MULTIBOOT_INFO_HAS_VBE. */
	mb_paddr_t	unused_mi_vbe_control_info;
	mb_paddr_t	unused_mi_vbe_mode_info;
	mb_paddr_t	unused_mi_vbe_interface_seg;
	mb_paddr_t	unused_mi_vbe_interface_off;
	uint32_t	unused_mi_vbe_interface_len;
};

/*
 * Drive information. This describes an entry in the drives table as
 * pointed to by mi_drives_addr.
 */
struct multiboot_drive {
	uint32_t	md_length;
	uint8_t		md_number;
	uint8_t		md_mode;
	uint16_t	md_cylinders;
	uint8_t		md_heads;
	uint8_t		md_sectors;

	/* The variable-sized 'ports' field comes here, so this structure
	 * can be longer. */
};

/*
 * Memory mapping. This describes an entry in the memory mappings table
 * as pointed to by mi_mmap_addr.
 *
 * Be aware that mm_size specifies the size of all other fields *except*
 * for mm_size. In order to jump between two different entries, you
 * have to count mm_size + 4 bytes.
 */
struct multiboot_mmap {
	uint32_t	mm_size;
	uint64_t	mm_base_addr;
	uint64_t	mm_length;
	uint32_t	mm_type;
} __packed;

/*
 * Modules. This describes an entry in the modules table as pointed
 * to by mi_mods_addr.
 */

struct multiboot_module {
	uint32_t	mmo_start;
	uint32_t	mmo_end;
	mb_paddr_t	mmo_string;
	uint32_t	mmo_reserved;
};
#endif

