/*
 * sys_sem.c - Acces a POSIX unamed semaphore.
 * 
 * Authors     Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_uspace.h>
#include <errno.h>
#include <thread.h>
#include <printk.h>
#include <vmm.h>
#include <remote_sem.h>
#include <syscalls.h>

//////////////////////////////////
int sys_sem( void         * vaddr,        // semaphore virtual  address 
             uint32_t       operation,    // requested operation type
             uint32_t     * value )       // pointer on in/out argument
{
	uint32_t             data;   
	paddr_t              paddr;
    error_t              error;

    thread_t           * this = CURRENT_THREAD;

    // check vaddr in user vspace
	error = vmm_v2p_translate( false , vaddr , &paddr );
	if( error )
    {
        printk("\n[ERROR] in %s : illegal semaphore virtual address = %x\n",
               __FUNCTION__ , (intptr_t)vaddr );
        this->errno = error;
        return -1;
    }

    // check value in user vspace
	error = vmm_v2p_translate( false , value , &paddr );
	if( error )
    {
        printk("\n[ERROR] in %s : illegal argument virtual address = %x\n",
               __FUNCTION__ , (intptr_t)value );
        this->errno = error;
        return -1;   
    }
    
    // execute requested operation
	switch( operation )
	{
        //////////////
	    case SEM_INIT:
        {
            // get argument
		    hal_copy_from_uspace( &data , value , sizeof(uint32_t) );

            // call init function
            error = remote_sem_create( (intptr_t)vaddr , data );

            if ( error )
            {
                printk("\n[ERROR] in %s : cannot create semaphore = %x\n",
                       __FUNCTION__ , (intptr_t)value );
                this->errno = error;
                return -1;
            }
            break;
        }
        //////////////////
        case SEM_GETVALUE:
        {
            // get extended pointer on remote semaphore
            xptr_t sem_xp = remote_sem_from_vaddr( (intptr_t)vaddr );

            if( sem_xp == XPTR_NULL )     // user error
            {
                printk("\n[ERROR] in %s : semaphore %x not registered\n",
                       __FUNCTION__ , (intptr_t)value );
                this->errno = EINVAL;
                return -1;
            }
            else                          // success
            {
                // get semaphore current value
		        remote_sem_get_value( sem_xp , &data );
 
                // return value to user
                hal_copy_to_uspace( value , &data , sizeof(uint32_t) );
            }
            break;
        }
        //////////////
	    case SEM_WAIT:
        {  
            // get extended pointer on remote semaphore
            xptr_t sem_xp = remote_sem_from_vaddr( (intptr_t)vaddr );

            if( sem_xp == XPTR_NULL )     // user error
            {
                printk("\n[ERROR] in %s : semaphore %x not registered\n",
                       __FUNCTION__ , (intptr_t)value );
                this->errno = EINVAL;
                return -1;
            }
            else                          // success
            {
                // wait semaphore available
                remote_sem_wait( sem_xp );
            }
            break;
        }
        //////////////
	    case SEM_POST:
        {
            // get extended pointer on remote semaphore
            xptr_t sem_xp = remote_sem_from_vaddr( (intptr_t)vaddr );

            if( sem_xp == XPTR_NULL )     // user error
            {
                printk("\n[ERROR] in %s : semaphore %x not registered\n",
                       __FUNCTION__ , (intptr_t)value );
                this->errno = EINVAL;
                return -1;
            }
            else                          // success
            {
                // release semaphore 
                remote_sem_post( sem_xp );
            }
			break;
        }
        /////////////////
	    case SEM_DESTROY:
        {
            // get extended pointer on remote semaphore
            xptr_t sem_xp = remote_sem_from_vaddr( (intptr_t)vaddr );

            if( sem_xp == XPTR_NULL )     // user error
            {
                printk("\n[ERROR] in %s : semaphore %x not registered\n",
                       __FUNCTION__ , (intptr_t)value );
                this->errno = EINVAL;
                return -1;
            }
            else                          // success 
            {
                // destroy semaphore
                remote_sem_destroy( sem_xp );
            }
            break;
	    }	
        ///////
	    default:  // undefined operation                       
        {
            printk("\n[PANIC] in %s : illegal operation type\n", __FUNCTION__ );
            hal_core_sleep();
        }
	}

    return 0;

}  // end sys_sem()
