/*
 * soclib_bdv.c - soclib simple block device driver implementation.
 *
 * Author     Alain Greiner (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH.is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH.is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH.; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <chdev.h>
#include <dev_ioc.h>
#include <soclib_bdv.h>
#include <printk.h>
#include <thread.h>
#include <spinlock.h>


///////////////////////////////////////
void soclib_bdv_init( chdev_t * chdev )
{
    // get extended pointer on SOCLIB_BDV peripheral base address
	xptr_t  bdv_xp = chdev->base;

    // set driver specific fields
    chdev->cmd = &soclib_bdv_cmd;
    chdev->isr = &soclib_bdv_isr;

    // get hardware device cluster and local pointer
    cxy_t      bdv_cxy  = GET_CXY( bdv_xp );
    uint32_t * bdv_ptr  = (uint32_t *)GET_PTR( bdv_xp );

    // get block_size and block_count  
	uint32_t block_size = hal_remote_lw( XPTR( bdv_cxy , bdv_ptr + BDV_BLOCK_SIZE_REG ) );
	uint32_t block_count = hal_remote_lw( XPTR( bdv_cxy , bdv_ptr + BDV_SIZE_REG ) );

    // set IOC device descriptor extension
    chdev->ext.ioc.size  = block_size;
    chdev->ext.ioc.count = block_count;

} // end soclib_bdv_init()


//////////////////////////////////////////////////////////////
void __attribute__ ((noinline)) soclib_bdv_cmd( xptr_t th_xp )
{
    uint32_t   cmd_type;    // IOC_READ / IOC_WRITE / IOC_SYNC_READ
    uint32_t   lba;
    uint32_t   count;
    xptr_t     buf_xp;
    xptr_t     ioc_xp;

    // get client thread cluster and local pointer
    cxy_t      th_cxy = GET_CXY( th_xp );
    thread_t * th_ptr = (thread_t *)GET_PTR( th_xp );

    // get command arguments and extended pointer on IOC device
    cmd_type =         hal_remote_lw ( XPTR( th_cxy , &th_ptr->ioc_cmd.type   ) );
    lba      =         hal_remote_lw ( XPTR( th_cxy , &th_ptr->ioc_cmd.lba    ) );
    count    =         hal_remote_lw ( XPTR( th_cxy , &th_ptr->ioc_cmd.count  ) );
    buf_xp   = (xptr_t)hal_remote_lwd( XPTR( th_cxy , &th_ptr->ioc_cmd.buf_xp ) );
    ioc_xp   = (xptr_t)hal_remote_lwd( XPTR( th_cxy , &th_ptr->ioc_cmd.dev_xp ) );

    // get IOC device cluster and local pointer
    cxy_t      ioc_cxy = GET_CXY( ioc_xp );
    chdev_t  * ioc_ptr = (chdev_t *)GET_PTR( ioc_xp );

    // get extended pointer on SOCLIB-BDV peripheral
    xptr_t     bdv_xp = hal_remote_lw( XPTR( ioc_cxy , &ioc_ptr->base ) );

    // get SOCLIB_BDV device cluster and local pointer
    cxy_t      bdv_cxy = GET_CXY( bdv_xp );
    uint32_t * bdv_ptr = (uint32_t *)GET_PTR( bdv_xp );

    // split buffer address in two 32 bits words
    uint32_t   buf_lsb = (uint32_t)(buf_xp);
    uint32_t   buf_msb = (uint32_t)(buf_xp>>32);

    // set operation
    uint32_t   op;
    if( cmd_type == IOC_WRITE ) op = BDV_OP_WRITE; 
    else                        op = BDV_OP_READ;

    // set SOCLIB_BDV registers to start one I/O operation
    hal_remote_sw( XPTR( bdv_cxy , bdv_ptr + BDV_IRQ_ENABLE_REG ) , 1       );
    hal_remote_sw( XPTR( bdv_cxy , bdv_ptr + BDV_BUFFER_REG     ) , buf_lsb ); 
    hal_remote_sw( XPTR( bdv_cxy , bdv_ptr + BDV_BUFFER_EXT_REG ) , buf_msb ); 
    hal_remote_sw( XPTR( bdv_cxy , bdv_ptr + BDV_LBA_REG        ) , lba     );
    hal_remote_sw( XPTR( bdv_cxy , bdv_ptr + BDV_COUNT_REG      ) , count   );
    hal_remote_sw( XPTR( bdv_cxy , bdv_ptr + BDV_OP_REG         ) , op      );

    // waiting policy  depends on the command type

    if( cmd_type == IOC_SYNC_READ )                  // polling policy
    {
        uint32_t status;
        while (1)
        {
            status = hal_remote_lw( XPTR( bdv_cxy , bdv_ptr + BDV_STATUS_REG ) );

            if( status == BDV_READ_SUCCESS ) // successfully completed
            {
                hal_remote_sw( XPTR( th_cxy , &th_ptr->ioc_cmd.error ) , 0 );
                break;
            }
            else if( status == BDV_BUSY )   // non completed
            {
                continue;
            }
            else                            // error reported
            {
                hal_remote_sw( XPTR( th_cxy , &th_ptr->ioc_cmd.error ) , 1 );
                break;
            }
        }
    }
    else                                            // descheduling + IRQ policy
    { 
        thread_block( CURRENT_THREAD , THREAD_BLOCKED_DEV_ISR );
        sched_yield( NULL );
    }
    
} // end soclib_bdv_cmd()


/////////////////////////////////////////////////////////////////
void __attribute__ ((noinline)) soclib_bdv_isr( chdev_t * chdev )
{
    // get extended pointer on client thread
    xptr_t root = XPTR( local_cxy , &chdev->wait_root );
    xptr_t client_xp = XLIST_FIRST_ELEMENT( root , thread_t , wait_list );

    // get extended pointer on server thread
    xptr_t server_xp = XPTR( local_cxy , &chdev->server );

    // get client thread cluster and local pointer
    cxy_t      client_cxy = GET_CXY( client_xp );
    thread_t * client_ptr = (thread_t *)GET_PTR( client_xp );

    // get SOCLIB_BDV device cluster and local pointer
    cxy_t      bdv_cxy  = GET_CXY( chdev->base );
    uint32_t * bdv_ptr  = (uint32_t *)GET_PTR( chdev->base );

    // get BDV status register and acknowledge IRQ
	uint32_t status = hal_remote_lw( XPTR( bdv_cxy , bdv_ptr + BDV_STATUS_REG ) );   

    // set operation status in command
	if((status != BDV_READ_SUCCESS) && (status != BDV_WRITE_SUCCESS)) 
    {
        hal_remote_sw( XPTR( client_cxy , &client_ptr->ioc_cmd.error ) , 1 );
    }
	else
    {
        hal_remote_sw( XPTR( client_cxy , &client_ptr->ioc_cmd.error ) , 0 );
    }

    // unblock server thread 
    thread_unblock( server_xp , THREAD_BLOCKED_DEV_ISR );

    // unblock client thread
    thread_unblock( client_xp , THREAD_BLOCKED_IO );

} // end soclib_bdv_isr()



