/*
 * hal_context.h - Generic Thread Context Access API definition.
 * 
 * Author  Alain Greiner    (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 * 
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef  _HAL_CONTEXT_H_
#define  _HAL_CONTEXT_H_

//////////////////////////////////////////////////////////////////////////////////////////
//        Generic Thread Context API definition (implementation in hal_context.c) 
//
// A thread context is defined by the two (core specific) structures hal_cpu_context_t
// and hal_fpu_context_t, defined in hal_context.c file, that are accessed with generic
// void* pointers stored in the thread descriptor.
// - the "hal_context_t" struct is used for the CPU registers values at context switch.
// - the "hal_fpu_context_t" struct is used for the FPU registers when required.
//////////////////////////////////////////////////////////////////////////////////////////

/**** Forward declarations ****/

struct thread_s;

/****************************************************************************************
 * This function allocates, from the local cluster, the physical memory required for 
 * the thread CPU context, initialises it, and links the context to the thread.
 ****************************************************************************************
 * @ thread  : pointer on the thread descriptor.
 * @ return 0 if success / return ENOMEM if error
 ***************************************************************************************/
error_t hal_cpu_context_create( struct thread_s * thread );

/****************************************************************************************
 * This function allocates, from the local cluster, the physical memory required for 
 * a thread CPU context, initialises it from values contained in "src" thread context,
 * and links the context to the "dst" thread.
 ****************************************************************************************
 * @ dst  : pointer on the destination thread descriptor.
 * @ src  : pointer on the source thread descriptor.
 * @ return 0 if success / return ENOMEM if error
 ***************************************************************************************/
error_t hal_cpu_context_copy( struct thread_s * dst,
                              struct thread_s * src );

/****************************************************************************************
 * This function releases the physical memory allocated for a thread CPU context.
 ****************************************************************************************
 * @ thread  : pointer on the thread descriptor.
 ***************************************************************************************/
void hal_cpu_context_destroy( struct thread_s * thread );

/****************************************************************************************
 * This function performs a context switch, saving the CPU register values into the
 * old thread, and initializing these registers with the values of the new thread.
 ****************************************************************************************
 * @ old     : pointer on current thread.
 * @ new     : pointer on the thread we want to switch to.
 ***************************************************************************************/
void hal_cpu_context_switch( struct thread_s * old , struct thread_s * new );

/****************************************************************************************
 * This function loads the relevant CPU registers from values contained in
 * the thread context. It should be called for a thread that has not been executed yet.
 * It reset the loadable flag in thread descriptor.
 ****************************************************************************************
 * @ thread  : pointer on the thread descriptor.
 ***************************************************************************************/
void hal_cpu_context_load( struct thread_s * thread );

/****************************************************************************************
 * This function allocates, from the local cluster, the physical memory required for 
 * the thread FPU context, and initialises the thread pointer.
 ****************************************************************************************
 * @ thread  : pointer on the thread descriptor.
 * @ return 0 if success / return ENOMEM if error
 ***************************************************************************************/
error_t hal_fpu_context_create( struct thread_s * thread );

/****************************************************************************************
 * This function allocates, from the local cluster, the physical memory required for 
 * a thread FPU context, initialises it from values contained in "src" thread context,
 * and link the context to the "dst" thread.
 ****************************************************************************************
 * @ dst  : pointer on the destination thread descriptor.
 * @ src  : pointer on the source thread descriptor.
 * @ return 0 if success / return ENOMEM if error
 ***************************************************************************************/
error_t hal_fpu_context_copy( struct thread_s * dst,
                           struct thread_s * src );

/****************************************************************************************
 * This function releases the physical memory allocated for a FPU context.
 ****************************************************************************************
 * @ thread  : pointer on the thread descriptor.
 ***************************************************************************************/
void hal_fpu_context_destroy( struct thread_s * thread );

/****************************************************************************************
 * This function saves in the thread uzone the FPU registers values.
 ****************************************************************************************
 * @ thread  : pointer on the thread descriptor.
 ***************************************************************************************/
void hal_fpu_context_save( struct thread_s * thread );

/****************************************************************************************
 * This function restores from the thread uzone the FPU registers values.
 ****************************************************************************************
 * @ thread  : pointer on the thread descriptor.
 ***************************************************************************************/
void hal_fpu_context_restore( struct thread_s * thread );

#endif	/* _HAL_CONTEXT_H_ */
