/*
 * hal_acpi.c - ACPI parser
 *
 * Copyright (c) 2017 Maxime Villard
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH.; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_boot.h>
#include <hal_acpi.h>
#include <hal_internal.h>

#include <memcpy.h>
#include <thread.h>
#include <string.h>
#include <process.h>
#include <printk.h>
#include <vmm.h>
#include <core.h>
#include <cluster.h>


/* XXX XXX libk XXX XXX */
int memcmp(char *s1, char *s2, size_t n)
{
	size_t i;
	for(i = 0; i < n; i++) {
		if (s1[i] != s2[i])
			return 1;
	}
	return 0;
}

static bool_t hal_acpi_rsdp_correct(uint8_t *ptr)
{
	uint8_t sum = 0;
	size_t i;

	/* Verify checksum */
	for (i = 0; i < 20; i++) {
		sum += ptr[i];
	}

	return (sum == 0);
}

static void *hal_acpi_find_rsdp(vaddr_t va_start, vaddr_t va_end)
{
	rsdp_t *rsdp;
	vaddr_t va = va_start;
	size_t i, n = PAGE_SIZE / ACPI_RSDP_ALIGN;
	char oem[ACPI_OEM_ID_SIZE + 1];
	uint8_t *ptr;

	/* The table is on a 16bit boundary */
	for (va = va_start; va < va_end; va += PAGE_SIZE) {
		for (i = 0; i < n; i++) {
			ptr = (uint8_t *)va + i * ACPI_RSDP_ALIGN;
			if (memcmp(ptr, ACPI_RSDP_SIGNATURE, ACPI_RSDP_SIGNATURE_SIZE))
				continue;
			if (!hal_acpi_rsdp_correct(ptr))
				continue;
			goto found;
		}
	}

	return NULL;

found:
	rsdp = (rsdp_t *)ptr;

	x86_printf("-> rsdp = %Z\n", rsdp);
	memcpy(&oem, rsdp->OemId, ACPI_OEM_ID_SIZE);
	oem[ACPI_OEM_ID_SIZE] = '\0';
	x86_printf("-> OEM: %s\n", oem);

	return rsdp;
}

void hal_acpi_init()
{
	rsdp_t *rsdp;
	paddr_t bios_min = 0x0E0000;
	paddr_t bios_max = 0x100000;
	vaddr_t vabase;
	vaddr_t va;
	paddr_t pa;
	size_t i, npages;

	npages = (bios_max - bios_min) / PAGE_SIZE;
	vabase = hal_gpt_bootstrap_valloc(npages);

	for (i = 0; i < npages; i++) {
		va = vabase + i * PAGE_SIZE;
		pa = bios_min + i * PAGE_SIZE;
		hal_gpt_enter(va, pa);
		invlpg(va);
	}

	/* First, find RSDP */
	rsdp = hal_acpi_find_rsdp(vabase, vabase + npages * PAGE_SIZE);
	if (rsdp == NULL) {
		x86_printf("[!] RSDP not found\n");
	}
}

