/*
 * remote_rwlock.h - kernel remote_rwlock definition.
 *
 * Authors   Alain Greiner   (2016)
 * 
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _REMOTE_RWLOCK_H_
#define _REMOTE_RWLOCK_H_

#include <kernel_config.h>
#include <hal_types.h>
#include <xlist.h>

/***************************************************************************************
 * This file defines a remote kernel lock, that supports several simultaneous read 
 * accesses, but only one write access. It implements a ticket based allocation policy.
 * It can be used to synchronize threads running in different clusters, because
 * all access functions use remote pointers.
 * - A reader take the lock to atomically increments the registered readers count.
 *   Then it release the lock and access the protected structure. It atomically 
 *   decrements the readers count without taking the lock when access is completed.
 * - A writer take the lock and keep it, but must wait completion of all current read 
 *   accesses before starting its own access.
 * When the lock is taken by another thread, the new-comers use a busy waiting policy.
 *
 * It uses a busy-waiting policy if the lock is already allocated to another thread.
 **************************************************************************************/

typedef struct remote_rwlock_s
{
    uint32_t     ticket;                /*! first free ticket index                   */
    uint32_t     current;               /*! ticket index of current owner             */
    uint32_t     count;                 /*! current number of reader threads          */
    xptr_t       owner;                 /*! extended pointer on writer thread         */
} 
remote_rwlock_t;

/***************************************************************************************
 * This function initializes a remote rwlock.
 ***************************************************************************************
 * @ lock_xp    : extended pointer on the remote rwlock
 **************************************************************************************/
void remote_rwlock_init( xptr_t lock_xp );

/***************************************************************************************
 * This blocking function get access to a remote rwlock for a reader.
 * It increments the calling thread locks count when the lock has been taken.
 ***************************************************************************************
 * @ lock_xp    : extended pointer on the remote rwlock
 **************************************************************************************/
void remote_rwlock_rd_lock( xptr_t lock_xp );

/***************************************************************************************
 * This function releases a remote rwlock for a reader.
 * It decrements the calling thread locks count when the lock has been released.
 ***************************************************************************************
 * @ lock_xp    : extended pointer on the remote rwlock
 **************************************************************************************/
void remote_rwlock_rd_unlock( xptr_t lock_xp );

/***************************************************************************************
 * This blocking function get access to a remote rwlock for a writer.
 * It increments the calling thread locks count when the lock has been taken.
 ***************************************************************************************
 * @ lock_xp    : extended pointer on the remote rwlock
 **************************************************************************************/
void remote_rwlock_wr_lock( xptr_t lock_xp );

/***************************************************************************************
 * This function releases a remote rwlock for a writer.
 * It decrements the calling thread locks count when the lock has been released.
 ***************************************************************************************
 * @ lock_xp    : extended pointer on the remote rwlock
 **************************************************************************************/
void remote_rwlock_wr_unlock( xptr_t lock_xp );

#endif
