/*
 * sys_thread_create.c - creates a new user thread
 *
 * Author     Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_types.h>
#include <hal_uspace.h>
#include <printk.h>
#include <errno.h>
#include <core.h>
#include <cluster.h>
#include <list.h>
#include <xlist.h>
#include <thread.h>
#include <scheduler.h>
#include <kmem.h>
#include <process.h>
#include <spinlock.h>
#include <dqdt.h>
#include <rpc.h>


//////////////////////////////////////////////////////////////////////////////////////////
// This function implements the pthread_create system call
//////////////////////////////////////////////////////////////////////////////////////////
int sys_thread_create ( thread_t       * new_thread,    // [out] argument
                        pthread_attr_t * user_attr,     // [in] argument
                        void           * start_func,    // [in] argument
                        void           * start_arg )    // [in] argument
{
	pthread_attr_t   k_attr;           // copy of pthread attributes in kernel space
	thread_t       * parent;           // pointer on thread executing the pthread_create
	xptr_t           parent_xp;        // extended pointer on created thread
	thread_t       * child_ptr;        // pointer on created child thread
	xptr_t           child_xp;         // extended pointer on created thread
	trdid_t          trdid;            // created thread identifier
	process_t      * process;          // pointer on local process descriptor
	paddr_t          paddr;            // unused, required by vmm_v2p_translate()
	error_t          error;

	uint32_t         tm_start;
	uint32_t         tm_end;

	tm_start = hal_get_cycles();

	// get parent thead pointer, extended pointer, and process pointer
	parent     = CURRENT_THREAD;
	parent_xp  = XPTR( local_cxy , parent );
	process    = parent->process;

	// check user_attr in user space
	error = vmm_v2p_translate( false , user_attr , &paddr );

	if( error )
	{
		printk("\n[ERROR] in %s : user_attr unmapped\n", __FUNCTION__ );
		parent->errno = EINVAL;
		return -1;
	}

	// check start_func in user space
	error = vmm_v2p_translate( false , start_func , &paddr );

	if( error )
	{
		printk("\n[ERROR] in %s : start_func unmapped\n", __FUNCTION__ );
		parent->errno = EINVAL;
		return -1;
	}

	// check start_arg in user space
	if( start_arg != NULL ) error = vmm_v2p_translate( false , start_arg , &paddr );

	if( error )
	{
		printk("\n[ERROR] in %s : start_arg unmapped\n", __FUNCTION__ );
		parent->errno = EINVAL;
		return -1;
	}

	// copy user_attr structure from user space to kernel space
	hal_copy_from_uspace( &k_attr , user_attr , sizeof(pthread_attr_t) );

	// check/set "cxy" attribute
	if( k_attr.attributes & PT_ATTR_CLUSTER_DEFINED )
	{
		if( cluster_is_undefined( k_attr.cxy ) )
		{
			printk("\n[ERROR] in %s : illegal target cluster attribute = %x\n",
			       __FUNCTION__ , k_attr.cxy );
			parent->errno = EINVAL;
			return -1;
		}
	}
	else
	{
		k_attr.cxy = dqdt_get_cluster_for_process();
	}

	// create the thread, using a RPC if required
	// this returns "error", "child", and "child_xp"

	if( k_attr.cxy == local_cxy )                         // target cluster is local
	{
		// create thread in local cluster
		error = thread_user_create( process->pid,
		                            start_func,
		                            start_arg,
		                            &k_attr,
		                            &child_ptr );

		child_xp = XPTR( local_cxy , child_ptr );
	}
	else                                                 // target cluster is remote
	{
		rpc_thread_user_create_client( k_attr.cxy,
		                               process->pid,
		                               start_func,
		                               start_arg,
		                               &k_attr,
		                               &child_xp,
		                               &error );

		child_ptr = (thread_t *)GET_PTR( child_xp );
	}

	// check successful thread creation
	if( error )
	{
		printk("\n[ERROR] in %s : cannot create thread\n", __FUNCTION__ );
		return ENOMEM;
	}

	// returns trdid to user space
	trdid = hal_remote_lw( XPTR( k_attr.cxy , &child_ptr->trdid ) );
	hal_copy_to_uspace( new_thread , &trdid , sizeof(pthread_t) );

	// register new-thread in parent-thread children list if required
	if( (k_attr.attributes & PT_ATTR_DETACH) == 0 )
	    thread_child_parent_link( parent_xp , child_xp );

	tm_end = hal_get_cycles();

	thread_dmsg("\n[INFO] %s created thread %x for process %x in cluster %x\n"
	            "  start_cycle = %d / end_cycle = %d\n",
	               trdid , process->pid , k_attr.cxy , tm_start , tm_end );
	return 0;
}

