/*
 * devfs.c - DEVFS File system API implementation.
 *
 * Author   Mohamed Lamine Karaoui (2014,2015)
 *          Alain Greiner (2016,2017)
 *
 * Copyright (c) Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_special.h>
#include <printk.h>
#include <chdev.h>
#include <cluster.h>
#include <vfs.h>
#include <kmem.h>
#include <devfs.h>

/////////////////////////////////////////////////////////////////////////////////////////
//     Extern variables
/////////////////////////////////////////////////////////////////////////////////////////

extern vfs_ctx_t            fs_context[];   // allocated in kernel_init.c
extern chdev_directory_t    chdev_dir;      // allocated in kernel_init.c

///////////////////////////////
devfs_ctx_t * devfs_ctx_alloc()
{
    kmem_req_t    req;

	req.type    = KMEM_DEVFS_CTX;
	req.size    = sizeof(devfs_ctx_t);
    req.flags   = AF_KERNEL | AF_ZERO;

	return (devfs_ctx_t *)kmem_alloc( &req );
}

/////////////////////////////////////////////
void devfs_ctx_init( devfs_ctx_t * devfs_ctx,
                     xptr_t        devfs_dev_inode_xp,
                     xptr_t        devfs_external_inode_xp )
{
    devfs_ctx->dev_inode_xp      = devfs_dev_inode_xp;
    devfs_ctx->external_inode_xp = devfs_external_inode_xp;

    fs_context[FS_TYPE_DEVFS].extend = devfs_ctx;
}

/////////////////////////////////////////////////
void devfs_ctx_destroy( devfs_ctx_t * devfs_ctx )
{
    kmem_req_t    req;

    req.type = KMEM_DEVFS_CTX;
    req.ptr  = devfs_ctx;
    kmem_free( &req );
}

///////////////////////////////////////////////////
void devfs_global_init( xptr_t   parent_inode_xp,
                        xptr_t * devfs_dev_inode_xp,
                        xptr_t * devfs_external_inode_xp )
{
    error_t  error;

    devfs_dmsg("\n[INFO] %s : enter in cluster %x\n",
               __FUNCTION__ , local_cxy );

    // creates DEVFS "dev" inode in cluster IO
    error = vfs_add_child_in_parent( LOCAL_CLUSTER->io_cxy,
                                     INODE_TYPE_DIR,
                                     FS_TYPE_DEVFS,
                                     parent_inode_xp,
                                     "dev",
                                     NULL,
                                     devfs_dev_inode_xp ); 

    assert( (error == 0) , __FUNCTION__ , "cannot create <dev>\n" );

    devfs_dmsg("\n[INFO] %s : <dev> created in cluster %x\n",
               __FUNCTION__ , local_cxy );

    // create DEVFS "external" inode in cluster IO
    error = vfs_add_child_in_parent( LOCAL_CLUSTER->io_cxy,
                                     INODE_TYPE_DIR,
                                     FS_TYPE_DEVFS,
                                     *devfs_dev_inode_xp,
                                     "external",
                                     NULL,
                                     devfs_external_inode_xp ); 

    assert( (error == 0) , __FUNCTION__ , "cannot create <external>\n" );

    devfs_dmsg("\n[INFO] %s : <external> created in cluster %x\n",
               __FUNCTION__ , local_cxy );
} 

///////////////////////////////////////////////////
void devfs_local_init( xptr_t   devfs_dev_inode_xp,
                       xptr_t   devfs_external_inode_xp,
                       xptr_t * devfs_internal_inode_xp )
{
    char          node_name[16];
    xptr_t        chdev_xp;
    cxy_t         chdev_cxy;
    xptr_t        inode_xp;
    uint32_t      channel;

    // create "internal" directory linked to "dev"
    snprintf( node_name , 16 , "internal_%x" , local_cxy );
    vfs_add_child_in_parent( local_cxy,
                             INODE_TYPE_DIR,
                             FS_TYPE_DEVFS,
                             devfs_dev_inode_xp,
                             node_name,
                             NULL,
                             devfs_internal_inode_xp );

    // create MMC chdev inode
    chdev_xp = chdev_dir.mmc[local_cxy];
    if( chdev_xp != XPTR_NULL)
    {
        vfs_add_child_in_parent( local_cxy,
                                 INODE_TYPE_DEV,
                                 FS_TYPE_DEVFS,
                                 *devfs_internal_inode_xp,
                                 "mmc",
                                 GET_PTR( chdev_xp ),
                                 &inode_xp );
    }

    // create DMA chdev inodes (one DMA channel per core)
    for( channel = 0 ; channel < LOCAL_CLUSTER->cores_nr ; channel++ )
    {
        chdev_xp = chdev_dir.dma[channel];
        if( chdev_xp != XPTR_NULL)
        {
            snprintf( node_name , 16 , "dma_%d" , channel );
            vfs_add_child_in_parent( local_cxy,
                                     INODE_TYPE_DEV,
                                     FS_TYPE_DEVFS,
                                     *devfs_internal_inode_xp,
                                     node_name,
                                     GET_PTR( chdev_xp ),
                                     &inode_xp );
        }
    }

    // create an IOB inode in cluster containing IOB chdev
    chdev_xp = chdev_dir.iob;
    if( chdev_xp != XPTR_NULL )
    {
        chdev_cxy = GET_CXY( chdev_xp );
        if( chdev_cxy == local_cxy )
        {
            vfs_add_child_in_parent( local_cxy,
                                     INODE_TYPE_DEV,
                                     FS_TYPE_DEVFS,
                                     devfs_external_inode_xp,
                                     "iob",
                                     GET_PTR( chdev_xp ),
                                     &inode_xp );
        }
    }
        
    // create a PIC inode in cluster containing PIC chdev
    chdev_xp = chdev_dir.pic;
    if( chdev_xp != XPTR_NULL )
    {
        chdev_cxy = GET_CXY( chdev_xp );
        if( chdev_cxy == local_cxy )
        {
            vfs_add_child_in_parent( local_cxy,
                                     INODE_TYPE_DEV,
                                     FS_TYPE_DEVFS,
                                     devfs_external_inode_xp,
                                     "pic",
                                     GET_PTR( chdev_xp ),
                                     &inode_xp );
        }
    }

    // create a TXT inode in each cluster containing a TXT chdev
    for( channel = 0 ; channel < CONFIG_MAX_TXT_CHANNELS ; channel++ )
    {
        chdev_xp = chdev_dir.txt[channel];
        if( chdev_xp != XPTR_NULL )
        {
            chdev_cxy = GET_CXY( chdev_xp );
            if( chdev_cxy == local_cxy )
            {
                snprintf( node_name , 16 , "txt_%d" , channel );
                vfs_add_child_in_parent( local_cxy,
                                         INODE_TYPE_DEV,
                                         FS_TYPE_DEVFS,
                                         devfs_external_inode_xp,
                                         node_name,
                                         GET_PTR( chdev_xp ),
                                         &inode_xp );
            }
        }
    }

    // create an IOC inode in each cluster containing an IOC chdev 
    for( channel = 0 ; channel < CONFIG_MAX_IOC_CHANNELS ; channel++ )
    {
        chdev_xp = chdev_dir.ioc[channel];
        if( chdev_xp != XPTR_NULL )
        {
            chdev_cxy = GET_CXY( chdev_xp );
            if( chdev_cxy == local_cxy )
            {
                snprintf( node_name , 16 , "ioc_%d" , channel );
                vfs_add_child_in_parent( local_cxy,
                                         INODE_TYPE_DEV,
                                         FS_TYPE_DEVFS,
                                         devfs_external_inode_xp,
                                         node_name,
                                         GET_PTR( chdev_xp ),
                                         &inode_xp );
            }
        }
    }

    // create a FBF inode in each cluster containing a FBF chdev
    for( channel = 0 ; channel < CONFIG_MAX_IOC_CHANNELS ; channel++ )
    {
        chdev_xp = chdev_dir.fbf[channel];
        if( chdev_xp != XPTR_NULL )
        {
            chdev_cxy = GET_CXY( chdev_xp );
            if( chdev_cxy == local_cxy )
            {
                snprintf( node_name , 16 , "fbf_%d" , channel );
                vfs_add_child_in_parent( local_cxy,
                                         INODE_TYPE_DEV,
                                         FS_TYPE_DEVFS,
                                         devfs_external_inode_xp,
                                         node_name,
                                         GET_PTR( chdev_xp ),
                                         &inode_xp );
            }
        }
    }

    // create a NIC_RX inode in each cluster containing a NIC_RX chdev 
    for( channel = 0 ; channel < CONFIG_MAX_NIC_CHANNELS ; channel++ )
    {
        chdev_xp = chdev_dir.nic_rx[channel];
        if( chdev_xp != XPTR_NULL )
        {
            chdev_cxy = GET_CXY( chdev_xp );
            if( chdev_cxy == local_cxy )
            {
                snprintf( node_name , 16 , "nic_rx_%d" , channel );
                vfs_add_child_in_parent( local_cxy,
                                         INODE_TYPE_DEV,
                                         FS_TYPE_DEVFS,
                                         devfs_external_inode_xp,
                                         node_name,
                                         GET_PTR( chdev_xp ),
                                         &inode_xp );
            }
        }
    }

    // create a NIC_TX inode in each cluster containing a NIC_TX chdev 
    for( channel = 0 ; channel < CONFIG_MAX_NIC_CHANNELS ; channel++ )
    {
        chdev_xp = chdev_dir.nic_tx[channel];
        if( chdev_xp != XPTR_NULL )
        {
            chdev_cxy = GET_CXY( chdev_xp );
            if( chdev_cxy == local_cxy )
            {
                snprintf( node_name , 16 , "nic_tx_%d" , channel );
                vfs_add_child_in_parent( local_cxy,
                                         INODE_TYPE_DEV,
                                         FS_TYPE_DEVFS,
                                         devfs_external_inode_xp,
                                         node_name,
                                         GET_PTR( chdev_xp ),
                                         &inode_xp );
            }
        }
    }
}  // end devfs_local_init()

