/*
 * sys_open.c - open a file.
 *
 * Author        Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_types.h>
#include <hal_uspace.h>
#include <errno.h>
#include <thread.h>
#include <printk.h>
#include <vfs.h>
#include <process.h>
#include <remote_spinlock.h>
#include <remote_rwlock.h>

///////////////////////////////////
int sys_open ( char     * pathname,
               uint32_t   flags,
               uint32_t   mode )
{
    error_t        error;
    xptr_t         file_xp;                          // extended pointer on vfs_file_t
    uint32_t       file_id;                          // file descriptor index
    char           kbuf[CONFIG_VFS_MAX_PATH_LENGTH];

    thread_t     * this     = CURRENT_THREAD;
    process_t    * process  = this->process;

    // check fd_array not full
    if( process_fd_array_full() )
    {
        printk("\n[ERROR] in %s : file descriptor array full for process %x\n",
               __FUNCTION__ , process->pid );
        this->errno = ENFILE;
        return -1;
    }

    // get pathname copy in kernel space
    error = hal_strcpy_from_uspace( kbuf , pathname , CONFIG_VFS_MAX_PATH_LENGTH );

    if( error )
    {
        printk("\n[ERROR] in %s : pathname too long\n", __FUNCTION__ );
        this->errno = ENFILE;
        return -1;
    }

    // get cluster and local pointer on reference process
    xptr_t      ref_xp  = process->ref_xp;
    process_t * ref_ptr = (process_t *)GET_PTR( ref_xp );
    cxy_t       ref_cxy = GET_CXY( ref_xp );

    // get extended pointer on cwd inode
    xptr_t cwd_xp = hal_remote_lwd( XPTR( ref_cxy , &ref_ptr->vfs_cwd_xp ) );

    // get the cwd lock in read mode from reference process
    remote_rwlock_rd_lock( XPTR( ref_cxy , &ref_ptr->cwd_lock ) );

    // call the relevant VFS function
    error = vfs_open( cwd_xp,
                      kbuf,
                      flags,
                      mode,
                      &file_xp,
                      &file_id );

    // release the cwd lock
    remote_rwlock_rd_unlock( XPTR( ref_cxy , &ref_ptr->cwd_lock ) );

    if( error )
    {
        printk("\n[ERROR] in %s : cannot create file descriptor\n", __FUNCTION__ );
        this->errno = ENFILE;
        return -1;
    }

    // update local fd_array
    remote_spinlock_lock( XPTR( local_cxy , &process->fd_array.lock ) );
    process->fd_array.array[file_id] = file_xp;
    remote_spinlock_unlock( XPTR( local_cxy , &process->fd_array.lock ) );

    return file_id;
}
