/*
 * hal_context.h - Generic Thread Context Access API definition.
 * 
 * Author  Alain Greiner    (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 * 
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef  _HAL_CONTEXT_H_
#define  _HAL_CONTEXT_H_

//////////////////////////////////////////////////////////////////////////////////////////
//        Generic Thread Context API definition (implementation in hal_context.c) 
//
// A thread context is defined by the two (core specific) structures hal_context_t
// and hal_uzone_t, defined in hal_context.c file, that are accessed with generic
// void* pointers stored in the thread descriptor.
// - the "hal_context_t" structure is used to store the CPU registers values that
//   have not been saved in the stack by the interrupt handler.
// - the "hal_uzone_t" structure is used to save the FPU registers when required.
//////////////////////////////////////////////////////////////////////////////////////////

/**** Forward declarations ****/

struct thread_s;

/****************************************************************************************
 * Define various SR values for TSAR-MIPS32 
 ***************************************************************************************/

#define SR_USR_MODE       0xFC11
#define SR_USR_MODE_FPU   0x2000FC11
#define SR_SYS_MODE       0xFC00

/****************************************************************************************
 * This structure defines the cpu_context for TSAR MIPS32.
 * These registers are saved/restored at each context switch.
 * WARNING : update the hal_cpu_context_save() and hal_cpu_context_restore()
 *           functions when modifying this structure. 
 ***************************************************************************************/

typedef struct hal_cpu_context_s
{
	uint32_t s0_16;      // slot 0
	uint32_t s1_17;      // slot 1
	uint32_t s2_18;      // slot 2
	uint32_t s3_19;      // slot 3
	uint32_t s4_20;      // slot 4
	uint32_t s5_21;      // slot 5
	uint32_t s6_22;      // slot 6
	uint32_t s7_23;      // slot 7
	uint32_t sp_29;      // slot 8
	uint32_t fp_30;      // slot 9
	uint32_t ra_31;      // slot 10
	uint32_t c0_sr;      // slot 11
	uint32_t c0_th;      // slot 12
	uint32_t c2_ptpr;    // slot 13
	uint32_t c2_mode;    // slot 14
} 
hal_cpu_context_t;

/****************************************************************************************
 * This structure defines the fpu_context for TSAR MIPS32.
 ***************************************************************************************/

typedef struct hal_fpu_context_s
{
	uint32_t   fpu_regs[32];     
}
hal_fpu_context_t;

/****************************************************************************************
 * This function allocates, from the local cluster, the physical memory required for 
 * the thread CPU context, initialises it, and links the context to the thread.
 * Seven registers are initialised:
 * - sp_29 / fp_30 / ra_31
 * - c0_sr / c0_th
 * - c2_ptpr / c2_mode
 ****************************************************************************************
 * @ thread  : pointer on the thread descriptor.
 * @ return 0 if success / return ENOMEM if error
 ***************************************************************************************/
error_t hal_cpu_context_create( struct thread_s * thread );

/****************************************************************************************
 * This function allocates, from the local cluster, the physical memory required for 
 * a thread CPU context, initialises it from values contained in "src" thread context,
 * and links the context to the "dst" thread.
 ****************************************************************************************
 * @ dst  : pointer on the destination thread descriptor.
 * @ src  : pointer on the source thread descriptor.
 * @ return 0 if success / return ENOMEM if error
 ***************************************************************************************/
error_t hal_cpu_context_copy( struct thread_s * dst,
                              struct thread_s * src );

/****************************************************************************************
 * This function releases the physical memory allocated for a thread CPU context.
 ****************************************************************************************
 * @ thread  : pointer on the thread descriptor.
 ***************************************************************************************/
void hal_cpu_context_destroy( struct thread_s * thread );

/****************************************************************************************
 * This function saves in the thread context the CPU registers values that have not 
 * been saved in the thread stack by the exception handler:
 * - GRs : s0 to S7 , sp , fp, ra
 * - CP0 : th
 * - CP2 : ptpr , mode
 ****************************************************************************************
 * @ thread  : pointer on the thread descriptor.
 ***************************************************************************************/
void hal_cpu_context_save( struct thread_s * thread );

/****************************************************************************************
 * This function restores from the thread context the CPU registers values that have not 
 * been saved in the thread stack by the exception handler.
 * - GRs : s0 to S7 , sp , fp, ra
 * - CP0 : th
 * - CP2 : ptpr , mode
 ****************************************************************************************
 * @ thread  : pointer on the thread descriptor.
 ***************************************************************************************/
void hal_cpu_context_restore( struct thread_s * thread );

/****************************************************************************************
 * This function loads the relevant CPU registers from values contained in
 * the thread context. It should be called for a thread that has not been executed yet.
 * - GRs : sp , fp , a0
 * - CP0 : sr , epc , th
 * - CP2 : ptpr , mode
 * It reset the loadable flag in thread descriptor.
 ****************************************************************************************
 * @ thread  : pointer on the thread descriptor.
 ***************************************************************************************/
void hal_cpu_context_load( struct thread_s * thread );



/****************************************************************************************
 * This function allocates, from the local cluster, the physical memory required for 
 * the thread FPU context, and initialises the thread pointer.
 ****************************************************************************************
 * @ thread  : pointer on the thread descriptor.
 * @ return 0 if success / return ENOMEM if error
 ***************************************************************************************/
error_t hal_fpu_context_create( struct thread_s * thread );

/****************************************************************************************
 * This function allocates, from the local cluster, the physical memory required for 
 * a thread FPU context, initialises it from values contained in "src" thread context,
 * and link the context to the "dst" thread.
 ****************************************************************************************
 * @ dst  : pointer on the destination thread descriptor.
 * @ src  : pointer on the source thread descriptor.
 * @ return 0 if success / return ENOMEM if error
 ***************************************************************************************/
error_t hal_fpu_context_copy( struct thread_s * dst,
                           struct thread_s * src );

/****************************************************************************************
 * This function releases the physical memory allocated for a FPU context.
 ****************************************************************************************
 * @ thread  : pointer on the thread descriptor.
 ***************************************************************************************/
void hal_fpu_context_destroy( struct thread_s * thread );

/****************************************************************************************
 * This function saves in the thread uzone the FPU registers values.
 ****************************************************************************************
 * @ thread  : pointer on the thread descriptor.
 ***************************************************************************************/
void hal_fpu_context_save( struct thread_s * thread );

/****************************************************************************************
 * This function restores from the thread uzone the FPU registers values.
 ****************************************************************************************
 * @ thread  : pointer on the thread descriptor.
 ***************************************************************************************/
void hal_fpu_context_restore( struct thread_s * thread );

#endif	/* _HAL_CONTEXT_H_ */
