/*
 * hal_do_exception.c - Architecture specific exception handler API definition
 * 
 * Authors    Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _HAL_EXCEPTION_H_
#define _HAL_EXCEPTION_H_

#include <hal_types.h>

//////////////////////////////////////////////////////////////////////////////////////////
//     Architecture specific exception handler API
//
// ALMOS-MKH defines three classes of exceptions:
//
// - NON_FATAL : exceptions such as "page unmapped" or "FPU unusable" can be non fatal.
//   => The hal_do_exception() function call the generic vmm_handle_page_fault(),
//      or the fpu_handle_exception() function, and the calling thread resumes execution
//      when the exception has been handled.
//
// - USER_ERROR : exceptions such a "illegal vaddr" or "illegal write access" are fatal.
//   => The hal_do_exception() function send a kill signal to the calling thread process,
//      and displays an error message on TXT0.
//
// - KERNEL_PANIC : "kernel mistakes" are abnormal events.
//   => The hal_do_exception() function calls the generic panic() function, to display
//      a message on TXT0, disable IRQs and call the hal_core_sleep() function.
//
// For all exceptions, the faulty core context has been saved in a registers array
// stored in the user thread descriptor (for a core in user mode), and in the
// kernel stack (for a core in kernel mode). 
//
// Any architecture specific implementation must implement this API.
//////////////////////////////////////////////////////////////////////////////////////////

/**** forward declaration  ****/

struct thread_s;

/*****************************************************************************************
 * This function is called by the hal_kentry() function when an exception is detected by
 * the hardware for a given thread running on a given core.
 *****************************************************************************************
 * @ this      : pointer on the faulty thread descriptor.
 * @ regs_tbl  : array containing the core registers values saved by hal_kentry().
 ****************************************************************************************/
void hal_do_exception( struct thread_s * this, 
                       reg_t           * regs_tbl );

#endif  // _HAL_EXCEPTION_H_
