/*
 * vmm.h - virtual memory management related operations
 *
 * Authors   Ghassan Almaless (2008,2009,2010,2011, 2012)
 *           Mohamed Lamine Karaoui (2015)
 *           Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#ifndef _VMM_H_
#define _VMM_H_

#include <hal_types.h>
#include <bits.h>
#include <list.h>
#include <grdxt.h>
#include <spinlock.h>
#include <hal_gpt.h>
#include <vseg.h>
#include <page.h>

/****  Forward declarations  ****/

struct process_s;

/*********************************************************************************************
 * This structure defines the STACK allocator used by the VMM to dynamically allocate
 * STACK vsegs requested or released by the user process.
 * This allocator handles a fixed size array of fixed size slots stores in the STACK zone.
 * The stack size and the number of slots are defined by the CONFIG_VMM_STACK_SIZE, and
 * CONFIG_THREAD
 * Each slot can contain one user stack vseg. The first page in the slot is not allocated
 * to detect stack overflow.
 * The slot index can be computed form the slot base address, and reversely.
 * All allocation / release operations are registered in the stack_bitmap, that completely
 * define the STACK zone state.
 * In this implementation, the max number of slots is 32.
 ********************************************************************************************/

typedef struct stack_mgr_s
{
    spinlock_t     lock;               /*! lock protecting STACK allocator                  */
    vpn_t          vpn_base;           /*! first page of STACK zone                         */
    bitmap_t       bitmap;             /*! bit bector of allocated stacks                   */
}
stack_mgr_t;

/*********************************************************************************************
 * This structure defines the MMAP allocator used by the VMM to dynamically allocate
 * MMAP vsegs requested or released by an user process.
 * This allocator should be only used in the reference cluster.
 * - allocation policy : all allocated vsegs occupy an integer number of pages that is
 *   power of 2, and are aligned on a page boundary. The requested number of pages is
 *   rounded if required. The first_free_vpn variable defines completely the MMAP zone state.
 *   It is never decremented, as the released vsegs are simply registered in a zombi_list.
 *   The relevant zombi_list is checked first for each allocation request.
 * - release policy : a released MMAP vseg is registered in an array of zombi_lists.
 *   This array is indexed by ln(number of pages), and each entry contains the root of
 *   a local list of zombi vsegs that have the same size. The physical memory allocated
 *   for a zombi vseg descriptor is not released, to use the "list" field.
 *   This physical memory allocated for MMAP vseg descriptors is actually released
 *   when the VMM is destroyed.
 ********************************************************************************************/

typedef struct mmap_mgr_s
{
    spinlock_t     lock;               /*! lock protecting MMAP allocator                   */
    vpn_t          vpn_base;           /*! first page of MMAP zone                          */
    vpn_t          vpn_size;           /*! number of pages in MMAP zone                     */
    vpn_t          first_free_vpn;     /*! first free page in MMAP zone                     */
    list_entry_t   zombi_list[32];     /*! array of roots of released vsegs lists           */
}
mmap_mgr_t;

/*********************************************************************************************
 * This structure defines the Virtual Memory Manager for a given process in a given cluster.
 * This local VMM provides three main services:
 * 1) It registers all vsegs statically or dynamically defined in the vseg list,
 *    and in the associated radix-tree.
 * 2) It allocates virtual memory space for the STACKS and MMAP vsegs,
 *    using dedicated allocators.
 * 3) It contains the local copy of the generic page table descriptor.
 ********************************************************************************************/

typedef struct vmm_s
{
	rwlock_t       vsegs_lock;         /*! lock protecting the vsegs list & radix tree      */
	list_entry_t   vsegs_root;         /*! all vsegs in same process and same cluster       */
	uint32_t       vsegs_nr;           /*! total number of local vsegs                      */
  	grdxt_t        grdxt;              /*! embedded generic vsegs radix tree (key is vpn)   */

    gpt_t          gpt;                /*! embedded generic page table descriptor           */

    stack_mgr_t    stack_mgr;          /*! embedded STACK vsegs allocator                   */
    mmap_mgr_t     mmap_mgr;           /*! embedded MMAP vsegs allocator                    */

	uint32_t       pgfault_nr;         /*! page fault counter                               */
	uint32_t       u_err_nr;           /*! TODO ??? [AG]                                    */
	uint32_t       m_err_nr;           /*! TODO ??? [AG]                                    */

    vpn_t          kent_vpn_base;      /*! kentry vseg first page                           */
    vpn_t          args_vpn_base;      /*! args vseg first page                             */
    vpn_t          envs_vpn_base;      /*! envs zone first page                             */
    vpn_t          heap_vpn_base;      /*! envs zone first page                             */
	vpn_t          code_vpn_base;      /*! code zone first page                             */
	vpn_t          data_vpn_base;      /*! data zone first page                             */

	intptr_t       entry_point;        /*! main thread entry point                          */

    vseg_t       * heap_vseg;          /*! pointer on local heap vseg descriptor            */
}
vmm_t;

/*********************************************************************************************
 * This structure is used to store the arguments of the mmap() system call.
 ********************************************************************************************/

typedef struct mmap_attr_s
{
	void     * addr;            /*! requested virtual address (unused : should be NULL)     */
	uint32_t   length;          /*! requested vseg size (bytes)                             */
	uint32_t   prot;            /*! access modes                                            */
	uint32_t   flags;           /*! only MAP_FILE / MAP_ANON / MAP_PRIVATE / MAP_SHARED     */
	fdid_t     fdid;            /*! file descriptor index (if MAP_FILE is set)              */
	int32_t    offset;          /*! file offset (if MAP_FILE is set)                        */
}
mmap_attr_t;

/*********************************************************************************************
 * This function initialises the virtual memory manager attached to a process.
 * - It initializes the VSL (list of vsegs and associated radix tree).
 * - It initializes the generic page table (empty).
 * - It initializes the STAK and MMAP allocators.
 * - It registers the "kentry", "args", "envs" and "heap" vsegs in the vsegs list.
 * Any error in this function gives a kernel panic.
 *********************************************************************************************
 * @ process   : pointer on process descriptor
 ********************************************************************************************/
void vmm_init( struct process_s * process );

/*********************************************************************************************
 * This function copies the content of a source VMM to a destination VMM.
 *********************************************************************************************
 * @ dst_process   : pointer on destination process descriptor.
 * @ src_process   : pointer on source process descriptor.
 * @ return 0 if success / return ENOMEM if failure.
 ********************************************************************************************/
error_t vmm_copy( struct process_s * dst_process,
                  struct process_s * src_process );

/*********************************************************************************************
 * This function removes all vsegs registered in in a virtual memory manager,
 * and releases the memory allocated to the local generic page table.
 *********************************************************************************************
 * @ process   : pointer on process descriptor.
 ********************************************************************************************/
void vmm_destroy( struct process_s * process );

/*********************************************************************************************
 * This function scans the list of vsegs registered in the VMM of a given process descriptor
 * to check if a given virtual region (defined by a base and size) overlap an existing vseg.
 *********************************************************************************************
 * @ process  : pointer on process descriptor.
 * @ base     : region virtual base address.
 * @ size     : region size (bytes).
 * @ returns NULL if no conflict / return conflicting vseg pointer if conflict.
 ********************************************************************************************/
vseg_t * vmm_check_conflict( struct process_s * process,
                             vpn_t              base,
                             vpn_t              size );

/*********************************************************************************************
 * This function allocates memory for a vseg descriptor, initialises it, and register it
 * in the VMM of the process. It checks the collision with pre-existing vsegs in VMM.
 * For STACK and MMAP types vseg, it does not use the base argument, but uses the VMM STACK
 * and MMAP specific allocators to get a base address in virtual space.
 * To comply with the "on-demand" paging policy, this function does NOT modify the
 * page table, and does not allocate physical memory for vseg data.
 *********************************************************************************************
 * @ vmm   : pointer on process descriptor.
 * @ base      : vseg base address
 * @ size      : vseg size (bytes)
 * @ type      : vseg type
 * @ returns pointer on vseg if success / returns NULL if no memory or conflict.
 ********************************************************************************************/
vseg_t * vmm_create_vseg( struct process_s * process,
                          intptr_t           base,
	                      intptr_t           size,
	                      uint32_t           type );

/*********************************************************************************************
 * This function removes a vseg identified by it's pointer from the VMM of the calling process.
 * - If the vseg has not the STACK or MMAP type, it is removed from the vsegs list,
 *   and the physical memory allocated to vseg descriptor is released to KMEM.
 * - If the vseg has the STACK type, it is removed from the vsegs list, the physical memory
 *   allocated to vseg descriptor is released to KMEM, and the stack slot is returned to the
 *   VMM STACK allocator.
 * - If the vseg has the MMAP type, it is removed from the vsegs list and is registered
 *   in the zombi_list of the VMM MMAP allocator for future reuse. The physical memory
 *   allocated to vseg descriptor is NOT released to KMEM.
 *********************************************************************************************
 * @ vseg      : pointer on vseg to be removed.
 ********************************************************************************************/
void vmm_remove_vseg( vseg_t * vseg );

/*********************************************************************************************
 * This function allocates physical memory from the local cluster to map all PTEs
 * of a "kernel" vseg (type KCODE , KDATA, or KDEV) in the page table of process_zero.
 * WARNING : It should not be used for "user" vsegs, that must be mapped using the
 * "on-demand-paging" policy.
 *********************************************************************************************
 * @ vseg     : pointer on the vseg to be mapped.
 * @ attr     : GPT attributes to be set for all vseg pages.
 * @ returns 0 if success / returns ENOMEM if no memory
 ********************************************************************************************/
error_t vmm_map_kernel_vseg( vseg_t           * vseg,
                             uint32_t           attr );

/*********************************************************************************************
 * This function unmaps all PTEs of a given vseg, in the generic page table associated
 * to a given process descriptor, and releases the corresponding physical memory.
 * It can be used for any type of vseg.
 *********************************************************************************************
 * @ process  : pointer on process descriptor.
 * @ vseg     : pointer on the vseg to be unmapped.
 ********************************************************************************************/
void vmm_unmap_vseg( struct process_s * process,
                     vseg_t           * vseg );

/*********************************************************************************************
 * This function removes a given region (defined by a base address and a size) from
 * the VMM of a given process descriptor. This can modify several vsegs:
 * (a) if the region is not entirely mapped in an existing vseg, it's an error.
 * (b) if the region has same base and size as an existing vseg, the vseg is removed.
 * (c) if the removed region cut the vseg in two parts, it is removed and re-created.
 * (d) if the removed region cut the vseg in three parts, it is removed, and two are created.
 * TODO : cases (c) and (d) are not implemented [AG]
 *********************************************************************************************
 * @ process   : pointer on process descriptor
 * @ base      : vseg base address
 * @ size      : vseg size (bytes)
 ********************************************************************************************/
error_t vmm_resize_vseg( struct process_s * process,
                         intptr_t           base,
                         intptr_t           size );

/*********************************************************************************************
 * This function checks that a given virtual address is contained in a registered vseg.
 * - if the vseg is registered in local process VMM, it returns the local vseg pointer.
 * - if the vseg is missing in local VMM, it uses a RPC to get it from the reference cluster,
 *   register it in local VMM and returns the local vseg pointer, if success.
 * - if the vseg is missing in reference VMM, it returns an user error.
 * It creates a kernel panic if there is not enough memory to create a new vseg descriptor
 * in the cluster containing the calling thread.
 *********************************************************************************************
 * @ process   : [in] pointer on process descriptor
 * @ vaddr     : [in] virtual address
 * @ vseg      : [out] pointer on found vseg
 * @ returns 0 if success / returns -1 if user error.
 *********************************************************************************************/
error_t vmm_get_vseg( struct process_s  * process,
                      intptr_t            vaddr,
                      vseg_t           ** vseg );            

/*********************************************************************************************
 * This function is called by the generic exception handler when a page fault 
 * has been detected in a given cluster. 
 * If the local cluster is not the reference cluster, it send a RPC_VMM_GET_PTE
 * to the reference cluster to get the missing PTE attributes and PPN, and update
 * the local page table. If the local cluster is the reference, it call directly
 * the vmm_get_pte() function.
 *********************************************************************************************
 * @ process   : pointer on process descriptor.
 * @ vseg      : pointer on involved vseg.
 * @ vpn       : VPN of the missing PTE.
 * @ returns 0 if success / returns ENOMEM if no memory.
 ********************************************************************************************/
error_t vmm_handle_page_fault( struct process_s * process,
                               vseg_t           * vseg,
                               vpn_t              vpn );

/*********************************************************************************************
 * This function returns in the "attr" and "ppn" arguments the PTE associated to a given
 * VPN for a given process. This function must be called by a thread running in the 
 * reference cluster. To get the PTE from another cluster, use the RPC_VMM_GET_PTE.
 * The vseg containing the searched VPN should be registered in the reference VMM.
 * If the PTE in the reference page table is unmapped, this function allocates the missing
 * physical page from the target cluster defined by the vseg type, initialize it,
 * and update the reference page table. It calls the RPC_PMEM_GET_PAGES to get and 
 * initialize the missing physical page, if the target cluster is not the reference cluster.
 *********************************************************************************************
 * @ process   : [in] pointer on process descriptor.
 * @ vpn       : [in] VPN defining the missing PTE.
 * @ attr      : [out] PTE attributes.
 * @ ppn       : [out] PTE ppn.
 * @ returns 0 if success / returns ENOMEM if error.
 ********************************************************************************************/
error_t vmm_get_pte( struct process_s * process,
                     vpn_t              vpn,
                     uint32_t         * attr,
                     ppn_t            * ppn );

/*********************************************************************************************
 * This function is called by the vmm_get_pte() function.
 * Depending on the vseg type, defined by the <vseg> argument, it returns the PPN 
 * (Physical Page Number) associated to a missing page defined by the <vpn> argument.
 * - For the VSEG_TYPE_FILE, it returns the physical page from the file mapper.
 *   For all other types, it allocates a new physical page from the cluster defined
 *   by the <vseg->cxy> field, or by the <vpn> MSB bits for a distributed vseg.
 * - For the VSEG_TYPE_CODE and VSEG_TYPE_DATA types, the allocated page is initialized
 *   from the .elf file mapper. For others vseg types it is not initialised.
 *********************************************************************************************
 * @ vseg   : local pointer on vseg containing the mising page.
 * @ vpn    : Virtual Page Number identifying the missing page.
 * @ ppn    : [out] returned Physical Page Number.
 ********************************************************************************************/
error_t vmm_get_one_ppn( vseg_t * vseg,
                         vpn_t    vpn,
                         ppn_t  * ppn );

/*********************************************************************************************
 * This function makes the virtual to physical address translation, using the calling
 * process page table. It uses identity mapping if required by the <ident> argument.
 * This address translation is required to configure the peripherals having a DMA
 * capability, or to implement the software L2/L3 cache cohérence, using the MMC device 
 * synchronisation primitives.
 * WARNING : the <ident> value must be defined by the CONFIG_KERNEL_IDENTITY_MAP parameter.
 *********************************************************************************************
 * @ ident     : [in] uses identity mapping if true.
 * @ ptr       : [in] virtual address.
 * @ paddr     : [out] pointer on buffer for physical address.
 * @ returns 0 if success / returns ENOMEM if error.
 ********************************************************************************************/
error_t vmm_v2p_translate( bool_t    ident,
                           void    * ptr,
                           paddr_t * paddr );



#endif /* _VMM_H_ */
