/*
 * soclib_bdv.c - soclib simple block device driver implementation.
 *
 * Author     Alain Greiner (2016)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH.is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH.is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH.; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <device.h>
#include <dev_ioc.h>
#include <soclib_bdv.h>
#include <thread.h>
#include <spinlock.h>


/////////////////////////////////////
void soclib_bdv_init( xptr_t dev_xp )
{
    // get IOC device descriptor cluster and local pointer
    cxy_t      dev_cxy  = GET_CXY( dev_xp );
    device_t * dev_ptr  = (device_t *)GET_PTR( dev_xp );
  
    // get extended pointer on SOCLIB_BDV peripheral base address
	xptr_t  bdv_xp = hal_remote_lwd( XPTR( dev_cxy , &dev_ptr->base ) );

    // get hardware device cluster and local pointer
    cxy_t      bdv_cxy  = GET_CXY( bdv_xp );
    uint32_t * bdv_ptr  = (uint32_t *)GET_PTR( bdv_xp );

    // get block_size and block_count  
	uint32_t block_size = hal_remote_lw( XPTR( bdv_cxy , bdv_ptr + BDV_BLOCK_SIZE_REG ) );
	uint32_t block_count = hal_remote_lw( XPTR( bdv_cxy , bdv_ptr + BDV_SIZE_REG ) );

    // set IOC device descriptor extension
    hal_remote_sw( XPTR( dev_cxy , &dev_ptr->ext.ioc.size  ) , block_size );
    hal_remote_sw( XPTR( dev_cxy , &dev_ptr->ext.ioc.count ) , block_count );

} // end soclib_bdv_init()


//////////////////////////////////////////////////////////////////
void __attribute__ ((noinline)) soclib_bdv_command( xptr_t th_xp )
{
    uint32_t   to_mem;       // command argument
    uint32_t   lba;          // command argument
    uint32_t   count;        // command argument
    xptr_t     buf_xp;       // command argument
    xptr_t     dev_xp;       // command argument

    // get client thread cluster and local pointer
    cxy_t      th_cxy = GET_CXY( th_xp );
    thread_t * th_ptr = (thread_t *)GET_PTR( th_xp );

    // get command arguments and extended pointer on IOC device
    to_mem =         hal_remote_lw ( XPTR( th_cxy , &th_ptr->dev.ioc.to_mem ) );
    lba    =         hal_remote_lw ( XPTR( th_cxy , &th_ptr->dev.ioc.lba    ) );
    count  =         hal_remote_lw ( XPTR( th_cxy , &th_ptr->dev.ioc.count  ) );
    buf_xp = (xptr_t)hal_remote_lwd( XPTR( th_cxy , &th_ptr->dev.ioc.buf_xp ) );
    dev_xp = (xptr_t)hal_remote_lwd( XPTR( th_cxy , &th_ptr->dev.ioc.dev_xp ) );

    // get IOC device cluster and local pointer
    cxy_t      dev_cxy = GET_CXY( dev_xp );
    device_t * dev_ptr = (device_t *)GET_PTR( dev_xp );

    // get extended pointer on SOCLIB-BDV peripheral
    xptr_t     bdv_xp = hal_remote_lw( XPTR( dev_cxy , &dev_ptr->base ) );

    // get SOCLIB_BDV device cluster and local pointer
    cxy_t      bdv_cxy = GET_CXY( bdv_xp );
    uint32_t * bdv_ptr = (uint32_t *)GET_PTR( bdv_xp );

    // split buffer address in two 32 bits words
    uint32_t   buf_lsb = (uint32_t)(buf_xp);
    uint32_t   buf_msb = (uint32_t)(buf_xp>>32);

    // get access type
    uint32_t   type =  to_mem ? BDV_OP_READ : BDV_OP_WRITE; 

    // set SOCLIB_BDV registers to start one I/O operation
    hal_remote_sw( XPTR( bdv_cxy , bdv_ptr + BDV_IRQ_ENABLE_REG ) , 1       );
    hal_remote_sw( XPTR( bdv_cxy , bdv_ptr + BDV_BUFFER_REG     ) , buf_lsb ); 
    hal_remote_sw( XPTR( bdv_cxy , bdv_ptr + BDV_BUFFER_EXT_REG ) , buf_msb ); 
    hal_remote_sw( XPTR( bdv_cxy , bdv_ptr + BDV_LBA_REG        ) , lba     );
    hal_remote_sw( XPTR( bdv_cxy , bdv_ptr + BDV_COUNT_REG      ) , count   );
    hal_remote_sw( XPTR( bdv_cxy , bdv_ptr + BDV_OP_REG         ) , type    );

    // Block and deschedule server thread
    thread_block( CURRENT_THREAD , THREAD_BLOCKED_DEV_ISR );
    sched_yield();
    
} // end soclib_bdv_cmd()


////////////////////////////////////////////////////////////////
void __attribute__ ((noinline)) soclib_bdv_isr( device_t * dev )
{
    // get extended pointer on client thread
    xptr_t root = XPTR( local_cxy , &dev->wait_root );
    xptr_t client_xp = XLIST_FIRST_ELEMENT( root , thread_t , wait_list );

    // get extended pointer on server thread
    xptr_t server_xp = XPTR( local_cxy , &dev->server );

    // get client thread cluster and local pointer
    cxy_t      client_cxy = GET_CXY( client_xp );
    thread_t * client_ptr = (thread_t *)GET_PTR( client_xp );

    // get SOCLIB_BDV device cluster and local pointer
    cxy_t      bdv_cxy  = GET_CXY( dev->base );
    uint32_t * bdv_ptr  = (uint32_t *)GET_PTR( dev->base );

    // get BDV status register and acknowledge IRQ
	uint32_t status = hal_remote_lw( XPTR( bdv_cxy , bdv_ptr + BDV_STATUS_REG ) );   

    // set operation status in command
	if((status != BDV_READ_SUCCESS) && (status != BDV_WRITE_SUCCESS)) 
    {
        hal_remote_sw( XPTR( client_cxy , &client_ptr->dev.ioc.error ) , 1 );
    }
	else
    {
        hal_remote_sw( XPTR( client_cxy , &client_ptr->dev.ioc.error ) , 0 );
    }

    // unblock server thread 
    thread_unblock( server_xp , THREAD_BLOCKED_DEV_ISR );

    // unblock client thread
    thread_unblock( client_xp , THREAD_BLOCKED_IO );

} // end soclib_bdv_isr()



