/*
 * soclib_tty.c - soclib tty driver definition.
 *
 * Author  Alain Greiner (2016)
 *
 * Copyright (c)  UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-kernel; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <dev_txt.h>
#include <device.h>
#include <spinlock.h>


/****************************************************************************************
 * This driver supports the soclib_multi_tty component.
 * It implements the generic TXT device API:
 * - transfer one single character from TTY to command "buffer" if to_mem is non-zero.
 * - transfer "count" characters from command "buffer" to TTY if "to_mem is zero.
 ***************************************************************************************/

/****************************************************************************************
 *     SOCLIB_TTY registers offsets  
 ***************************************************************************************/

#define TTY_WRITE_REG          0
#define TTY_STATUS_REG         1
#define TTY_READ_REG           2
#define TTY_CONFIG_REG         3

#define TTY_SPAN               4       // number of registers per channel

/****************************************************************************************
 * masks for TTY_STATUS_REG 
 ***************************************************************************************/

#define TTY_STATUS_RX_FULL     1       // TTY_READ_REG full if 1
#define TTY_STATUS_TX_FULL     2       // TTY_WRITE_REG full if 1

/****************************************************************************************
 * masks for TTY_CONFIG_REG 
 ***************************************************************************************/

#define TTY_CONFIG_RX_ENABLE   1       // TTY_RX IRQ enabled if 1
#define TTY_CONFIG_TX_ENABLE   2       // TTY_TX IRQ enabled if 1

/****************************************************************************************
 * This function masks both the TTY_RX and TTY_TX IRQs from the SOCLIB_TTY device.
 * Thes IRQs are unmasked by the soclib_tty_cmd() function.
 ****************************************************************************************
 * @ dev     : extended pointer on the generic TXT device descriptor.
 ***************************************************************************************/
extern void soclib_tty_init( xptr_t dev_xp );

/****************************************************************************************
 * This function handles the command registered in the thread descriptor identified
 * by the <xp_thread> argument.
 * - For the TXT_READ and TXT_WRITE commands, it only unmask the proper TTY_RX / TTY_TX
 *   IRQ, and blocks the TXT device server thread on the THREAD_BLOCKED_DEV_ISR, as the
 *   data transfer is done by the ISR.
 * - For the TXT_SYNC_READ command, that should be only used by the kernel do display
 *   log or debug messages, it directly access the SOCLIB_TTY registers, using
 *   a busy waiting policy if required.
 ****************************************************************************************
 * @ xp_thread : extended pointer on client thread descriptor.
 ***************************************************************************************/
extern void soclib_tty_command( xptr_t thread_xp );

/****************************************************************************************
 * This ISR should be executed only for the TXT_READ and TXT_WRITE commands.
 * It get the command arguments from the first client thread in the TXT device queue:
 * - if TXT_READ, it transfer one byte from the TTY_READ_REG to the command buffer.
 *   It simply returns for retry if TTY_READ_REG is empty.
 * - if TXT_WRITE, it tries to transfer several bytes from the command buffer to the 
 *   TTY_WRITE_REG. If the TTY_WRITE_REG is full, it updates the "count" and "buffer" 
 *   command arguments and returns for retry.
 * When the I/O operation is completed, it set the status field in the command, unblock
 * the server thread, and unblock the client thread. 
 ****************************************************************************************
 * @ dev     : local pointer on generic TXT device descriptor.
 ***************************************************************************************/
extern void soclib_tty_isr( device_t * dev );

