/*
 * sys_stat.c - Return statistics on a file or directory.
 * 
 * Author    Alain Greiner  (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <hal_uspace.h>
#include <hal_special.h>
#include <errno.h>
#include <thread.h>
#include <printk.h>
#include <vfs.h>
#include <vmm.h>
#include <process.h>

//////////////////////////////////////////
int sys_stat( uint32_t            file_id,
              struct vfs_stat_s * stat )
{
    error_t           error;
    paddr_t           paddr;
    struct vfs_stat_s k_stat;
    xptr_t            file_xp;
	
	thread_t  * this    = CURRENT_THREAD;
	process_t * process = this->process;

    // check stat structure in user space
    error = vmm_v2p_translate( false , stat , &paddr );

	if( error )
	{
        printk("\n[ERROR] in %s : stat structure unmapped  for thread %x in process %x\n",
               __FUNCTION__ , this->trdid , process->pid );
		this->errno = EINVAL;
		return -1;
	}	

    // get extended pointer on remote file descriptor
    file_xp = process_fd_get_xptr( process , file_id );

    if( file_xp == XPTR_NULL )
    {
        printk("\n[ERROR] in %s : undefined file descriptor for thread %x in process %x\n",
               __FUNCTION__ , this->trdid , process->pid );
        this->errno = EBADFD;
        return -1;
    }

    // call the relevant VFS function
    error = vfs_stat( file_xp,
                      &k_stat );
    if( error )
	{
        printk("\n[ERROR] in %s : cannot access file %d for thread %x in process %x\n",
               __FUNCTION__ , file_id , this->trdid , process->pid );
		this->errno = error;
		return -1;
	}
   
    // copy stat to user space
    hal_copy_to_uspace( stat , &k_stat , sizeof(struct vfs_stat_s) );

    hal_fence();

	return 0;

}  // end sys_stat()

