/*
 * scheduler.c - Core scheduler implementation.
 * 
 * Author    Alain Greiner (2016)
 *
 * Copyright (c)  UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH. is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH. is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH.; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_types.h>
#include <hal_irqmask.h>
#include <hal_context.h>
#include <printk.h>
#include <list.h>
#include <core.h>
#include <thread.h>
#include <chdev.h>
#include <scheduler.h>

///////////////////////////////////////////////////////////////////////////////////////////
// Extern global variables
///////////////////////////////////////////////////////////////////////////////////////////

extern chdev_directory_t    chdev_dir;            // allocated in kernel_init.c file


////////////////////////////////
void sched_init( core_t * core )
{
    scheduler_t * sched = &core->scheduler;

    sched->u_threads_nr   = 0;
    sched->k_threads_nr   = 0;

    sched->current        = CURRENT_THREAD;
    sched->idle           = NULL;             // initialized in kernel_init()
    sched->u_last         = NULL;             // initialized in sched_register_thread()
    sched->k_last         = NULL;             // initialized in sched_register_thread()

    // initialise threads lists
    list_root_init( &sched->u_root );
    list_root_init( &sched->k_root );

}  // end sched_init()

////////////////////////////////////////////
void sched_register_thread( core_t   * core,
                            thread_t * thread )
{
    scheduler_t * sched = &core->scheduler;
    thread_type_t type  = thread->type;

    // take lock protecting sheduler lists
    spinlock_lock( &sched->lock );

    if( type == THREAD_USER )
    {
        // register thread in scheduler user list
        list_add_last( &sched->u_root , &thread->sched_list );
        sched->u_threads_nr++;

        // initialize u_last field if first user thread
        if( sched->u_last == NULL ) sched->u_last = &thread->sched_list;
    }
    else // kernel thread
    {
        // register thread in scheduler kernel list
        list_add_last( &sched->k_root , &thread->sched_list );
        sched->k_threads_nr++;

        // initialize k_last field if first kernel thread
        if( sched->k_last == NULL ) sched->k_last = &thread->sched_list; 
    }

    // release lock 
    spinlock_unlock( &sched->lock );

}  // end sched_register()

/////////////////////////////////////////////
void sched_remove_thread( thread_t * thread )
{
    core_t       * core  = thread->core;
    scheduler_t  * sched = &core->scheduler;
    thread_type_t  type  = thread->type;

    // take lock protecting sheduler lists
    spinlock_lock( &sched->lock );

    if( type == THREAD_USER )
    {
        // remove thread from user list
        list_unlink( &thread->sched_list );
        sched->u_threads_nr--;

        // reset the u_last field if list empty
        if( sched->u_threads_nr == 0 ) sched->u_last = NULL;
    }
    else // kernel thread 
    {
        // remove thread from kernel list
        list_unlink( &thread->sched_list );
        sched->k_threads_nr--;

        // reset the k_last field if list empty
        if( sched->k_threads_nr == 0 ) sched->k_last = NULL;
    }

    // release lock
    spinlock_unlock( &sched->lock );

}  // end sched_remove()

///////////////////////////////////////////
void sched_kill_thread( thread_t * thread )
{
    // check thread locks
    if( thread_can_yield() == false )
    {
        panic("thread %x in process %x on core[%x][%d]"
              " did not released all locks",
              thread->trdid , thread->process->pid, 
              local_cxy , thread->core->lid ); 
    }

    // remove thread from scheduler
    sched_remove_thread( thread );

    // reset the THREAD_SIG_KILL signal
    thread_reset_signal( thread , THREAD_SIG_KILL );

}  // end sched_kill_thread()

////////////////////////////////////////
thread_t * sched_select( core_t * core )
{
    thread_t    * thread;

    scheduler_t * sched = &core->scheduler;

    sched_dmsg("\n[DMSG] %s : enter core[%x,%d] / cycle %d\n",
    __FUNCTION__ , local_cxy , core->lid , hal_time_stamp() );

    // take lock protecting sheduler lists
    spinlock_lock( &sched->lock );

    list_entry_t * current;
    list_entry_t * last;

    // first : scan the kernel threads list if not empty 
    if( list_is_empty( &sched->k_root ) == false )
    {
        last    = sched->k_last;
        current = sched->k_last;
        do
        {
            // get next entry in kernel list
            current = list_next( &sched->k_root , current );

            // skip the root that does not contain a thread
            if( current == NULL ) current = sched->k_root.next;

            // get thread pointer for this entry
            thread = LIST_ELEMENT( current , thread_t , sched_list );

            // return thread if not idle_thread and runnable 
            if( (thread->type != THREAD_IDLE) && (thread->blocked == 0) ) 
            {
                // release lock 
                spinlock_unlock( &sched->lock );

                sched_dmsg("\n[DMSG] %s : exit core[%x,%d] / k_thread = %x / cycle %d\n",
                __FUNCTION__ , local_cxy , core->lid , thread->trdid , hal_time_stamp() );

                return thread;
            }
        }
        while( current != last );
    }

    // second : scan the user threads list if not empty 
    if( list_is_empty( &sched->u_root ) == false )
    {
        last    = sched->u_last;
        current = sched->u_last;
        do
        {
            // get next entry in user list
            current = list_next( &sched->u_root , current );

            // skip the root that does not contain a thread
            if( current == NULL ) current = sched->u_root.next;

            // get thread pointer for this entry
            thread = LIST_ELEMENT( current , thread_t , sched_list );

            // return thread if runnable
            if( thread->blocked == 0 )
            {
                // release lock 
                spinlock_unlock( &sched->lock );

                sched_dmsg("\n[DMSG] %s : exit core[%x,%d] / u_thread = %x / cycle %d\n",
                __FUNCTION__ , local_cxy , core->lid , thread->trdid , hal_time_stamp() );
                return thread;
            }
        }
        while( current != last );
    }

    // release lock 
    spinlock_unlock( &sched->lock );

    sched_dmsg("\n[DMSG] %s : exit core[%x,%d] / idle = %x / cycle %d\n",
    __FUNCTION__ , local_cxy , core->lid , sched->idle->trdid , hal_time_stamp() );

    // third : return idle thread if no runnable thread
    return sched->idle;

}  // end sched_select()

//////////////////////////////////////////
void sched_handle_signals( core_t * core )
{
    list_entry_t * iter;
    thread_t     * thread;
    scheduler_t  * sched = &core->scheduler;

    sched_dmsg("\n[DMSG] %s : enter / thread %x on core[%x,%d]\n",
    __FUNCTION__, CURRENT_THREAD->trdid , local_cxy , core->lid );

    // take lock protecting threads lists
    spinlock_lock( &sched->lock );

    // handle user threads
    LIST_FOREACH( &sched->u_root , iter )
    {
        thread = LIST_ELEMENT( iter , thread_t , sched_list );
        if( thread->signals & THREAD_SIG_KILL )  sched_kill_thread( thread );
    }

    // handle kernel threads
    LIST_FOREACH( &sched->k_root , iter )
    {
        thread = LIST_ELEMENT( iter , thread_t , sched_list );
        if( thread->signals & THREAD_SIG_KILL )  sched_kill_thread( thread );
    }

    // release lock 
    spinlock_unlock( &sched->lock );

    sched_dmsg("\n[DMSG] %s : exit / thread %x on core[%x,%d]\n",
    __FUNCTION__, CURRENT_THREAD->trdid , local_cxy , core->lid );

} // end sched_handle_signals()

///////////////////////////////////
void sched_yield( thread_t * next )
{
    reg_t         sr_save;

    thread_t    * current = CURRENT_THREAD;
    core_t      * core    = current->core;
    scheduler_t * sched   = &core->scheduler;

    sched_dmsg("\n[DMSG] %s : thread %x on core[%x,%d] enter / cycle %d\n",
    __FUNCTION__, current->trdid, local_cxy, core->lid, hal_time_stamp() );

    // delay the yield if current thread has locks
    if( thread_can_yield() == false )
    {
        current->flags |= THREAD_FLAG_SCHED;
        return;
    }

    // first loop on all threads to handle pending signals
    sched_handle_signals( core );

    // second loop on threads to select next thread if required
    if( next == NULL ) next = sched_select( core );

    // check next thread attached to same core as the calling thread
    assert( (next->core == current->core), __FUNCTION__ , "next core != current core\n");

    // check next thread not blocked
    assert( (next->blocked == 0), __FUNCTION__ , "next thread is blocked\n");

    // switch contexts and update scheduler state if next != current
	if( next != current )
    {
        sched_dmsg("\n[DMSG] %s : trd %x (%s) on core[%x,%d] => trd %x (%s) / cycle %d\n",
        __FUNCTION__, current->trdid, thread_type_str(current->type), local_cxy, core->lid, 
        next->trdid, thread_type_str(next->type), hal_time_stamp() );

        // calling thread desactivate IRQs
        hal_disable_irq( &sr_save );

        // update scheduler 
        if( current->type == THREAD_USER ) sched->u_last = &current->sched_list;
        else                               sched->k_last = &current->sched_list;
        sched->current = next;

        // handle FPU
	    if( next->type == THREAD_USER )
        {
	        if( next == core->fpu_owner )  hal_fpu_enable();
	        else                           hal_fpu_disable();
        }

        // switch contexts 
        hal_cpu_context_switch( current , next );

        // restore IRQs when calling thread resume
        hal_restore_irq( sr_save );
    }
    else
    {
        sched_dmsg("\n[DMSG] %s : thread %x on core[%x,%d] continue / cycle %d\n",
        __FUNCTION__, current->trdid, local_cxy, core->lid, hal_time_stamp() );
    }
}  // end sched_yield()

////////////////////
void sched_display()
{
    list_entry_t * iter;
    thread_t     * thread;
    uint32_t       save_sr;

    thread_t     * current = CURRENT_THREAD;
    core_t       * core    = current->core;
    scheduler_t  * sched   = &core->scheduler;
    
    // get pointers on TXT0 chdev
    xptr_t    txt0_xp  = chdev_dir.txt[0];
    cxy_t     txt0_cxy = GET_CXY( txt0_xp );
    chdev_t * txt0_ptr = GET_PTR( txt0_xp );

    // get extended pointer on remote TXT0 chdev lock
    xptr_t  lock_xp = XPTR( txt0_cxy , &txt0_ptr->wait_lock );

    // get TXT0 lock in busy waiting mode
    remote_spinlock_lock_busy( lock_xp , &save_sr );

    nolock_printk("\n***** scheduler state for core[%x,%d]\n"
           "kernel_threads = %d / user_threads = %d / current = %x\n",
            local_cxy , core->lid, 
            sched->k_threads_nr, sched->u_threads_nr, sched->current->trdid );

    // display kernel threads
    LIST_FOREACH( &sched->k_root , iter )
    {
        thread = LIST_ELEMENT( iter , thread_t , sched_list );
        nolock_printk(" - type = %s / trdid = %x / pid = %x / func = %x / blocked_vect = %x\n",
        thread_type_str( thread->type ), thread->trdid, thread->process->pid,
        thread->entry_func, thread->blocked );
    }

    // display user threads
    LIST_FOREACH( &sched->u_root , iter )
    {
        thread = LIST_ELEMENT( iter , thread_t , sched_list );
        nolock_printk(" - type = %s / trdid = %x / pid = %x / func = %x / blocked_vect = %x\n",
        thread_type_str( thread->type ), thread->trdid, thread->process->pid,
        thread->entry_func, thread->blocked );
    }

    // release TXT0 lock
    remote_spinlock_unlock_busy( lock_xp , save_sr );

}  // end sched_display()

