/*
 * sys_kill.c - Send a signal to a given process.
 * 
 * Author    Alain Greiner (2016,2017)
 *
 * Copyright (c)  UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <kernel_config.h>
#include <hal_types.h>
#include <hal_irqmask.h>
#include <errno.h>
#include <thread.h>
#include <printk.h>
#include <process.h>
#include <signal.h>
#include <cluster.h>
#include <rpc.h>

///////////////////////////
int sys_kill( pid_t    pid,
              uint32_t sig_id )
{
    uint32_t    save_sr;       // required to enable IRQs

#if CONFIG_SYSCALL_DEBUG
uint64_t    tm_start;
uint64_t    tm_end;
tm_start = hal_get_cycles();
#endif

    thread_t  * this    = CURRENT_THREAD;
    process_t * process = this->process;

    // get owner process cluster and lpid
    cxy_t   owner_cxy  = CXY_FROM_PID( pid );
    lpid_t  lpid       = LPID_FROM_PID( pid );

    // check pid
    if( (lpid >= CONFIG_MAX_PROCESS_PER_CLUSTER) || cluster_is_undefined( owner_cxy ) )
    {
        printk("\n[ERROR] in %s : illegal target PID = %d for thread %x in process %x\n",
        __FUNCTION__ , pid , this->trdid , pid );
		this->errno = EINVAL;
        return -1;
    }

    // check sig_id
    if( (sig_id != SIGSTOP) && (sig_id != SIGCONT) && (sig_id != SIGKILL) ) 
    {
        printk("\n[ERROR] in %s : illegal signal type for thread %x in process %x\n",
        __FUNCTION__ , sig_id , this->trdid , pid );
		this->errno = EINVAL;
        return -1;
    }

    // enable IRQs
    hal_enable_irq( &save_sr );

    // execute process_make_kill() function in owner cluster
    if( local_cxy == owner_cxy )                                // owner is local 
    {
        process_make_kill( process , sig_id );
    }
    else                                                        // owner is remote
    {
        rpc_process_make_kill_client( owner_cxy , process , sig_id );
    }

    // restore IRQs
    hal_restore_irq( save_sr );

    hal_fence();

#if CONFIG_SYSCALL_DEBUG 
tm_end = hal_get_cycles();
syscall_dmsg("\n[DBG] %s exit : core[%x,%d] / thread %x in process %x / cycle %d\n"
"process %x killed / cost = %d\n",
__FUNCTION__ , local_cxy , this->core->lid , this->trdid , this->process->pid ,
tm_start , pid , (uint32_t)(tm_end - tm_start) );
#endif
 
	return 0;

}  // end sys_kill()

