/*
 * sys_thread_exit.c - terminates the execution of calling thread
 * 
 * Authors   Alain Greiner (2016,2017)
 *
 * Copyright (c) UPMC Sorbonne Universites
 *
 * This file is part of ALMOS-MKH.
 *
 * ALMOS-MKH is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by
 * the Free Software Foundation; version 2.0 of the License.
 *
 * ALMOS-MKH is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ALMOS-MKH; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <hal_types.h>
#include <thread.h>
#include <core.h>
#include <vmm.h>
#include <scheduler.h>
#include <printk.h>

////////////////////////////////////////
int sys_thread_exit( void * exit_value )
{
    paddr_t      paddr;
    error_t	     error;

#if CONFIG_SYSCALL_DEBUG
uint32_t     tm_start;
uint32_t     tm_end;
tm_start = hal_get_cycles();
#endif

	thread_t  * this    = CURRENT_THREAD;
    process_t * process = this->process;

    // check all locks released
	if( !thread_can_yield() )
	{
        printk("\n[ERROR] in %s : locks not released / thread %x in process %x\n",
        __FUNCTION__, this->trdid, process->pid );
        this->errno = EINVAL;
        return -1;
    }

    // register the exit_value pointer in this thread descriptor
    this->join_value = exit_value;

    if( (this->flags & THREAD_FLAG_DETACHED) == 0 )    // this thread is joinable
    {
        // check exit_value in user space
        error = vmm_v2p_translate( false , exit_value , &paddr );
	    if( error )
        {
            printk("\n[ERROR] in %s : illegal pointer = %x / thread %x in process %x\n",
            __FUNCTION__ , (intptr_t)exit_value, this->trdid , process->pid );
            this->errno = EINVAL;
            return -1;
        }

        // take the lock protecting the join 
        remote_spinlock_lock( XPTR( local_cxy, &this->join_lock ) );

        if( this->flags & THREAD_FLAG_JOIN_DONE )       // parent thread arrived first
        {
            // unblock the parent thread
            thread_unblock( this->join_xp , THREAD_BLOCKED_EXIT );

            // reset the JOIN_DONE flag in this thread
            this->flags &= ~THREAD_FLAG_JOIN_DONE;

            // release the lock protecting the flags
	        remote_spinlock_unlock( XPTR( local_cxy, &this->join_lock ) );
        }
        else                                           // this thread arrived first
        {
            // block this thread
            thread_block( this , THREAD_BLOCKED_JOIN );

            // release the lock protecting the flags
	        remote_spinlock_unlock( XPTR( local_cxy, &this->join_lock ) );

            // deschedule
            sched_yield( "WAITING JOIN" );
        }     
    }

#if CONFIG_SYSCALL_DEBUG
tm_end = hal_get_cycles();
syscall_dmsg("\n[DBG] %s : core[%x,%d] / thread %x in process %x / cycle %d\n"
"thread %x killed / cost = %d\n",
__FUNCTION__ , local_cxy , this->core->lid , this->trdid , this->process->pid , tm_start ,
this->trdid , (uint32_t)(tm_end - tm_start) );
#endif

    // suicide using a rpc because  a thread cannot kill itself
    rpc_thread_kill_client( local_cxy , this );

    return 0;   // never executed but required by compiler

}  // end sys_thread exit
